﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
/// Notice.
/// PUX拠点から、VisualStudio のテストエクスプローラ経由テストの場合、*.runsettings ファイルの TestRunParameters に以下パラメタを宣言してください.
/// NCL/CI 及び、NN_TEST_SCRIPT_PROXY_CONFIGURATION 環境変数が設定されたコマンドライン実施の場合は宣言は不要です.
/// <Parameter name = "EnableRunTestOnIDE_FromPUX" value="true" />

using CsTestAssistants;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;


namespace testFatal_RebootWithQuest
{
    using System.Linq;
    using NintendoServices = CsTestAssistants.NintendoServices;
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;
    using System.Runtime.Serialization;

    /// <summary>
    /// テストクラス
    /// </summary>
    [TestClass]
    public class UnitMain : TestClassBase
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[FatalTest]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ターゲットログのマッチング
        /// </summary>
        /// <param name="executor"></param>
        /// <param name="expected"></param>
        /// <returns></returns>
        //!----------------------------------------------------------------------------
        private bool MatchTargetLog(SigloHelper.CommodityExecutor.Context executor, string expected)
        {
            // 直前の実行結果出力から接頭辞 [target] を削除
            var output = Regex.Replace(
                executor.OutputStream.Standard.ToString(),
                @"^\[target\] ", "", RegexOptions.Multiline);

            // 期待されるパターンとのマッチング
            var match = Regex.Match(output, expected);
            Log.WriteLine("target log expected pattern : {0}", expected);
            Log.WriteLine("target log matching result : {0}", match.Success);
            return match.Success;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// quest フラグが立っている状態で Fatal した時に自動再起動するテスト
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-74183")]
        public void TestForRebootWithQuest()
        {
            var RebootInterval = new System.TimeSpan(0, 0, 20); //再起動後の待ち時間
            var LaunchInterval = 25; //アプリ起動時の待ち時間

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // make contents
                //  Application : 0x01001a500005e0b0, version 0
                var owner = new ID64(0x01001a500005e0b0);
                var @params = new List<TestApplication.GenerateParameter<int>>();
                {
                    // app
                    var param = new TestApplication.ApplicationParameter(owner, 0, 1 * 1024 * 1024);
                    param.UseSmallCode = true;
                    @params.Add(param);
                }
                var contents = TestApplication.MakeContents(intermediate, @params);
                var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];

                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");

                // finally
                scope.AddDisposer(() =>
                {
                    // questフラグを false に戻す
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "debug disable-rid-mode"
                        ));

                    // 再起動待ち時間を デフォルト値(300秒)に戻す
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "debug set-integer-fwdbg --name fatal --key quest_reboot_interval_second 300"
                        ));

                    // ゲームカード内容の削除
                    if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall --all"
                        ));
                });

                // ゲームカード内容の削除
                if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all"));

                // アプリをインストール
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {app.NspPath}"));

                // 1. 再起動待ち時間を N 秒に変更（デフォルトは300秒）
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-integer-fwdbg --name fatal --key quest_reboot_interval_second 5"
                    ));

                // 2. quest フラグを true 状態にする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug enable-rid-mode"
                    ));

                // 3. 再起動
                ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                // 4. Fatal 画面に遷移
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug throw-fatal 0"
                    ));

                // 再起動を待つ
                System.Threading.Thread.Sleep(RebootInterval);

                // 5.アプリを起動する。(リセットしていればアプリが起動できる。Fatalのままの場合アプリが起動できない)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommand($"application launch 0x{app.Identifier} AND util sleep {LaunchInterval}"));

                // 6.ログをチェックしてアプリ起動確認
                var expected =
                    $"\\s+ContentType\\s+:\\s+Application" +
                    $"\\s+ApplicationID\\s+:\\s+0x{app.Identifier}";
                ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
            }
        }
    }
}
