﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init/init_Malloc.h>
#include <nn/fatalsrv/fatalsrv_Server.h>

#include <nn/fatal/fatal_ApiPrivate.h>

#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_Screen.h"
#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_Environment.h"
#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_Settings.h"
#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_Font.h"
#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_MessageManager.h"

namespace {
    class FatalDrawScreen : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nn::fatalsrv::InitializeEnvironmentInfo();
            nn::fatalsrv::InitializeSharedFont();

            nn::settings::GetLanguageCode(&s_OriginalLanguageCode);
        }

        static void TearDownTestCase()
        {
            nn::settings::system::SetLanguageCode(s_OriginalLanguageCode);
        }


    private:
        static nn::settings::LanguageCode s_OriginalLanguageCode;
    };
    nn::settings::LanguageCode FatalDrawScreen::s_OriginalLanguageCode = {};

    nn::fatalsrv::CpuContext g_CpuContext = {};

    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_DrawThreadStack[20 * 1024];

    void TestFatalScreen(const nn::settings::LanguageCode& languageCode)
    {
        NN_LOG("Language: %s\n", languageCode.string);
        nn::settings::system::SetLanguageCode(languageCode);
        nn::fatalsrv::UpdateLanguageCode();

        // 描画関数は実際の環境と同じスタックサイズのスレッドで動かす
        nn::os::ThreadType drawTaskThread;
        nn::os::CreateThread(&drawTaskThread, [](void*) {
            nn::fatalsrv::FatalContext fatalContext = {};
            NNT_ASSERT_RESULT_SUCCESS(nn::fatalsrv::ShowFatalScreen(fatalContext, g_CpuContext, 0));
        }, nullptr, g_DrawThreadStack, sizeof(g_DrawThreadStack), nn::os::DefaultThreadPriority);

        nn::os::StartThread(&drawTaskThread);
        nn::os::WaitThread(&drawTaskThread);
        nn::os::DestroyThread(&drawTaskThread);
    }

}

namespace
{
    // INFO: fatal_Main.cpp と同じサイズのヒープにする
    const int HeapSize = 2 * 1024 * 1024 + 64 * 1024; // 2MB + 64KB
    NN_ALIGNAS(4096) nn::Bit8 g_Heap[HeapSize];
}

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_Heap, sizeof(g_Heap));
}

TEST_F(FatalDrawScreen, Initialize)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::fatalsrv::InitializeTextData());

    auto argc = nnt::GetHostArgc();
    int languageIndex = 0;
    bool notStop = argc <= 1 ; // 引数なし実行の場合はプログラムを終了しない。（ローカルで画面を表示し続けて確認するためのもの。）

    if (argc > 1)
    {
        languageIndex = strtol(nnt::GetHostArgv()[1], nullptr, 10);
    }

    auto languageCount = nn::settings::GetAvailableLanguageCodeCount();
    std::unique_ptr<nn::settings::LanguageCode[]> languageCodes(new nn::settings::LanguageCode[languageCount]);
    nn::settings::GetAvailableLanguageCodes(languageCodes.get(), languageCount);

    TestFatalScreen(languageCodes.get()[languageIndex]);

    while (notStop)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    }
}
