﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *  Copyright 2010 iGware, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of iGware, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of iGware, Inc.
 */


/*
 * Test the ES Concurrency model.
 *
 * Each individual ES Op may have global data that is used during the duration
 * of that call only, so low level ES functions must not be overlapped.
 *
 * The other restriction is that only one Import or Export operation can be
 * active at a time (end-to-end).
 *
 * So a two level mutex structure is required:
 *
 *   A lower level mutex that protects individual ES calls.
 *   A higher level mutex that protects Import and Export operations.
 *
 * The mutexes will nest, since Import/Export call lower level functions.
 * The higher level mutex will always be acquired first, so there should
 * be no deadlock issues.  Readonly operations will only need to acquire
 * the lower level mutex.
 */
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>

#include <nn/escore/escore.h>
#include <nn/iosc/iosc.h>
#include <nn/ioscrypto/iosctypes.h>
#include <nnt/escoreUtil/testEscore_util_istorage.h>

USING_ESCORE_UTIL_NAMESPACE
USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE

#include "../Common/testEs_Est_utils.h"

/*
 * Objects for test title to import
 */
#include "../Common/testEs_Data3_enc.cpp"
#include "../Common/testEs_Data3_dec.cpp"
#include "../Common/testEs_Data5_enc.cpp"
#include "../Common/testEs_Data5_dec.cpp"

#if defined( MIXED_OPS_TEST )
const u64 ES_TEST_TITLE_ID = 0x5000007700000099ULL;
const u32 ES_TEST_CONTENT_ID = 0xbabecafe;
const u32 ES_TEST_CONTENT_ID_2 = 0xbeefcafe;
#endif


/*
 * New CA00000004 CTR Cert Chain (SHA256 signatures)
 */
#include "../Common/testEs_Ca_cert_new.cpp"
#include "../Common/testEs_Cp_cert_new.cpp"
#include "../Common/testEs_Xs_cert_new.cpp"
#include "../Common/testEs_Sp_cert_new.cpp"

/* TMD and ticket signed by new SHA256 certs */
#include "../Common/testEs_Tkt_h256.cpp"
#include "../Common/testEs_Tkt1_h256.cpp"


#if defined( BASIC_OPS_TEST ) || defined( MIXED_OPS_TEST )
/* Certificates */
const int ES_TEST_CERT_CHAIN_LEN = 4;  // CA, XS, CP, MS

static const void *__certs[ES_TEST_CERT_CHAIN_LEN + 2] = {caNewCert, xsNewCert, cpNewCert, msCert, NULL, NULL};
static const u32 __nCerts = ES_TEST_CERT_CHAIN_LEN;
#endif

#if defined( MIXED_OPS_TEST )
static const void *__oldCerts[ES_TEST_CERT_CHAIN_LEN + 2] = {caCert, xsCert, cpCert, msCert, NULL, NULL};
static const u32 __nOldCerts = ES_TEST_CERT_CHAIN_LEN;
#endif

/*
 * Local buffers
 */
#if defined( MIXED_OPS_TEST )
static u8 __outDataBuf[1024] ATTR_SHA256_ALIGN;
#endif

/*
 * Useful macros
 */
#define CHECK_GOOD( result )                                           \
    do                                                                 \
    {                                                                  \
        if( !result.IsSuccess() )                                      \
        {                                                              \
            ES_TEST_LOG( "FAIL:  Bad result at line %d\n", __LINE__ ); \
            if( rv == ES_ERR_OK )                                      \
            {                                                          \
                rv = ES_ERR_INVALID;                                   \
            }                                                          \
            goto end;                                                  \
        }                                                              \
    } while( 0 )

#define CHECK_BAD( result )                                                              \
    do                                                                                   \
    {                                                                                    \
        if( result.IsSuccess() )                                                         \
        {                                                                                \
            ES_TEST_LOG( "FAIL:  Operation should have failed at line %d\n", __LINE__ ); \
            if( rv == ES_ERR_OK )                                                        \
            {                                                                            \
                rv = ES_ERR_INVALID;                                                     \
            }                                                                            \
            goto end;                                                                    \
        }                                                                                \
    } while( 0 )

#define MIN( a, b ) ( ( ( a ) < ( b ) ) ? ( a ) : ( b ) )


/*
 * Flags to determine which tests to run
 */
//#define BASIC_OPS_TEST
//#define MIXED_OPS_TEST


/*
 * Mutexes
 */
static nn::os::Mutex esImportOpMutex( true );
static nn::os::Mutex esBasicOpMutex( true );

#define ES_LOCKING

#if defined( ES_LOCKING )
/*
 * See if locking is working
 */
#if defined( MIXED_OPS_TEST )
static int __basicLockFail = 0;
static int __importLockFail = 0;
#endif

#if defined( BASIC_OPS_TEST ) || defined( MIXED_OPS_TEST )
static void __esImportOpLock()
{
    if( !esImportOpMutex.TryLock() )
    {
        /*
         * Only one thread competes for this lock, so this
         * should not happen
         */
        __importLockFail++;
        esImportOpMutex.Lock();
    }
}

static void __esImportOpUnlock()
{
    esImportOpMutex.Unlock();
}

static void __esBasicOpLock()
{
    if( !esBasicOpMutex.TryLock() )
    {
        /*
         * This count should be non-zero eventually
         */
        __basicLockFail++;
        esBasicOpMutex.Lock();
    }
}

static void __esBasicOpUnlock()
{
    esBasicOpMutex.Unlock();
}
#endif

/*
 * Macros for lock operations
 */
#define ES_LOCK_IMPORT_OPS() __esImportOpLock()
#define ES_UNLOCK_IMPORT_OPS() __esImportOpUnlock()
#define ES_LOCK_BASIC_OPS() __esBasicOpLock()
#define ES_UNLOCK_BASIC_OPS() __esBasicOpUnlock()

#else /* ES_LOCKING */

#define ES_LOCK_IMPORT_OPS()
#define ES_UNLOCK_IMPORT_OPS()
#define ES_LOCK_BASIC_OPS()
#define ES_UNLOCK_BASIC_OPS()

#endif /* ES_LOCKING */


/*
 * Simple parameter block passed to each thread
 */
struct ThreadParam
{
    int threadNum;
    nn::os::Event *pWaitEvent;
};


static ESError __testSetup()
{
    nn::Result result;
    ESError rv = ES_ERR_OK;

/*
 * Initialize the mutexes in unlocked state
 */
#ifdef CTR_TEST
    result = esImportOpMutex.TryInitialize( false );
    CHECK_GOOD( result );
    result = esBasicOpMutex.TryInitialize( false );
    CHECK_GOOD( result );
#endif

    return rv;
}




#if defined( BASIC_OPS_TEST ) || defined( MIXED_OPS_TEST )
static ESError __testReadTicket( MemoryInputStream &ticket )
{
    ESError rv = ES_ERR_OK;
    ETicket et;
    u32 tickSize;
    ESTicketId ticketId;
    ESTitleId titleId;

    ES_LOCK_BASIC_OPS();
    rv = et.Set( ticket, __certs, __nCerts, true );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set and verify ticket, rv=%d\n", rv );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = et.GetTitleId( &titleId );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get TitleId, rv=%d\n", rv );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = et.GetTicketId( &ticketId );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get TicketId, rv=%d\n", rv );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = et.GetSize( &tickSize );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get ticket size, rv=%d\n", rv );
        goto end;
    }

    // ES_TEST_LOG("Passed import ticket in place test\n");
    ES_TEST_LOG( "Read ticket: tickId %016llx, titleId %016llx, tickSize %ld\n", ticketId, titleId, tickSize );

end:
    return rv;
}
#endif


#if defined( BASIC_OPS_TEST )

const int RANDOM_DELAY_THRESHOLD = 5000; /* milliseconds */

static void __basicOpWorkerThread( ThreadParam *param )
{
    ESError rv = ES_ERR_OK;
    s32 loopCount = 0;
    s32 delay = 100; /* milliseconds */
    nn::os::TimerEvent timer( nn::os::EventClearMode_ManualClear );

    ES_TEST_LOG( "-> Thread %d waiting for start event\n", param->threadNum );
    param->pWaitEvent->Wait();
    ES_TEST_LOG( "-> Thread %d starting ...\n", param->threadNum );

    /*
     * Main loop to drive the test in each thread
     */
    do
    {
        if( param->threadNum == 1 )
        {
#ifdef RANDOM_DELAY_THRESHOLD
            rv = IOSC_GenerateRand( (u8 *)&delay, sizeof( delay ) );
            delay %= RANDOM_DELAY_THRESHOLD;
            if( delay < 0 )
            {
                delay = -delay;
            }
            delay += 100;
// ES_TEST_LOG("-> Thread %d delay %d ms\n", param->threadNum, delay);
#endif
            timer.StartOneShot( nn::fnd::TimeSpan::FromMilliSeconds( delay ) );
            timer.Wait();
        }
        if( ( ++loopCount & 0x1f ) == 0 )
        {
            ES_TEST_LOG( "-> Thread %d loop %d (ILF %d, BLF %d)\n", param->threadNum, loopCount, __importLockFail, __basicLockFail );
        }
        if( param->threadNum == 1 )
        {
            MemoryInputStream ticket( tkt_h256, sizeof( tkt_h256 ) );
            rv = __testReadTicket( ticket );
        }
        else
        {
            MemoryInputStream ticket( tkt1_h256, sizeof( tkt1_h256 ) );
            rv = __testReadTicket( ticket );
        }
    } while( rv == ES_ERR_OK );

    ES_TEST_LOG( "=> FAIL: Thread %d loop %d exiting ...\n", param->threadNum, loopCount );
}


static ESError __testBasicOps()
{
    ESError rv = ES_ERR_OK;
    RESULT_NS( Result ) result;
    nn::os::Thread thread1;
    ThreadParam thread1Param;
    nn::os::Event thread1Wait( false );
    nn::os::StackBuffer<4096> thread1Stack;
    nn::os::Thread thread2;
    ThreadParam thread2Param;
    nn::os::Event thread2Wait( false );
    nn::os::StackBuffer<4096> thread2Stack;

    ES_TEST_LOG( "Start Basic ES Operations threaded test\n" );

    /*
     * This procedure runs in the main thread and creates the
     * worker threads to run the tests
     */
    thread1Param.threadNum = 1;
    thread1Param.pWaitEvent = &thread1Wait;
    thread1.Start( __basicOpWorkerThread, &thread1Param, thread1Stack );
    thread1.ChangePriority( nn::os::DEFAULT_THREAD_PRIORITY + 1 );

    thread2Param.threadNum = 2;
    thread2Param.pWaitEvent = &thread2Wait;
    thread2.Start( __basicOpWorkerThread, &thread2Param, thread2Stack );
    thread2.ChangePriority( nn::os::DEFAULT_THREAD_PRIORITY + 2 );

    /*
     * Release both threads, but thread1 is higher priority.  It will
     * sleep on a random timer and preempt thread2 to test locking.
     */
    thread2Wait.Signal();
    /* Give thread2 a head start ... */
    nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds( 1000 ) );
    thread1Wait.Signal();

    /*
     * Wait for the workers to finish
     */
    ES_TEST_LOG( "Main thread: wait for worker threads to exit\n" );

    /*
     * Thread 1 is higher priority, so it will step on thread 2.  If
     * thread2 exits, that means a failure has occurred.
     */
    thread2.Join();
    ES_TEST_LOG( "Main thread: thread2 exited\n" );
#ifdef USE_OBSOLETE_CODE
    /* XXX this doesn't work:  need to kill thread1 */
    thread1.Join();
    ES_TEST_LOG( "Main thread: thread1 exited\n" );
#endif

    thread2.Finalize();
    thread1.Finalize();

    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "FAIL: Basic ES Operations threaded test\n" );
        goto end;
    }

    ES_TEST_LOG( "End of Basic ES Operations threaded test\n" );

end:
    return rv;
}
#endif  // BASIC_OPS_TEST


#if defined( MIXED_OPS_TEST )

static ESError __testImportTitle()
{
    ESError rv = ES_ERR_OK;
    MemoryInputStream ticket( tkt3_dp, sizeof( tkt3_dp ) );
    MemoryInputStream tmd( tmd1, sizeof( tmd1 ) );
    ETicketService es;
    u32 i, importedBytes, bytes;

    ES_LOCK_IMPORT_OPS();

    ES_LOCK_BASIC_OPS();
    rv = es.ImportTitleInit( ES_TEST_TITLE_ID, ticket, tmd, __oldCerts, __nOldCerts );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to initialize title import, rv=%d\n", rv );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportContentBegin( ES_TEST_CONTENT_ID, tmd, __oldCerts, __nOldCerts );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to begin content import, rv=%d\n", rv );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportContentData( encryptedContent, sizeof( encryptedContent ), __outDataBuf );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to import content data, rv=%d\n", rv );
        goto end;
    }

    for( i = 0; i < sizeof( decryptedContent ); i++ )
    {
        if( __outDataBuf[i] != decryptedContent[i] )
        {
            ES_TEST_LOG( "Failed to decrypt data %u:0x%02x:0x%02x\n", i, decryptedContent[i], __outDataBuf[i] );
            rv = ES_ERR_FAIL;
            goto end;
        }
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportContentEnd();
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to verify content data, rv=%d at line %d\n", rv, __LINE__ );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportContentBegin( ES_TEST_CONTENT_ID_2, tmd, __oldCerts, __nOldCerts );
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to begin content import, rv=%d\n", rv );
        goto end;
    }

    importedBytes = 0;

    while( importedBytes < sizeof( cnt5_enc ) )
    {
        bytes = MIN( sizeof( cnt5_enc ) - importedBytes, sizeof( __outDataBuf ) );

        ES_LOCK_BASIC_OPS();
        rv = es.ImportContentData( &cnt5_enc[importedBytes], bytes, __outDataBuf );
        ES_UNLOCK_BASIC_OPS();
        if( rv != ES_ERR_OK )
        {
            ES_TEST_LOG( "Failed to import content data, rv=%d\n", rv );
            goto end;
        }

        for( i = 0; i < bytes; i++ )
        {
            if( ( importedBytes + i ) >= sizeof( cnt5 ) )
            {
                break;
            }

            if( __outDataBuf[i] != cnt5[importedBytes + i] )
            {
                ES_TEST_LOG( "Failed to decrypt data %u:0x%02x:0x%02x\n", i, cnt5[importedBytes + i], __outDataBuf[i] );
                rv = ES_ERR_FAIL;
                goto end;
            }
        }

        importedBytes += bytes;
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportContentEnd();
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to verify content data, rv=%d at line %d (ILF %d, BLF %d)\n", rv, __LINE__, __importLockFail, __basicLockFail );
        goto end;
    }

    ES_LOCK_BASIC_OPS();
    rv = es.ImportTitleDone();
    ES_UNLOCK_BASIC_OPS();
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to complete title import, rv=%d\n", rv );
        goto end;
    }

    // ES_TEST_LOG("Passed import title test\n");
    ES_TEST_LOG( "Passed import title test (ILF %d, BLF %d)\n", __importLockFail, __basicLockFail );

end:
    ES_UNLOCK_IMPORT_OPS();
    return rv;
}  // NOLINT (readability/fn_size)


static void __mixedOpWorkerThread( ThreadParam *param )
{
    ESError rv = ES_ERR_OK;
    s32 loopCount = 0;
    s32 loopMask = 0x7;
    s32 delay = 100; /* milliseconds */
    nn::os::TimerEvent timer( nn::os::EventClearMode_ManualClear );

    ES_TEST_LOG( "-> Thread %d waiting for start event\n", param->threadNum );
    param->pWaitEvent->Wait();
    ES_TEST_LOG( "-> Thread %d starting ...\n", param->threadNum );

    /*
     * Main loop to drive the test in each thread
     */
    do
    {
        if( param->threadNum == 1 )
        {
#ifdef RANDOM_DELAY_THRESHOLD
            rv = IOSC_GenerateRand( (u8 *)&delay, sizeof( delay ) );
            delay %= RANDOM_DELAY_THRESHOLD;
            if( delay < 0 )
            {
                delay = -delay;
            }
            delay += 1000;
            // ES_TEST_LOG("-> Thread %d delay %d ms\n", param->threadNum, delay);
            ES_TEST_LOG( "-> Thread %d delay %d ms (ILF %d, BLF %d)\n", param->threadNum, delay, __importLockFail, __basicLockFail );
#endif
            timer.StartOneShot( nn::fnd::TimeSpan::FromMilliSeconds( delay ) );
            timer.Wait();
            loopMask = 0x1f;
        }
        if( ( ++loopCount & loopMask ) == 0 )
        {
            // ES_TEST_LOG("-> Thread %d loop %d\n", param->threadNum, loopCount);
            ES_TEST_LOG( "-> Thread %d loop %d (ILF %d, BLF %d)\n", param->threadNum, loopCount, __importLockFail, __basicLockFail );
        }
        if( param->threadNum == 1 )
        {
            MemoryInputStream ticket( tkt_h256, sizeof( tkt_h256 ) );
            rv = __testReadTicket( ticket );
        }
        else
        {
            rv = __testImportTitle();
        }
    } while( rv == ES_ERR_OK );

    // ES_TEST_LOG("=> FAIL: Thread %d loop %d exiting ...\n", param->threadNum, loopCount);
    ES_TEST_LOG( "=> FAIL: Thread %d loop %d exiting (ILF %d, BLF %d) ...\n", param->threadNum, loopCount, __importLockFail, __basicLockFail );
}


static ESError __testMixedOps()
{
    ESError rv = ES_ERR_OK;
    RESULT_NS( Result ) result;
    nn::os::Thread thread1;
    ThreadParam thread1Param;
    nn::os::Event thread1Wait( false );
    nn::os::StackBuffer<4096> thread1Stack;
    nn::os::Thread thread2;
    ThreadParam thread2Param;
    nn::os::Event thread2Wait( false );
    nn::os::StackBuffer<4096> thread2Stack;

    ES_TEST_LOG( "Start Mixed ES Operations threaded test\n" );

    /*
     * This procedure runs in the main thread and creates the
     * worker threads to run the tests
     */
    thread1Param.threadNum = 1;
    thread1Param.pWaitEvent = &thread1Wait;
    thread1.Start( __mixedOpWorkerThread, &thread1Param, thread1Stack );
    thread1.ChangePriority( nn::os::DEFAULT_THREAD_PRIORITY + 1 );

    thread2Param.threadNum = 2;
    thread2Param.pWaitEvent = &thread2Wait;
    thread2.Start( __mixedOpWorkerThread, &thread2Param, thread2Stack );
    thread2.ChangePriority( nn::os::DEFAULT_THREAD_PRIORITY + 2 );

    /*
     * Release both threads, but thread1 is higher priority.  It will
     * sleep on a random timer and preempt thread2 to test locking.
     */
    thread2Wait.Signal();
    /* Give thread2 a head start ... */
    nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds( 1000 ) );
    thread1Wait.Signal();

    /*
     * Wait for the workers to finish
     */
    ES_TEST_LOG( "Main thread: wait for worker threads to exit\n" );

    /*
     * Thread 1 is higher priority, so it will step on thread 2.  If
     * thread2 exits, that means a failure has occurred.
     */
    thread2.Join();
    ES_TEST_LOG( "Main thread: thread2 exited\n" );
#ifdef USE_OBSOLETE_CODE
    /* XXX this doesn't work:  need to kill thread1 */
    thread1.Join();
    ES_TEST_LOG( "Main thread: thread1 exited\n" );
#endif

    thread2.Finalize();
    thread1.Finalize();

    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "FAIL: Mixed ES Operations threaded test\n" );
        goto end;
    }

    ES_TEST_LOG( "End of Mixed ES Operations threaded test\n" );

end:
    return rv;
}
#endif  // MIXED_OPS_TEST


static ESError __testCleanup()
{
#ifdef CTR_TEST
    esImportOpMutex.Finalize();
    esBasicOpMutex.Finalize();
#endif

    return ES_ERR_OK;
}


TEST( ThreadsTest, Threads )
{
    ESError rv = ES_ERR_OK;

    rv = __testSetup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

#if defined( BASIC_OPS_TEST )
    rv = __testBasicOps();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }
#endif

#if defined( MIXED_OPS_TEST )
    rv = __testMixedOps();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }
#endif

    rv = __testCleanup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

end:
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "***** Passed ES concurrency tests *****\n" );
    }

    return;
}
