﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */


/*
 * Test the linux FileStream storage class, which is required for testing
 * ES functionality.  This test probably only makes sense on Linux, but there
 * is an implementation of FileStream on CTR as well.  The linux implementation
 * is not as complete, since we only need it for testing.
 */
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>

#include <nn/escore/escore.h>
#include <nnt/escoreUtil/testEscore_util_istorage.h>

USING_ESCORE_UTIL_NAMESPACE
USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE

#include "../Common/testEs_Est_utils.h"


static u8 __inBuf[1024];
static u8 __outBuf[1024];


#define CHECK_GOOD( result )                                           \
    do                                                                 \
    {                                                                  \
        if( !result.IsSuccess() )                                      \
        {                                                              \
            ES_TEST_LOG( "FAIL:  Bad result at line %d\n", __LINE__ ); \
            if( rv == ES_ERR_OK )                                      \
            {                                                          \
                rv = ES_ERR_INVALID;                                   \
            }                                                          \
            goto end;                                                  \
        }                                                              \
    } while( 0 )

#define CHECK_BAD( result )                                                              \
    do                                                                                   \
    {                                                                                    \
        if( result.IsSuccess() )                                                         \
        {                                                                                \
            ES_TEST_LOG( "FAIL:  Operation should have failed at line %d\n", __LINE__ ); \
            if( rv == ES_ERR_OK )                                                        \
            {                                                                            \
                rv = ES_ERR_INVALID;                                                     \
            }                                                                            \
            goto end;                                                                    \
        }                                                                                \
    } while( 0 )

#define MIN( a, b ) ( ( ( a ) < ( b ) ) ? ( a ) : ( b ) )

static const char *makeAbsPath( const char *fileName )
{
    static std::string path;

    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/Istorage/testEs_Istorage/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    return path.c_str();
}

/*
 * Put a simple-minded pattern in the buffer based on file offset
 */
static void __fillBuf( u8 *buf, u32 len, u32 offset )
{
    while( len-- > 0 )
    {
        *buf++ = static_cast<u8>( offset++ & 0xff );
    }

    return;
}


/*
 * Check a pattern from __fillBuf
 */
static bool __checkBuf( u8 *buf, u32 len, u32 offset )
{
    u8 *start = buf;

    while( len-- > 0 )
    {
        if( *buf != static_cast<u8>( offset & 0xff ) )
        {
            NN_LOG( "__checkbuf: data miscompare at buf offset %td addr %0*" PRIxPTR " expect %02x != %02x\n", ( buf - start ),
                    (int)sizeof( buf ) * 2, (uintptr_t)buf, *buf, offset & 0xff );
            return false;
        }
        buf++, offset++;
    }

    return true;
}


static ESError __testSetup()
{
    return ES_ERR_OK;
}


static ESError __testBasicFileOps()
{
    ESError rv = ES_ERR_OK;
    RESULT_NS( Result ) result;
    FileStream ticketStream;
    s64 fileSize;
    s64 filePos;
    s32 writeCount;
    s32 readCount;
    s32 readSize;
    s64 totalRead;

    ES_TEST_LOG( "Basic File Operations test\n" );

    const char *path = "istorage_test.tmp";

    result = ticketStream.TryInitialize( makeAbsPath(path), OPEN_MODE_WRITE | OPEN_MODE_CREATE );
    CHECK_GOOD( result );
    result = ticketStream.TryGetSize( &fileSize );
    CHECK_GOOD( result );
    if( fileSize != 0LL )
    {
        ES_TEST_LOG( "Failed:  file size on newly created file != 0 (%lld)\n", (int64_t)fileSize );
        rv = ES_ERR_INVALID;
        goto end;
    }
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != 0LL )
    {
        ES_TEST_LOG( "Failed:  file position on newly created file != 0 (%lld)\n", (int64_t)filePos );
        rv = ES_ERR_INVALID;
        goto end;
    }
    __fillBuf( __outBuf, sizeof( __outBuf ), (u32)filePos );
    result = ticketStream.TryWrite( &writeCount, __outBuf, sizeof( __outBuf ) );
    CHECK_GOOD( result );
    ES_TEST_LOG( "Wrote %d bytes to file %s\n", writeCount, path );
    result = ticketStream.TryGetSize( &fileSize );
    CHECK_GOOD( result );
    if( fileSize != sizeof( __outBuf ) )
    {
        ES_TEST_LOG( "Failed:  file size of file is wrong %lld != %zd\n", (int64_t)fileSize, sizeof( __outBuf ) );
        rv = ES_ERR_INVALID;
        goto end;
    }
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( (u32)filePos != sizeof( __outBuf ) )
    {
        ES_TEST_LOG( "Failed:  file position is wrong %PRId64 != %zd\n", (int64_t)filePos, sizeof( __outBuf ) );
        rv = ES_ERR_INVALID;
        goto end;
    }

    /*
     * Test Seek and SetPosition specifiers
     */
    result = ticketStream.TrySeek( 0, POSITION_BASE_BEGIN );
    CHECK_GOOD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != 0LL )
    {
        ES_TEST_LOG( "Failed:  file position after seek to 0 = %lld\n", (int64_t)filePos );
        rv = ES_ERR_INVALID;
        goto end;
    }

    result = ticketStream.TrySeek( 0, POSITION_BASE_END );
    CHECK_GOOD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != fileSize )
    {
        ES_TEST_LOG( "Failed:  file position after seek to EOF = %lld (should be %PRId64)\n", (int64_t)filePos, (int64_t)fileSize );
        rv = ES_ERR_INVALID;
        goto end;
    }

    result = ticketStream.TrySeek( -3LL, POSITION_BASE_END );
    CHECK_GOOD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != fileSize - 3 )
    {
        ES_TEST_LOG( "Failed:  file position after seek to EOF - 3 = %lld (should be %PRId64)\n", (int64_t)filePos, (int64_t)fileSize - 3 );
        rv = ES_ERR_INVALID;
        goto end;
    }

    result = ticketStream.TrySeek( -7LL, POSITION_BASE_CURRENT );
    CHECK_GOOD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != fileSize - 10LL )
    {
        ES_TEST_LOG( "Failed:  file position after seek to CUR - 7 = %lld (should be %PRId64)\n", (int64_t)filePos, (int64_t)fileSize - 10 );
        rv = ES_ERR_INVALID;
        goto end;
    }

    result = ticketStream.TrySeek( -3LL, POSITION_BASE_BEGIN );
    CHECK_BAD( result );
    result = ticketStream.TrySetPosition( -11LL );
    CHECK_BAD( result );

    /*
     * Try to read when opened for WRITE only
     */
    result = ticketStream.TryRead( &readCount, __inBuf, sizeof( __inBuf ) );
    CHECK_BAD( result );

    /*
     * Extend the file
     */
    result = ticketStream.TrySeek( 0, POSITION_BASE_END );
    CHECK_GOOD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    for( int i = 0; i < 20; i++ )
    {
        __fillBuf( __outBuf, sizeof( __outBuf ), ( u32 )( filePos & 0xffffffff ) );
        result = ticketStream.TryWrite( &writeCount, __outBuf, sizeof( __outBuf ) );
        CHECK_GOOD( result );
        filePos += sizeof( __outBuf );
    }
    __fillBuf( __outBuf, sizeof( __outBuf ), ( u32 )( filePos & 0xffffffff ) );
    result = ticketStream.TryWrite( &writeCount, __outBuf, 23 );
    CHECK_GOOD( result );

    ticketStream.Finalize();

    /*
     * Open the file for READ and verify the contents
     */
    result = ticketStream.TryInitialize( makeAbsPath(path), OPEN_MODE_READ );
    CHECK_GOOD( result );
    result = ticketStream.TryGetSize( &fileSize );
    CHECK_GOOD( result );
    ES_TEST_LOG( "Opened file %s for READ, size %lld\n", path, (int64_t)fileSize );

    /*
     * Try to write when opened for READ only
     */
    result = ticketStream.TryWrite( &writeCount, __outBuf, sizeof( __outBuf ) );
    CHECK_BAD( result );
    result = ticketStream.TryGetPosition( &filePos );
    CHECK_GOOD( result );
    if( filePos != 0LL )
    {
        ES_TEST_LOG( "Failed:  bogus write modified file position 0 != %lld\n", (int64_t)filePos );
        rv = ES_ERR_INVALID;
        goto end;
    }

    totalRead = 0;
    while( totalRead < fileSize )
    {
        readSize = static_cast<s32>( MIN( static_cast<s64>( sizeof( __inBuf ) ), fileSize - totalRead ) );
        result = ticketStream.TryRead( &readCount, __inBuf, readSize );
        CHECK_GOOD( result );
        if( !__checkBuf( __inBuf, readCount, ( u32 )( filePos & 0xffffffff ) ) )
        {
            ES_TEST_LOG( "Failed:  file data is not as expected\n" );
            rv = ES_ERR_INVALID;
            goto end;
        }
        filePos += readCount;
        totalRead += readCount;
    }

    ES_TEST_LOG( "End of Basic File Operations test\n" );

end:
    return rv;
}  // NOLINT (readability/fn_size)


static ESError __testReadWriteSameStream()
{
    ESError rv = ES_ERR_OK;
    RESULT_NS( Result ) result;
    FileStream ticketStream;
    s64 fileSize;
    s64 readPos;
    s64 writePos;
    s32 writeCount;
    s32 readCount;
    s32 readSize;

    /*
     * Nintendo wants to be able to have one stream be used for intermixed read
     * and write operations and get the expected results.  The application is
     * rewriting a large ticket in place.
     */
    ES_TEST_LOG( "Read/Write Same Stream\n" );

    const char *path = "istorage_test.tmp";

    result = ticketStream.TryInitialize( makeAbsPath(path), OPEN_MODE_READ | OPEN_MODE_WRITE );
    CHECK_GOOD( result );
    result = ticketStream.TryGetSize( &fileSize );
    CHECK_GOOD( result );

    ES_TEST_LOG( "File %s size %lld\n", path, fileSize );

    readPos = 0;
    writePos = 0;

    /*
     * Loop through reading each chunk and then backing up and rewriting it
     */
    for( int i = 0; readPos < fileSize; i++ )
    {
        readSize = static_cast<s32>( MIN( sizeof( __inBuf ), fileSize - readPos ) );
        result = ticketStream.TrySeek( readPos, POSITION_BASE_BEGIN );
        CHECK_GOOD( result );
        result = ticketStream.TryRead( &readCount, __inBuf, readSize );
        CHECK_GOOD( result );
        if( readCount != readSize )
        {
            ES_TEST_LOG( "Failed:  short read from %s (exp %d, got %d)\n", path, readSize, readCount );
            rv = ES_ERR_INVALID;
            goto end;
        }
        if( !__checkBuf( __inBuf, readCount, ( u32 )( readPos & 0xffffffff ) ) )
        {
            ES_TEST_LOG( "Failed:  file data is not as expected\n" );
            rv = ES_ERR_INVALID;
            goto end;
        }
        readPos += readCount;
        result = ticketStream.TrySeek( writePos, POSITION_BASE_BEGIN );
        CHECK_GOOD( result );
        memset( __outBuf, i, sizeof( __outBuf ) );
        if( readCount > (s32)sizeof( __outBuf ) )
        {
            /*
             * This "can't happen", but check for grins
             */
            ES_TEST_LOG( "Failed:  output buffer too small!\n" );
            rv = ES_ERR_INVALID;
            goto end;
        }
        result = ticketStream.TryWrite( &writeCount, __outBuf, readCount );
        CHECK_GOOD( result );
        if( readCount != writeCount )
        {
            ES_TEST_LOG( "Failed:  short write to %s (exp %d, wrote %d)\n", path, readCount, writeCount );
            rv = ES_ERR_INVALID;
            goto end;
        }
        writePos += writeCount;
    }

    /*
     * Go back and check that the writes are correctly reflected in the file
     */
    readPos = 0;
    for( int i = 0; readPos < fileSize; i++ )
    {
        readSize = static_cast<s32>( MIN( sizeof( __inBuf ), fileSize - readPos ) );
        result = ticketStream.TrySeek( readPos, POSITION_BASE_BEGIN );
        CHECK_GOOD( result );
        result = ticketStream.TryRead( &readCount, __inBuf, readSize );
        CHECK_GOOD( result );
        if( readCount != readSize )
        {
            ES_TEST_LOG( "Failed:  short read from %s (exp %d, got %d)\n", path, readSize, readCount );
            rv = ES_ERR_INVALID;
            goto end;
        }
        memset( __outBuf, i, sizeof( __outBuf ) );
        if( memcmp( __inBuf, __outBuf, readCount ) != 0 )
        {
            ES_TEST_LOG( "Failed:  file data is wrong after the rewrite!\n" );
            rv = ES_ERR_INVALID;
            goto end;
        }
        readPos += readCount;
    }

    ES_TEST_LOG( "Passed:  file data is correct after the rewrite!\n" );

    ticketStream.Finalize();

    ES_TEST_LOG( "End of Read/Write Same Stream test\n" );

end:
    return rv;
}  // NOLINT (readability/fn_size)


static ESError __testCleanup()
{
    return ES_ERR_OK;
}

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

TEST( IStorageTest, IStorage )
{
    ESError rv = ES_ERR_OK;

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();
    rv = __testSetup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testBasicFileOps();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testReadWriteSameStream();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testCleanup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

end:
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "***** Passed istorage tests *****\n" );
    }

    return;
}
