﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2008, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished proprietary information of BroadOn Communications Corp.,
 *  and are protected by Federal copyright law. They may not be disclosed
 *  to third parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <cstring>
#include <iostream>

#include <nnt/nntest.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>

#define TESTNAME "NinDevCertTest"
#define NEED_NUMTICKS

#include "testEs_Testcommon.h"

#include <cstdio>
//#include <sys/types.h>
//#include <sys/stat.h>
#include <nn/ioscrypto/iosccert.h>

/*
 * Third set of experiments from N 1/21/10 are
 * certs 0x08, 0x10 for dev and 0x0b, 0x1f for prod
 * These verify successfully.
 */
NN_ALIGNAS( 64 ) u8 ninDevPubKey[] = {
    0x00, 0xe4, 0x84, 0x23, 0xac, 0x7e, 0x49, 0xe3, 0x69, 0x5d, 0x61, 0x6c, 0x6f, 0xf0, 0x20, 0xfb, 0x6a, 0x5c, 0x47, 0x45,
    0x63, 0x77, 0x4d, 0x55, 0x04, 0x64, 0x50, 0x5f, 0x23, 0x67, 0x01, 0x88, 0x99, 0x94, 0xce, 0xe0, 0x26, 0x0e, 0xbb, 0x8f,
    0x1b, 0x0e, 0xd3, 0xa8, 0xe1, 0x47, 0x12, 0xa2, 0x5a, 0x11, 0xc4, 0x85, 0x77, 0xa3, 0xce, 0xdd, 0xb4, 0xc2, 0x7c, 0x4d,
};

const char ninCertD1[] = "pki_data/etd_0015_dev_cert.raw";
const char ninCertD2[] = "pki_data/etd_002f_dev_cert.raw";
const char ninCertD3[] = "pki_data/etd_ffff_dev_cert.raw";

NN_ALIGNAS( 64 ) u8 ninProdPubKey[] = {
    0x00, 0x4e, 0x3b, 0xb7, 0x4d, 0x5d, 0x95, 0x9e, 0x68, 0xce, 0x90, 0x04, 0x34, 0xfe, 0x9e, 0x4a, 0x3f, 0x09, 0x4a, 0x33,
    0x77, 0x1f, 0xa7, 0xc0, 0xe4, 0xb0, 0x23, 0x26, 0x4d, 0x98, 0x01, 0x4c, 0xa1, 0xfc, 0x79, 0x9d, 0x3f, 0xa5, 0x21, 0x71,
    0xd5, 0xf9, 0xbd, 0x5b, 0x17, 0x77, 0xec, 0x0f, 0xef, 0x7a, 0x38, 0xd1, 0x66, 0x9b, 0xbf, 0x83, 0x03, 0x25, 0x84, 0x3a,
};

const char ninCertP1[] = "pki_data/etd_000f_prod_cert.raw";
const char ninCertP2[] = "pki_data/etd_0036_prod_cert.raw";

NN_ALIGNAS( ALIGN_SIZE ) static IOSCEccSignedCert ninDevCert;

static void readFile( const char *fileName, void* buffer, size_t size)
{
    std::string path;
    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/CryptoRegression/testEs_CryptoRegression/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    nn::fs::FileHandle file;
    nn::Result result;
    result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    size_t readSize = 0;
    result = nn::fs::ReadFile(&readSize, file, 0, buffer, size, nn::fs::ReadOption());
    if (readSize == 0)
    {
        LOG_MESSAGE( "read failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

int verifyDevCert( const char *fileName, IOSCPublicKeyHandle verifyKeyHandle )
{
    IOSCPublicKeyHandle certKeyHandle;
    IOSCHashContext hashCtx;
    IOSCHash256 hashVal;
    int hashType = IOSC_HASH_SHA256;
    int hashSize = 32;
    IOSCError rv;

    readFile(fileName, ninDevCert, sizeof( ninDevCert ));

    rv = IOSC_CreateObject( &certKeyHandle, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "CreateObject" );

    /*
     * Verify the certificate by importing it
     */
    rv = IOSC_ImportCertificate( ninDevCert, verifyKeyHandle, certKeyHandle );
    if( rv == IOSC_ERROR_OK )
    {
        LOG_PASS( "ImportCertificate OK for %s", fileName );
    }
    else
    {
        LOG_FAIL( "ImportCertificate FAILED for %s", fileName );
    }

    /*
     * Now try verifying it manually (first with SHA256)
     */
    rv = IOSC_GenerateHash( hashCtx, NULL, 0, hashType | IOSC_HASH_FIRST, NULL );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "GenerateHash FIRST" );

    rv = IOSC_GenerateHash( hashCtx, &ninDevCert[128], sizeof( ninDevCert ) - 128, hashType | IOSC_HASH_LAST, hashVal );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "GenerateHash LAST" );

    rv = IOSC_VerifyPublicKeySign( hashVal, hashSize, verifyKeyHandle, &ninDevCert[4] );
    if( rv == IOSC_ERROR_OK )
    {
        LOG_PASS( "VerifySign H256 OK for %s", fileName );
    }
    else
    {
        LOG_FAIL( "VerifySign H256 FAILED for %s", fileName );
    }

    rv = IOSC_DeleteObject( certKeyHandle );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "DeleteObject" );

    return 0;
}

int doNinDevCertTest()
{
    IOSCError rv;
    IOSCPublicKeyHandle verifyKeyHandle;

    /*
     * Test with keys and certs from Nintendo
     */
    rv = IOSC_CreateObject( &verifyKeyHandle, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "CreateObject" );
    rv = IOSC_ImportPublicKey( ninDevPubKey, 0, verifyKeyHandle );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "Import N dev public key" );

    /*
     * Verify the certs with Dev Public Key
     */
    verifyDevCert( ninCertD1, verifyKeyHandle );
    verifyDevCert( ninCertD2, verifyKeyHandle );
    verifyDevCert( ninCertD3, verifyKeyHandle );

    rv = IOSC_DeleteObject( verifyKeyHandle );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "DeleteObject" );

    rv = IOSC_CreateObject( &verifyKeyHandle, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "CreateObject" );
    rv = IOSC_ImportPublicKey( ninProdPubKey, 0, verifyKeyHandle );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "Import N prod public key" );

    /*
     * Verify the certs with Prod Public Key
     */
    verifyDevCert( ninCertP1, verifyKeyHandle );
    verifyDevCert( ninCertP2, verifyKeyHandle );

    rv = IOSC_DeleteObject( verifyKeyHandle );
    CHECK_RETURN( rv, IOSC_ERROR_OK, "DeleteObject" );

    return 0;
}


#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doNinDevCertTest();
    EXIT( 0 );
    return 0;
}
#endif
