﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/esUtil/testEs_Publish.h>
#include <nnt/esUtil/testEs_Utility.h>
#include "es_TicketDatabase.h"

const char* MountName = "ticketdatabase";
const char* TicketDatabasePath = "ticketdatabase:/";
const int MaxTicketCount = 512;
const int TicketSize = 1024;

class TicketDatabaseTest : public ::testing::Test
{
private:
#ifdef USE_RANDOM_DEVICE
    std::random_device randomDevice;
#endif
    std::mt19937_64 randomEngine64;

protected:
    nn::es::TicketDatabase ticketDatabase;

    uint32_t CreateTicketAtRandom(void* outTicketBuffer, int outTicketBufferSize, nn::es::TicketInfo* outTicketInfo)
    {
        nn::es::TitleKey titleKey;
        nn::es::DeviceId deviceId = randomEngine64();
        nn::es::TicketId ticketId = randomEngine64();
        nn::es::RightsIdIncludingKeyId rightsId;
        nn::es::AccountId accountId = static_cast<uint32_t>(randomEngine64());

        uint64_t upperRandomNumber = randomEngine64();
        uint64_t lowerRandomNumber = randomEngine64();
        memcpy(&titleKey.key, &upperRandomNumber, sizeof(uint64_t));
        memcpy(reinterpret_cast<uint8_t*>(&titleKey.key) + sizeof(uint64_t), &lowerRandomNumber, sizeof(uint64_t));

        upperRandomNumber = randomEngine64();
        lowerRandomNumber = randomEngine64();
        memcpy(&rightsId, &upperRandomNumber, sizeof(uint64_t));
        memcpy(reinterpret_cast<uint8_t*>(&rightsId) + sizeof(uint64_t), &lowerRandomNumber, sizeof(uint64_t));

        uint32_t ticketSize = nnt::es::PublishTicket(outTicketBuffer, outTicketBufferSize, titleKey, deviceId, ticketId, rightsId, accountId);

        *outTicketInfo = nnt::es::CreateTicketInfo(outTicketBuffer, ticketSize);

        return ticketSize;
    }

    virtual void SetUp()
    {
        ticketDatabase.Initialize();
        ticketDatabase.DeleteAllTicket();

        NN_SDK_ASSERT_EQUAL(ticketDatabase.CountTicket(), 0);
    }

    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost(MountName, "./"));
    }

    static void TearDownTestCase()
    {
        nn::fs::Unmount(MountName);
    }

public:
#ifdef USE_RANDOM_DEVICE
    TicketDatabaseTest() : randomEngine64(randomDevice()), ticketDataBase(MaxTicketCount, TicketDatabasePath, nullptr) {}
#else
    TicketDatabaseTest() : randomEngine64(1), ticketDatabase(MaxTicketCount, TicketDatabasePath, nullptr) {}
#endif
};

TEST_F(TicketDatabaseTest, Import)
{
    for (int i = 1; i <= MaxTicketCount; i++)
    {
        char ticketBuffer[TicketSize];
        nn::es::TicketInfo ticketInfo;
        uint32_t ticketSize;

        NN_LOG("Try Import Ticket: %d / %d\n", i, MaxTicketCount);

        // チケットデータベースが保存上限に達していないことを確認
        EXPECT_FALSE(ticketDatabase.IsFull());

        // ランダムにチケットを生成してインポートする
        // RightsId と TicketId の組が一致するチケットが既に保存されていると失敗するので、成功するまでインポートを繰り返す
        while (NN_STATIC_CONDITION(true))
        {
            ticketSize = CreateTicketAtRandom(ticketBuffer, TicketSize, &ticketInfo);

            nn::Result result = ticketDatabase.ImportTicket(ticketBuffer, ticketSize);

            if (result.IsSuccess())
            {
                break;
            }
        }

        // チケットの枚数が正しいか確認
        EXPECT_EQ(i, ticketDatabase.CountTicket());

        // インポートしたチケットが存在するか確認
        EXPECT_TRUE(ticketDatabase.ExistTicket(ticketInfo.rightsId, ticketInfo.ticketId).IsSuccess());

        // インポートしたチケットが存在し、RigthsId に対応した ticketId が正しく取得できるか確認
        nn::es::TicketId outTicketId;
        EXPECT_TRUE(ticketDatabase.FindTicket(&outTicketId, ticketInfo.rightsId).IsSuccess());
        EXPECT_EQ(ticketInfo.ticketId, outTicketId);

        // インポートしたチケットのサイズが正しいか確認
        EXPECT_EQ(ticketSize, ticketDatabase.GetTicketSize(ticketInfo.rightsId, ticketInfo.ticketId));

        // インポートしたチケットのデータが正しいか確認
        char outTicketBuffer[TicketSize];
        size_t tmpTicketSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ticketDatabase.GetTicketData(&tmpTicketSize, outTicketBuffer, sizeof(outTicketBuffer), ticketInfo.rightsId, ticketInfo.ticketId));
        EXPECT_EQ(tmpTicketSize, static_cast<size_t>(ticketSize));
        EXPECT_TRUE(std::memcmp(ticketBuffer, outTicketBuffer, ticketSize) == 0);
    }

    // チケットデータベースの上限まで保存した後に、チケットを全削除して正しく反映されているか確認
    {
        // チケットデータベースが保存上限に達していることを確認
        EXPECT_TRUE(ticketDatabase.IsFull());
        EXPECT_EQ(MaxTicketCount, ticketDatabase.CountTicket());

        // チケットを全削除して反映されているか確認
        ticketDatabase.DeleteAllTicket();
        EXPECT_EQ(0, ticketDatabase.CountTicket());
    }
}

TEST_F(TicketDatabaseTest, Delete)
{
    char ticketBuffer[TicketSize];
    nn::es::TicketInfo ticketInfo;
    uint32_t ticketSize;

    // チケットをインポートして正しく情報が反映されているか確認
    {
        ticketSize = CreateTicketAtRandom(ticketBuffer, TicketSize, &ticketInfo);

        ticketDatabase.ImportTicket(ticketBuffer, ticketSize);

        // チケットの枚数が正しいか確認
        EXPECT_EQ(1, ticketDatabase.CountTicket());

        // インポートしたチケットが存在するか確認
        EXPECT_TRUE(ticketDatabase.ExistTicket(ticketInfo.rightsId, ticketInfo.ticketId).IsSuccess());

        // インポートしたチケットが存在し、RigthsId に対応した ticketId が正しく取得できるか確認
        nn::es::TicketId outTicketId;
        EXPECT_TRUE(ticketDatabase.FindTicket(&outTicketId, ticketInfo.rightsId).IsSuccess());
        EXPECT_EQ(ticketInfo.ticketId, outTicketId);

        // インポートしたチケットのサイズが正しいか確認
        EXPECT_EQ(ticketSize, ticketDatabase.GetTicketSize(ticketInfo.rightsId, ticketInfo.ticketId));

        // インポートしたチケットのデータが正しいか確認
        char outTicketBuffer[TicketSize];
        size_t tmpTicketSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ticketDatabase.GetTicketData(&tmpTicketSize, outTicketBuffer, sizeof(outTicketBuffer), ticketInfo.rightsId, ticketInfo.ticketId));
        EXPECT_EQ(tmpTicketSize, static_cast<size_t>(ticketSize));
        EXPECT_TRUE(std::memcmp(ticketBuffer, outTicketBuffer, ticketSize) == 0);
    }

    // チケットを削除して正しく情報が反映されているか確認
    {
        ticketDatabase.DeleteTicket(ticketInfo.rightsId, ticketInfo.ticketId);

        // チケットの枚数が正しいか確認
        EXPECT_EQ(0, ticketDatabase.CountTicket());

        // 削除したチケットが存在しないことを確認
        EXPECT_FALSE(ticketDatabase.ExistTicket(ticketInfo.rightsId, ticketInfo.ticketId).IsSuccess());

        // 削除したチケットが見つからないことを確認
        nn::es::TicketId outTicketId;
        EXPECT_FALSE(ticketDatabase.FindTicket(&outTicketId, ticketInfo.rightsId).IsSuccess());

        // 削除したチケットのサイズが 0 となっているか確認
        EXPECT_EQ(0, ticketDatabase.GetTicketSize(ticketInfo.rightsId, ticketInfo.ticketId));
    }
}
