﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/err/err_SystemApi.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_StringUtil.h>

#include <err_StringUtil.h>

using namespace nn;

namespace
{
    NN_ALIGNAS(8) char g_StartupParamBuffer[err::ErrorViewerStartupParamSizeMax];
}

TEST(SystemApi, ConvertResultToErrorCode)
{
    auto result = nn::fs::ResultAllocationMemoryFailed();
    auto errorCode = err::ConvertResultToErrorCode(result);
    EXPECT_EQ(result.GetModule() + 2000, errorCode.category);
    EXPECT_EQ(result.GetDescription(), errorCode.number);
}

TEST(SystemApi, GetErrorCodeString)
{
    err::ErrorCode errorCode;
    errorCode.category = 2345;
    errorCode.number = 6789;
    char errorCodeString[err::ErrorCode::StringLengthMax];
    err::GetErrorCodeString(errorCodeString, sizeof(errorCodeString), errorCode);
    EXPECT_STREQ("2345-6789", errorCodeString);
}

TEST(StringUtil, MakeErrorCodeString)
{
    err::ErrorCode errorCode;
    errorCode.category = 2345u;
    errorCode.number = 6789u;
    char errorCodeStr[err::ErrorCode::StringLengthMax];
    err::detail::MakeErrorCodeString(errorCodeStr, err::ErrorCode::StringLengthMax, errorCode);
    EXPECT_STREQ("2345-6789", errorCodeStr);
}

TEST(StringUtil, MakeApplicationErrorCodeString)
{
    ns::ApplicationErrorCodeCategory category;
    util::Strlcpy(category.value, "ABCDE", sizeof(category.value));
    err::ApplicationErrorCodeNumber number = 1234u;
    char errorCodeStr[err::ErrorCode::StringLengthMax];
    err::detail::MakeApplicationErrorCodeString(errorCodeStr, err::ErrorCode::StringLengthMax, category, number);
    EXPECT_STREQ("2-ABCDE-1234", errorCodeStr);
}

TEST(StringUtil, ParseErrorCodeString)
{
    err::ErrorCode errorCode;
    err::detail::ParseErrorCodeString(&errorCode, "2345-6789");
    EXPECT_EQ(2345u, errorCode.category);
    EXPECT_EQ(6789u, errorCode.number);
}

TEST(StringUtil, ParseApplicationErrorCodeString)
{
    ns::ApplicationErrorCodeCategory category;
    err::ApplicationErrorCodeNumber number;
    err::detail::ParseApplicationErrorCodeString(&category, &number, "2-ABCDE-1234");
    EXPECT_STREQ("ABCDE", category.value);
    EXPECT_EQ(1234u, number);
}

TEST(StringUtil, TryParseErrorCodeString)
{
    err::ErrorCode errorCode;
    EXPECT_TRUE(err::detail::TryParseErrorCodeString(&errorCode, "2345-6789"));
    EXPECT_EQ(2345u, errorCode.category);
    EXPECT_EQ(6789u, errorCode.number);

    EXPECT_FALSE(err::detail::TryParseErrorCodeString(&errorCode, "2-ABCDE-1234"));
    EXPECT_FALSE(err::detail::TryParseErrorCodeString(&errorCode, "ABCD-6789"));
    EXPECT_FALSE(err::detail::TryParseErrorCodeString(&errorCode, "2345-ABCD"));
    EXPECT_FALSE(err::detail::TryParseErrorCodeString(&errorCode, "2345-"));
    EXPECT_FALSE(err::detail::TryParseErrorCodeString(&errorCode, "-6789"));
}

TEST(StringUtil, TryParseApplicationErrorCodeString)
{
    ns::ApplicationErrorCodeCategory category;
    err::ApplicationErrorCodeNumber number;
    EXPECT_TRUE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "2-ABCDE-1234"));
    EXPECT_STREQ(category.value, "ABCDE");
    EXPECT_EQ(1234u, number);

    EXPECT_FALSE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "2345-6789"));
    EXPECT_FALSE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "2-ABCDE-AAAA"));
    EXPECT_FALSE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "2--6789"));
    EXPECT_FALSE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "2-ABCDE-"));
    EXPECT_FALSE(err::detail::TryParseApplicationErrorCodeString(&category, &number, "ABCDE-1245"));
}

TEST(CreateErrorViewerStartupParamForRecordedError, RecordedSystemData)
{
    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    memset(g_StartupParamBuffer, 0, sizeof(g_StartupParamBuffer));

    ASSERT_TRUE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2345-6789", nullptr, timeOfOccurrence));
    auto pParam = reinterpret_cast<const err::ErrorViewerStartupParamForRecordedSystemData*>(g_StartupParamBuffer);
    EXPECT_EQ(err::ErrorType::RecordedSystemData, pParam->common.errorType);
    EXPECT_FALSE(pParam->common.isJumpEnabled);
    EXPECT_EQ(2345u, pParam->errorCode.category);
    EXPECT_EQ(6789u, pParam->errorCode.number);
    EXPECT_EQ(1234, pParam->timeOfOccurrence.value);
    EXPECT_EQ(sizeof(err::ErrorViewerStartupParamForRecordedSystemData), actualParamSize);
}

TEST(CreateErrorViewerStartupParamForRecordedError, RecordedSystemError)
{
    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    memset(g_StartupParamBuffer, 0, sizeof(g_StartupParamBuffer));

    ASSERT_TRUE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2345-6789", "Description", timeOfOccurrence));
    auto pParam = reinterpret_cast<const err::ErrorViewerStartupParamForRecordedSystemError*>(g_StartupParamBuffer);
    EXPECT_EQ(err::ErrorType::RecordedSystemError, pParam->common.errorType);
    EXPECT_FALSE(pParam->common.isJumpEnabled);
    EXPECT_EQ(2345u, pParam->errorCode.category);
    EXPECT_EQ(6789u, pParam->errorCode.number);
    EXPECT_STREQ("Description", pParam->message);
    EXPECT_EQ(1234, pParam->timeOfOccurrence.value);
    EXPECT_EQ(sizeof(err::ErrorViewerStartupParamForRecordedSystemError), actualParamSize);
}

TEST(CreateErrorViewerStartupParamForRecordedError, RecordedApplicationError)
{
    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    memset(g_StartupParamBuffer, 0, sizeof(g_StartupParamBuffer));

    ASSERT_TRUE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2-ABCDE-6789", "Description", timeOfOccurrence));
    auto pParam = reinterpret_cast<const err::ErrorViewerStartupParamForRecordedApplicationError*>(g_StartupParamBuffer);
    EXPECT_EQ(err::ErrorType::RecordedApplicationError, pParam->common.errorType);
    EXPECT_FALSE(pParam->common.isJumpEnabled);
    EXPECT_STREQ("ABCDE", pParam->applicationErrorCodeCategory.value);
    EXPECT_EQ(6789u, pParam->applicationErrorCodeNumber);
    EXPECT_EQ(1234, pParam->timeOfOccurrence.value);
    EXPECT_STREQ("Description", pParam->message);
    EXPECT_EQ(sizeof(err::ErrorViewerStartupParamForRecordedApplicationError), actualParamSize);
}

TEST(CreateErrorViewerStartupParamForRecordedError, InvalidArgument)
{
    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    // エラーコードでもアプリケーションエラーコードでもないエラーコード文字列
    EXPECT_FALSE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "ABCDE-6789", nullptr, timeOfOccurrence));
    // メッセージを持たないアプリケーションエラー
    EXPECT_FALSE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2-ABCDE-6789", nullptr, timeOfOccurrence));
}
