﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include "testEns_TaskLargeSize.h"
#include "testEns_TaskUnstoppableEcho.h"

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void CreateLargeSizeTask(nn::ens::AsyncContext* pOutContext, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskLargeSize);

        pTask->SetParameter(span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }

    void CreateUnstoppableEchoTask(nn::ens::AsyncContext* pOutContext, int* pOutput, int input, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskUnstoppableEcho);

        pTask->SetParameter(pOutput, input, span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }
}

class EnsTask : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsTask, Echo)
{
    nn::ens::AsyncContext context;
    int output = 0;

    nn::os::Tick begin = nn::os::GetSystemTick();

    CreateUnstoppableEchoTask(&context, &output, 12345, nn::TimeSpan::FromMilliSeconds(120));
    context.GetEvent().Wait();

    NNT_EXPECT_RESULT_SUCCESS(context.GetResult());
    EXPECT_EQ(output, 12345);

    nn::os::Tick end = nn::os::GetSystemTick();

    EXPECT_GE((end - begin).ToTimeSpan().GetMilliSeconds(), 100);
}

TEST_F(EnsTask, Cancel)
{
    // タスク完了前のキャンセル
    {
        nn::ens::AsyncContext context;
        int output = 0;

        CreateUnstoppableEchoTask(&context, &output, 12345, nn::TimeSpan::FromMilliSeconds(100));

        EXPECT_FALSE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(50)));
        context.Cancel();
        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context.GetResult());
        EXPECT_EQ(output, 0);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(60));

        // タスク完了後に処理結果が上書きされないことを確認する。
        NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context.GetResult());
        EXPECT_EQ(output, 0);
    }
    // タスク完了後のキャンセル
    {
        nn::ens::AsyncContext context;
        int output = 0;

        CreateUnstoppableEchoTask(&context, &output, 12345, nn::TimeSpan::FromMilliSeconds(100));

        context.GetEvent().Wait();
        context.Cancel();

        NNT_EXPECT_RESULT_SUCCESS(context.GetResult());
        EXPECT_EQ(output, 12345);
    }
    // タスク実行前のキャンセル
    {
        nn::ens::AsyncContext context1;
        nn::ens::AsyncContext context2;
        int output1 = 0;
        int output2 = 0;

        CreateUnstoppableEchoTask(&context1, &output1, 12345, nn::TimeSpan::FromMilliSeconds(200));
        CreateUnstoppableEchoTask(&context2, &output2, 67890, nn::TimeSpan::FromMilliSeconds(200));

        nn::os::Tick begin = nn::os::GetSystemTick();

        context2.Cancel();
        context2.GetEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        EXPECT_LT((end - begin).ToTimeSpan().GetMilliSeconds(), 50);

        NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context2.GetResult());
        EXPECT_EQ(output2, 0);

        context1.GetEvent().Wait();
        context2.GetEvent().Wait();

        NNT_EXPECT_RESULT_SUCCESS(context1.GetResult());
        EXPECT_EQ(output1, 12345);

        // タスクが実行されていないことを確認する。
        NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context2.GetResult());
        EXPECT_EQ(output2, 0);
    }
}

TEST_F(EnsTask, OutOfMemory)
{
    NN_LOG("# sizeof (nnt::ens::TaskLargeSize) = %zu\n", sizeof (nnt::ens::TaskLargeSize));

    // (nn::ens::RequiredMemorySizeMin - アロケータの管理領域) / sizeof (nnt::ens::TaskLargeSize)
    // = (1 MB - 16 KB) / 131288 B ≒ 7.86
    // 上記より、8 個目は確保できないはず。
    nn::ens::AsyncContext contexts[8];

    for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
    {
        CreateLargeSizeTask(&contexts[i], nn::TimeSpan::FromMilliSeconds(50));
    }
    for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
    {
        contexts[i].GetEvent().Wait();
    }

    for (int i = 0; i < NN_ARRAY_SIZE(contexts) - 1; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(contexts[i].GetResult());
    }

    NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultOutOfMemory, contexts[NN_ARRAY_SIZE(contexts) - 1].GetResult());
}
