﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

TEST(EnsMessagePackWriter, Case1)
{
    nn::ens::detail::util::MessagePackWriter writer;

    nn::Bit8 buffer[4096] = {};

    writer.SetBuffer(buffer, sizeof (buffer));

    writer.WriteMap(2);
    {
        writer.WriteKey("compact");
        writer.WriteValue(true);

        writer.WriteKey("schema");
        writer.WriteValue(0);
    }

    nn::Bit8 data[] =
    {
        0x82,
        0xA7, 'c', 'o', 'm', 'p', 'a', 'c', 't', 0xC3,
        0xA6, 's', 'c', 'h', 'e', 'm', 'a', 0x00,
    };

    EXPECT_EQ(writer.GetWrittenSize(), sizeof (data));
    EXPECT_TRUE(std::memcmp(writer.GetBuffer(), data, sizeof (data)) == 0);
}

TEST(EnsMessagePackWriter, Case2)
{
    nn::ens::detail::util::MessagePackWriter writer;

    nn::Bit8 buffer[4096] = {};

    writer.SetBuffer(buffer, sizeof (buffer));

    writer.WriteMap(1);
    {
        writer.WriteKey("k1");
        writer.WriteMap(1);
        {
            writer.WriteKey("k2");
            writer.WriteMap(1);
            {
                writer.WriteKey("k3");
                writer.WriteValue(100);
            }
        }
    }

    nn::ens::detail::util::ResponseStructureReader<1> reader;

    int8_t value = 0;
    reader.Add("$.k1.k2.k3", &value);

    EXPECT_TRUE(reader.Read<nn::ens::detail::util::MessagePackReader>(writer.GetBuffer(), writer.GetWrittenSize()));
    EXPECT_EQ(value, 100);
}

TEST(EnsMessagePackWriter, AllInOne)
{
    nn::ens::detail::util::MessagePackWriter writer;

    static nn::Bit8 s_Buffer[2 * 1024 * 1024] = {};

    writer.SetBuffer(s_Buffer, sizeof (s_Buffer));

    writer.WriteMap(44);
    {
        writer.WriteKeyValue("key_nil01", nullptr);

        writer.WriteKeyValue("key_bol01", false);
        writer.WriteKeyValue("key_bol02", true);

        writer.WriteKeyValue("key_fxi01", 0);
        writer.WriteKeyValue("key_fxi02", 127);
        writer.WriteKeyValue("key_fxi03", -1);
        writer.WriteKeyValue("key_fxi04", -32);

        writer.WriteKeyValue("key_int01", -33);
        writer.WriteKeyValue("key_int02", INT16_MAX);
        writer.WriteKeyValue("key_int03", INT32_MAX);
        writer.WriteKeyValue("key_int04", INT64_MAX);
        writer.WriteKeyValue("key_int05", UINT8_MAX);
        writer.WriteKeyValue("key_int06", UINT16_MAX);
        writer.WriteKeyValue("key_int07", UINT32_MAX);
        writer.WriteKeyValue("key_int08", UINT64_MAX);

        writer.WriteKeyValue("key_flt01", 0.125f);
        writer.WriteKeyValue("key_flt02", 0.125);

        static char s_String[UINT16_MAX + 1] = {};

        for (size_t i = 0; i < sizeof (s_String); i++)
        {
            s_String[i] = 's';
        }

        writer.WriteKeyValue("key_str01", s_String, 0);
        writer.WriteKeyValue("key_str02", s_String, UINT8_MAX);
        writer.WriteKeyValue("key_str03", s_String, UINT16_MAX);
        writer.WriteKeyValue("key_str04", s_String, sizeof (s_String));

        static nn::Bit8 s_Binary[UINT16_MAX + 1] = {};

        for (size_t i = 0; i < sizeof (s_Binary); i++)
        {
            s_Binary[i] = 'b';
        }

        writer.WriteKeyValue("key_bin01", s_Binary, 0);
        writer.WriteKeyValue("key_bin02", s_Binary, UINT8_MAX);
        writer.WriteKeyValue("key_bin03", s_Binary, UINT16_MAX);
        writer.WriteKeyValue("key_bin04", s_Binary, sizeof (s_Binary));

        static nn::Bit8 s_Extention[UINT16_MAX + 1] = {};

        for (size_t i = 0; i < sizeof (s_Extention); i++)
        {
            s_Extention[i] = 'e';
        }

        writer.WriteKeyValue("key_ext01", static_cast<int8_t>(127), s_Extention, 0);
        writer.WriteKeyValue("key_ext02", static_cast<int8_t>(127), s_Extention, UINT8_MAX);
        writer.WriteKeyValue("key_ext03", static_cast<int8_t>(127), s_Extention, UINT16_MAX);
        writer.WriteKeyValue("key_ext04", static_cast<int8_t>(127), s_Extention, sizeof (s_Extention));
        writer.WriteKeyValue("key_ext05", static_cast<int8_t>(1), s_Extention, 1);
        writer.WriteKeyValue("key_ext06", static_cast<int8_t>(1), s_Extention, 2);
        writer.WriteKeyValue("key_ext07", static_cast<int8_t>(1), s_Extention, 4);
        writer.WriteKeyValue("key_ext08", static_cast<int8_t>(1), s_Extention, 8);
        writer.WriteKeyValue("key_ext09", static_cast<int8_t>(1), s_Extention, 16);
        writer.WriteKeyValue("key_ext10", static_cast<int8_t>(-1), s_Extention, 0);
        writer.WriteKeyValue("key_ext11", static_cast<int8_t>(-128), s_Extention, 0);

        writer.WriteKey("key_map01");
        writer.WriteMap(0);

        writer.WriteKey("key_map02");
        writer.WriteMap(16);
        {
            for (int i = 0; i < 16; i++)
            {
                char key[32] = {};
                nn::util::SNPrintf(key, sizeof (key), "subkey_%02d", i + 1);

                writer.WriteKeyValue(key, 0);
            }
        }

        writer.WriteKey("key_map03");
        writer.WriteMap(UINT8_MAX + 1);
        {
            for (int i = 0; i < UINT8_MAX + 1; i++)
            {
                char key[32] = {};
                nn::util::SNPrintf(key, sizeof (key), "subkey_%03d", i + 1);

                writer.WriteKeyValue(key, 0);
            }
        }

        writer.WriteKey("key_map04");
        writer.WriteMap(UINT16_MAX + 1);
        {
            for (int i = 0; i < UINT16_MAX + 1; i++)
            {
                char key[32] = {};
                nn::util::SNPrintf(key, sizeof (key), "subkey_%05d", i + 1);

                writer.WriteKeyValue(key, 0);
            }
        }

        writer.WriteKey("key_arr01");
        writer.WriteArray(0);

        writer.WriteKey("key_arr02");
        writer.WriteArray(16);
        {
            for (int i = 0; i < 16; i++)
            {
                writer.WriteValue(i);
            }
        }

        writer.WriteKey("key_arr03");
        writer.WriteArray(UINT8_MAX + 1);
        {
            for (int i = 0; i < UINT8_MAX + 1; i++)
            {
                writer.WriteValue(i);
            }
        }

        writer.WriteKey("key_arr04");
        writer.WriteArray(UINT16_MAX + 1);
        {
            for (int i = 0; i < UINT16_MAX + 1; i++)
            {
                writer.WriteValue(i);
            }
        }
    }

    NN_LOG("WrittenSize = %zu\n", writer.GetWrittenSize());

    nn::ens::detail::util::MessagePackReader reader;

    reader.SetCallback
    (
        [](const nn::ens::detail::util::JsonPath& jsonPath,
            const nn::ens::detail::util::DataHolder& holder, void*) -> bool
        {
            if (jsonPath.Compare("$.key_nil01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Nil);
            }
            if (jsonPath.Compare("$.key_bol01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == false);
            }
            if (jsonPath.Compare("$.key_bol02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == true);
            }
            if (jsonPath.Compare("$.key_fxi01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == 0);
            }
            if (jsonPath.Compare("$.key_fxi02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == 127);
            }
            if (jsonPath.Compare("$.key_fxi03"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == -1);
            }
            if (jsonPath.Compare("$.key_fxi04"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == -32);
            }
            if (jsonPath.Compare("$.key_int01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == -33);
            }
            if (jsonPath.Compare("$.key_int02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == INT16_MAX);
            }
            if (jsonPath.Compare("$.key_int03"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == INT32_MAX);
            }
            if (jsonPath.Compare("$.key_int04"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == INT64_MAX);
            }
            if (jsonPath.Compare("$.key_int05"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == UINT8_MAX);
            }
            if (jsonPath.Compare("$.key_int06"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == UINT16_MAX);
            }
            if (jsonPath.Compare("$.key_int07"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == UINT32_MAX);
            }
            if (jsonPath.Compare("$.key_int08"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    !holder.integer.isSigned && holder.integer.u64 == UINT64_MAX);
            }
            if (jsonPath.Compare("$.key_flt01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Float &&
                    holder.float64 == 0.125);
            }
            if (jsonPath.Compare("$.key_flt02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Float &&
                    holder.float64 == 0.125);
            }
            if (jsonPath.Match("$.key_str*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_String)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_str01") &&
                    holder.string.length != 0)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_str02") &&
                    holder.string.length != UINT8_MAX)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_str03") &&
                    holder.string.length != UINT16_MAX)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_str04") &&
                    holder.string.length != UINT16_MAX + 1)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.string.length; i++)
                {
                    if (holder.string.pValue[i] != 's')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Match("$.key_bin*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_Binary)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_bin01") &&
                    holder.binary.length != 0)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_bin02") &&
                    holder.binary.length != UINT8_MAX)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_bin03") &&
                    holder.binary.length != UINT16_MAX)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_bin04") &&
                    holder.binary.length != UINT16_MAX + 1)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.binary.length; i++)
                {
                    const char* pValue = reinterpret_cast<const char*>(holder.binary.pValue);

                    if (pValue[i] != 'b')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Match("$.key_ext*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_Extension)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext01") &&
                    (holder.extension.type != 127 || holder.extension.length != 0))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext02") &&
                    (holder.extension.type != 127 || holder.extension.length != UINT8_MAX))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext03") &&
                    (holder.extension.type != 127 || holder.extension.length != UINT16_MAX))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext04") &&
                    (holder.extension.type != 127 || holder.extension.length != UINT16_MAX + 1))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext05") &&
                    (holder.extension.type != 1 || holder.extension.length != 1))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext06") &&
                    (holder.extension.type != 1 || holder.extension.length != 2))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext07") &&
                    (holder.extension.type != 1 || holder.extension.length != 4))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext08") &&
                    (holder.extension.type != 1 || holder.extension.length != 8))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext09") &&
                    (holder.extension.type != 1 || holder.extension.length != 16))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext10") &&
                    (holder.extension.type != -1 || holder.extension.length != 0))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ext11") &&
                    (holder.extension.type != -128 || holder.extension.length != 0))
                {
                    return false;
                }

                for (size_t i = 0; i < holder.extension.length; i++)
                {
                    const char* pValue = reinterpret_cast<const char*>(holder.extension.pValue);

                    if (pValue[i] != 'e')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Match("$.key_map*"))
            {
                if (!(holder.type == nn::ens::detail::util::DataHolderType_BeginMap ||
                    holder.type == nn::ens::detail::util::DataHolderType_EndMap))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_map01") && holder.integer.u64 != 0)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_map02") && holder.integer.u64 != 16)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_map03") && holder.integer.u64 != UINT8_MAX + 1)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_map04") && holder.integer.u64 != UINT16_MAX + 1)
                {
                    return false;
                }

                return true;
            }
            if (jsonPath.Match("$.key_map*.subkey_*"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == 0);
            }
            if (jsonPath.Match("$.key_arr*[]"))
            {
                if (!(holder.type == nn::ens::detail::util::DataHolderType_BeginArray ||
                    holder.type == nn::ens::detail::util::DataHolderType_EndArray))
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_arr01[]") && holder.integer.u64 != 0)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_arr02[]") && holder.integer.u64 != 16)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_arr03[]") && holder.integer.u64 != UINT8_MAX + 1)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_arr04[]") && holder.integer.u64 != UINT16_MAX + 1)
                {
                    return false;
                }

                return true;
            }
            // 一つのキー内に複数のワイルドカードは表現できない。
            if (jsonPath.Match("$.key_arr01[*]") || jsonPath.Match("$.key_arr02[*]") ||
                jsonPath.Match("$.key_arr03[*]") || jsonPath.Match("$.key_arr04[*]"))
            {
                int keyIndex = 0;
                int arrayIndex = 0;

                std::sscanf(jsonPath.GetString(), "$.key_arr%d[%d]", &keyIndex, &arrayIndex);

                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.isSigned && holder.integer.s64 == arrayIndex);
            }

            if (jsonPath.Compare("$"))
            {
                return true;
            }

            return false;
        },
        nullptr);

    EXPECT_TRUE(reader.Read(writer.GetBuffer(), writer.GetWrittenSize()));
} // NOLINT(impl/function_size)
