﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

#include "testEns_MessagePackFileInputStream.h"

TEST(EnsMessagePackReader, Case1)
{
    nn::ens::detail::util::MessagePackReader reader;

    nn::Bit8 data[] =
    {
        0x82,
        0xA7, 'c', 'o', 'm', 'p', 'a', 'c', 't', 0xC3,
        0xA6, 's', 'c', 'h', 'e', 'm', 'a', 0x00,
    };

    for (size_t i = 0; i < sizeof (data); i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }

    EXPECT_TRUE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, Case2)
{
    nn::ens::detail::util::ResponseStructureReader<1> reader;

    nn::Bit8 data[] =
    {
        0x81,
        0xA2, 'k', '1',
        0x81,
        0xA2, 'k', '2',
        0x81,
        0xA2, 'k', '3',
        0x81,
        0xA2, 'k', '4',
        0x81,
        0xA2, 'k', '5',
        0x81,
        0xA2, 'k', '6',
        0x81,
        0xA2, 'k', '7',
        0x81,
        0xA2, 'k', '8',
        0x81,
        0xA2, 'k', '9',
        0x7F,
    };

    int8_t value = 0;
    reader.Add("$.k1.k2.k3.k4.k5.k6.k7.k8.k9", &value);

    EXPECT_TRUE(reader.Read<nn::ens::detail::util::MessagePackReader>(data, sizeof (data)));
    EXPECT_EQ(value, 0x7F);
}

TEST(EnsMessagePackReader, MultipleRoot)
{
    nn::ens::detail::util::MessagePackReader reader;

    nn::Bit8 data[] =
    {
        0x82,
        0xA7, 'c', 'o', 'm', 'p', 'a', 'c', 't', 0xC3,
        0xA6, 's', 'c', 'h', 'e', 'm', 'a', 0x00,
        0x82,
        0xA7, 'c', 'o', 'm', 'p', 'a', 'c', 't', 0xC3,
        0xA6, 's', 'c', 'h', 'e', 'm', 'a', 0x00,
    };

    for (size_t i = 0; i < sizeof (data) / 2; i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }

    EXPECT_TRUE(reader.Read(data, sizeof (data) / 2));

    for (size_t i = sizeof (data) / 2 + 1; i <= sizeof (data); i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }
}

TEST(EnsMessagePackReader, LongPath1)
{
    nn::ens::detail::util::MessagePackReader reader;

    // PathSize - null 終端 - "$."
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2;

    nn::Bit8 data[3 + KeyLength + 1] =
    {
        0x81,
        0xD9, static_cast<nn::Bit8>(KeyLength),
    };

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[3 + i] = 'x';
    }

    data[3 + KeyLength] = 0x7F;

    EXPECT_TRUE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, LongPath2)
{
    nn::ens::detail::util::MessagePackReader reader;

    // PathSize - null 終端 - "$." + 1
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 + 1;

    nn::Bit8 data[3 + KeyLength + 1] =
    {
        0x81,
        0xD9, static_cast<nn::Bit8>(KeyLength),
    };

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[3 + i] = 'x';
    }

    data[3 + KeyLength] = 0x7F;

    EXPECT_FALSE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, LongPath3)
{
    nn::ens::detail::util::MessagePackReader reader;

    // PathSize - null 終端 - "$." - ".s"
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 - 2;

    nn::Bit8 data[3 + KeyLength + 4] =
    {
        0x81,
        0xD9, static_cast<nn::Bit8>(KeyLength),
    };

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[3 + i] = 'x';
    }

    // $.xxxxxx.s
    data[3 + KeyLength + 0] = 0x81;
    data[3 + KeyLength + 1] = 0xA1;
    data[3 + KeyLength + 2] = 's';
    data[3 + KeyLength + 3] = 0x7F;

    EXPECT_TRUE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, LongPath4)
{
    nn::ens::detail::util::MessagePackReader reader;

    // PathSize - null 終端 - "$." - ".s" + 1
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 - 2 + 1;

    nn::Bit8 data[3 + KeyLength + 4] =
    {
        0x81,
        0xD9, static_cast<nn::Bit8>(KeyLength),
    };

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[3 + i] = 'x';
    }

    // $.xxxxxx.s
    data[3 + KeyLength + 0] = 0x81;
    data[3 + KeyLength + 1] = 0xA1;
    data[3 + KeyLength + 2] = 's';
    data[3 + KeyLength + 3] = 0x7F;

    EXPECT_FALSE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, Depth1)
{
    nn::ens::detail::util::MessagePackReader reader;

    // MAP とキーが出現する度に、Node が深くなる。 (2 × 15 < NodeDepthMax)
    nn::Bit8 data[] =
    {
        0x81,
        0xA3, 'k', '0', '1',
        0x81,
        0xA3, 'k', '0', '2',
        0x81,
        0xA3, 'k', '0', '3',
        0x81,
        0xA3, 'k', '0', '4',
        0x81,
        0xA3, 'k', '0', '5',
        0x81,
        0xA3, 'k', '0', '6',
        0x81,
        0xA3, 'k', '0', '7',
        0x81,
        0xA3, 'k', '0', '8',
        0x81,
        0xA3, 'k', '0', '9',
        0x81,
        0xA3, 'k', '1', '0',
        0x81,
        0xA3, 'k', '1', '1',
        0x81,
        0xA3, 'k', '1', '2',
        0x81,
        0xA3, 'k', '1', '3',
        0x81,
        0xA3, 'k', '1', '4',
        0x81,
        0xA3, 'k', '1', '5',
        0x7F,
    };

    EXPECT_TRUE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, Depth2)
{
    nn::ens::detail::util::MessagePackReader reader;

    // MAP とキーが出現する度に、Node が深くなる。 (2 × 16 > NodeDepthMax)
    nn::Bit8 data[] =
    {
        0x81,
        0xA3, 'k', '0', '1',
        0x81,
        0xA3, 'k', '0', '2',
        0x81,
        0xA3, 'k', '0', '3',
        0x81,
        0xA3, 'k', '0', '4',
        0x81,
        0xA3, 'k', '0', '5',
        0x81,
        0xA3, 'k', '0', '6',
        0x81,
        0xA3, 'k', '0', '7',
        0x81,
        0xA3, 'k', '0', '8',
        0x81,
        0xA3, 'k', '0', '9',
        0x81,
        0xA3, 'k', '1', '0',
        0x81,
        0xA3, 'k', '1', '1',
        0x81,
        0xA3, 'k', '1', '2',
        0x81,
        0xA3, 'k', '1', '3',
        0x81,
        0xA3, 'k', '1', '4',
        0x81,
        0xA3, 'k', '1', '5',
        0x81,
        0xA3, 'k', '1', '6',
        0x7F,
    };

    EXPECT_FALSE(reader.Read(data, sizeof (data)));
}

TEST(EnsMessagePackReader, AllInOne)
{
    nn::ens::detail::util::MessagePackReader reader;

    nn::Bit8 data[] =
    {
        0xDE, 0x00, 0x2E,
        // nil
        0xA9, 'k', 'e', 'y', '_', 'n', 'i', 'l', '0', '1', 0xC0,
        // false
        0xA9, 'k', 'e', 'y', '_', 'b', 'o', 'l', '0', '1', 0xC2,
        // true
        0xA9, 'k', 'e', 'y', '_', 'b', 'o', 'l', '0', '2', 0xC3,
        // positive fixint(1)
        0xA9, 'k', 'e', 'y', '_', 'f', 'x', 'i', '0', '1', 0x00,
        // positive fixint(2)
        0xA9, 'k', 'e', 'y', '_', 'f', 'x', 'i', '0', '2', 0x7F,
        // negative fixint(1)
        0xA9, 'k', 'e', 'y', '_', 'f', 'x', 'i', '0', '3', 0xFF,
        // negative fixint(2)
        0xA9, 'k', 'e', 'y', '_', 'f', 'x', 'i', '0', '4', 0xE0,
        // int 8
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '1', 0xD0, 0x7F,
        // int 16
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '2', 0xD1, 0x00, 0x7F,
        // int 32
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '3', 0xD2, 0x00, 0x00, 0x00, 0x7F,
        // int 64
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '4', 0xD3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F,
        // uint 8
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '5', 0xCC, 0x7F,
        // uint 16
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '6', 0xCD, 0x00, 0x7F,
        // uint 32
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '7', 0xCE, 0x00, 0x00, 0x00, 0x7F,
        // uint 64
        0xA9, 'k', 'e', 'y', '_', 'i', 'n', 't', '0', '8', 0xCF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F,
        // fixstr(1)
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '1', 0xA0,
        // fixstr(2)
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '2', 0xA1, 's',
        // str 8(1)
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '3', 0xD9, 0x00,
        // str 8(2)
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '4', 0xD9, 0x01, 's',
        // str 16
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '5', 0xDA, 0x00, 0x01, 's',
        // str 32
        0xA9, 'k', 'e', 'y', '_', 's', 't', 'r', '0', '6', 0xDB, 0x00, 0x00, 0x00, 0x01, 's',
        // float 32
        0xA9, 'k', 'e', 'y', '_', 'f', 'l', 't', '0', '1', 0xCA, 0x3E, 0x00, 0x00, 0x00,
        // float 64
        0xA9, 'k', 'e', 'y', '_', 'f', 'l', 't', '0', '2', 0xCB, 0x3F, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        // bin 8(1)
        0xA9, 'k', 'e', 'y', '_', 'b', 'i', 'n', '0', '1', 0xC4, 0x00,
        // bin 8(2)
        0xA9, 'k', 'e', 'y', '_', 'b', 'i', 'n', '0', '2', 0xC4, 0x01, 'b',
        // bin 16
        0xA9, 'k', 'e', 'y', '_', 'b', 'i', 'n', '0', '3', 0xC5, 0x00, 0x01, 'b',
        // bin 32
        0xA9, 'k', 'e', 'y', '_', 'b', 'i', 'n', '0', '4', 0xC6, 0x00, 0x00, 0x00, 0x01, 'b',
        // fixext 1
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '1', 0xD4, 0x7F, 'e',
        // fixext 2
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '2', 0xD5, 0x7F, 'e', 'e',
        // fixext 4
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '3', 0xD6, 0x7F, 'e', 'e', 'e', 'e',
        // fixext 8
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '4', 0xD7, 0x7F, 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e',
        // fixext 16
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '5', 0xD8, 0x7F, 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e',
        // ext 8(1)
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '6', 0xC7, 0x00, 0x7F,
        // ext 8(2)
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '7', 0xC7, 0x01, 0x7F, 'e',
        // ext 16
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '8', 0xC8, 0x00, 0x01, 0x7F, 'e',
        // ext 32
        0xA9, 'k', 'e', 'y', '_', 'e', 'x', 't', '0', '9', 0xC9, 0x00, 0x00, 0x00, 0x01, 0x7F, 'e',
        // fixmap(1)
        0xA9, 'k', 'e', 'y', '_', 'm', 'a', 'p', '0', '1', 0x80,
        // fixmap(2)
        0xA9, 'k', 'e', 'y', '_', 'm', 'a', 'p', '0', '2', 0x81, 0xA6, 's', 'u', 'b', 'k', 'e', 'y', 0x00,
        // map 16(1)
        0xA9, 'k', 'e', 'y', '_', 'm', 'a', 'p', '0', '3', 0xDE, 0x00, 0x00,
        // map 16(2)
        0xA9, 'k', 'e', 'y', '_', 'm', 'a', 'p', '0', '4', 0xDE, 0x00, 0x01, 0xA6, 's', 'u', 'b', 'k', 'e', 'y', 0x00,
        // map 32
        0xA9, 'k', 'e', 'y', '_', 'm', 'a', 'p', '0', '5', 0xDF, 0x00, 0x00, 0x00, 0x01, 0xA6, 's', 'u', 'b', 'k', 'e', 'y', 0x00,
        // fixarray(1)
        0xA9, 'k', 'e', 'y', '_', 'a', 'r', 'y', '0', '1', 0xA0,
        // fixarray(2)
        0xA9, 'k', 'e', 'y', '_', 'a', 'r', 'y', '0', '2', 0xA1, 0x00,
        // array 16(1)
        0xA9, 'k', 'e', 'y', '_', 'a', 'r', 'y', '0', '3', 0xDC, 0x00, 0x00,
        // array 16(2)
        0xA9, 'k', 'e', 'y', '_', 'a', 'r', 'y', '0', '4', 0xDC, 0x00, 0x01, 0x00,
        // array 32
        0xA9, 'k', 'e', 'y', '_', 'a', 'r', 'y', '0', '5', 0xDD, 0x00, 0x00, 0x00, 0x01, 0x00,
    };

    reader.SetCallback
    (
        [](const nn::ens::detail::util::JsonPath& jsonPath,
            const nn::ens::detail::util::DataHolder& holder, void*) -> bool
        {
            if (jsonPath.Compare("$.key_nil01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Nil);
            }
            if (jsonPath.Compare("$.key_bol01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == false);
            }
            if (jsonPath.Compare("$.key_bol02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == true);
            }
            if (jsonPath.Compare("$.key_fxi01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == 0);
            }
            if (jsonPath.Compare("$.key_fxi02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == 127);
            }
            if (jsonPath.Compare("$.key_fxi03"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == -1);
            }
            if (jsonPath.Compare("$.key_fxi04"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == -32);
            }
            if (jsonPath.Match("$.key_int*"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == 127);
            }
            if (jsonPath.Match("$.key_flt*"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Float &&
                    holder.float64 == 0.125);
            }
            if (jsonPath.Match("$.key_str*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_String)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.string.length; i++)
                {
                    if (holder.string.pValue[i] != 's')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Match("$.key_bin*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_Binary)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.binary.length; i++)
                {
                    const char* pValue = reinterpret_cast<const char*>(holder.binary.pValue);

                    if (pValue[i] != 'b')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Match("$.key_ext*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_Extension)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.extension.length; i++)
                {
                    const char* pValue = reinterpret_cast<const char*>(holder.extension.pValue);

                    if (pValue[i] != 'e')
                    {
                        return false;
                    }
                }

                return (holder.extension.type == 127);
            }
            if (jsonPath.Compare("$.key_map04"))
            {
                return (holder.integer.u64 == 1);
            }
            if (jsonPath.Compare("$.key_ary04"))
            {
                return (holder.integer.u64 == 1);
            }

            return true;
        },
        nullptr
    );

    EXPECT_TRUE(reader.Read(data, sizeof (data)));
} // NOLINT(impl/function_size)

TEST(EnsMessagePackReader, FileInputStream1)
{
    static nn::Bit8 s_FileSystemCacheBuffer[4 * 1024];

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", s_FileSystemCacheBuffer, sizeof (s_FileSystemCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    const char* FilePath = "rom:/sample.msgpack";

    int64_t fileSize = 0;

    {
        nn::fs::FileHandle handle = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
    }

    // "key_bin02" のデータサイズが 1000 B で一番大きい。
    static nn::Bit8 s_DataBuffer[1000];

    // バッファ不足
    {
        nnt::ens::MessagePackFileInputStream stream;

        NNT_ASSERT_RESULT_SUCCESS(stream.Open(FilePath));

        NN_UTIL_SCOPE_EXIT
        {
            stream.Close();
        };

        nn::ens::detail::util::MessagePackReader reader;

        EXPECT_FALSE(reader.Read(&stream, s_DataBuffer, sizeof (s_DataBuffer) - 1));
    }

    // 読み込みキャッシュなし
    {
        nnt::ens::MessagePackFileInputStream stream;

        NNT_ASSERT_RESULT_SUCCESS(stream.Open(FilePath));

        NN_UTIL_SCOPE_EXIT
        {
            stream.Close();
        };

        nn::ens::detail::util::MessagePackReader reader;

        nn::os::Tick beginTick = nn::os::GetSystemTick();

        EXPECT_TRUE(reader.Read(&stream, s_DataBuffer, sizeof (s_DataBuffer)));

        NN_LOG("FileInputStream(ReadCache = null) -> %lld us\n",
            (nn::os::GetSystemTick() - beginTick).ToTimeSpan().GetMicroSeconds());

        NNT_EXPECT_RESULT_SUCCESS(stream.GetLastReadError());
    }

    static nn::Bit8 s_ReadCache[4096];

    const size_t ReadCacheSizePattern[] =
    {
        1, 2, 4, 8, 16, 32, 128, 512, 999, 1000, 1001, 2048, sizeof (s_ReadCache)
    };

    // 読み込みキャッシュあり
    for (int i = 0; i < NN_ARRAY_SIZE(ReadCacheSizePattern); i++)
    {
        nnt::ens::MessagePackFileInputStream stream(s_ReadCache, ReadCacheSizePattern[i]);

        NNT_ASSERT_RESULT_SUCCESS(stream.Open(FilePath));

        NN_UTIL_SCOPE_EXIT
        {
            stream.Close();
        };

        nn::ens::detail::util::MessagePackReader reader;

        nn::os::Tick beginTick = nn::os::GetSystemTick();

        if (ReadCacheSizePattern[i] < static_cast<size_t>(fileSize))
        {
            EXPECT_TRUE(reader.Read(&stream, s_DataBuffer, sizeof (s_DataBuffer)));
        }
        else
        {
            // 読み込みキャッシュがファイルサイズより大きい場合、ワークバッファは不要なはず。
            EXPECT_TRUE(reader.Read(&stream, nullptr, 0));
        }

        NN_LOG("FileInputStream(ReadCache = %4d) -> %lld us\n",
            ReadCacheSizePattern[i], (nn::os::GetSystemTick() - beginTick).ToTimeSpan().GetMicroSeconds());

        NNT_EXPECT_RESULT_SUCCESS(stream.GetLastReadError());
    }
} // NOLINT(impl/function_size)

TEST(EnsMessagePackReader, FileInputStream2)
{
    static nn::Bit8 s_FileSystemCacheBuffer[4 * 1024];

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", s_FileSystemCacheBuffer, sizeof (s_FileSystemCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    const char* FilePaths[] =
    {
        // 壊れた MessagePack ファイル（連想配列の要素数が少ない）
        "rom:/sample_broken1.msgpack",
        // 壊れた MessagePack ファイル（連想配列の要素数が多い）
        "rom:/sample_broken2.msgpack",
    };

    static nn::Bit8 s_DataBuffer[1000];

    for (int f = 0; f < NN_ARRAY_SIZE(FilePaths); f++)
    {
        // 読み込みキャッシュなし
        {
            nnt::ens::MessagePackFileInputStream stream;

            NNT_ASSERT_RESULT_SUCCESS(stream.Open(FilePaths[f]));

            NN_UTIL_SCOPE_EXIT
            {
                stream.Close();
            };

            nn::ens::detail::util::MessagePackReader reader;

            EXPECT_FALSE(reader.Read(&stream, s_DataBuffer, sizeof (s_DataBuffer)));
        }

        static nn::Bit8 s_ReadCache[4096];

        const size_t ReadCacheSizePattern[] =
        {
            1, 2, 4, 8, 16, 32, 128, 512, 999, 1000, 1001, 2048, sizeof (s_ReadCache)
        };

        // 読み込みキャッシュあり
        for (int i = 0; i < NN_ARRAY_SIZE(ReadCacheSizePattern); i++)
        {
            nnt::ens::MessagePackFileInputStream stream(s_ReadCache, ReadCacheSizePattern[i]);

            NNT_ASSERT_RESULT_SUCCESS(stream.Open(FilePaths[f]));

            NN_UTIL_SCOPE_EXIT
            {
                stream.Close();
            };

            nn::ens::detail::util::MessagePackReader reader;

            EXPECT_FALSE(reader.Read(&stream, s_DataBuffer, sizeof (s_DataBuffer)));
        }
    }
}
