﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../Common/testEns_Common.h"

#include <nn/ens/ens_SimpleMessagePackReader.h>
#include <nn/ens/ens_SimpleMessagePackWriter.h>

TEST(EnsSimpleMessagePackWriter, Case1)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    writer.WriteMap(2);
    {
        writer.WriteKeyValue("compact", true);
        writer.WriteKeyValue("schema", 0);
    }

    EXPECT_TRUE(writer.IsValid());

    nn::Bit8 data[] =
    {
        0x82,
        0xA7, 'c', 'o', 'm', 'p', 'a', 'c', 't', 0xC3,
        0xA6, 's', 'c', 'h', 'e', 'm', 'a', 0x00,
    };

    EXPECT_EQ(writer.GetWrittenSize(), sizeof (data));
    EXPECT_TRUE(std::memcmp(buffer, data, sizeof (data)) == 0);
}

TEST(EnsSimpleMessagePackWriter, Case2)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    writer.WriteMap(1);
    {
        writer.WriteKeyMap("k1", 1);
        {
            writer.WriteKeyMap("k2", 1);
            {
                writer.WriteKeyValue("k3", 100);
            }
        }
    }

    EXPECT_TRUE(writer.IsValid());

    nn::ens::SimpleMessagePackReader reader(buffer, writer.GetWrittenSize());

    int8_t value = 0;
    EXPECT_TRUE(reader.Get(&value, "$.k1.k2.k3"));
    EXPECT_EQ(value, 100);
}

TEST(EnsSimpleMessagePackWriter, Case3)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    writer.WriteValue(100);

    EXPECT_TRUE(writer.IsValid());

    nn::ens::SimpleMessagePackReader reader(buffer, writer.GetWrittenSize());

    int value;
    EXPECT_TRUE(reader.Get(&value, "$"));
    EXPECT_EQ(value, 100);
}

TEST(EnsSimpleMessagePackWriter, Case4)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    writer.WriteArray(3);
    {
        writer.WriteValue(1);
        writer.WriteValue(2);
        writer.WriteValue(3);
    }

    EXPECT_TRUE(writer.IsValid());

    nn::ens::SimpleMessagePackReader reader(buffer, writer.GetWrittenSize());

    int values[3];
    EXPECT_TRUE(reader.Get(&values[0], "$[0]"));
    EXPECT_TRUE(reader.Get(&values[1], "$[1]"));
    EXPECT_TRUE(reader.Get(&values[2], "$[2]"));
    EXPECT_EQ(values[0], 1);
    EXPECT_EQ(values[1], 2);
    EXPECT_EQ(values[2], 3);
}

TEST(EnsSimpleMessagePackWriter, Case5)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    struct
    {
        int key1;
        struct
        {
            int a;
            int b;
        }
        key2;
        int key3[3];
    }
    s1 = {100, {100, 100}, {1, 2, 3}}, s2 = {};

    writer.WriteMap(3);
    EXPECT_FALSE(writer.IsValid());
    {
        writer.WriteKeyValue("key1", s1.key1);
        EXPECT_FALSE(writer.IsValid());
        writer.WriteKeyMap("key2", 2);
        EXPECT_FALSE(writer.IsValid());
        {
            writer.WriteKeyValue("a", s1.key2.a);
            EXPECT_FALSE(writer.IsValid());
            writer.WriteKeyValue("b", s1.key2.b);
            EXPECT_FALSE(writer.IsValid());
        }
        writer.WriteKeyArray("key3", 3);
        EXPECT_FALSE(writer.IsValid());
        {
            writer.WriteValue(s1.key3[0]);
            EXPECT_FALSE(writer.IsValid());
            writer.WriteValue(s1.key3[1]);
            EXPECT_FALSE(writer.IsValid());
            writer.WriteValue(s1.key3[2]);
            EXPECT_TRUE(writer.IsValid());
        }
    }

    nn::ens::SimpleMessagePackReader reader(buffer, writer.GetWrittenSize());

    EXPECT_TRUE(reader.Get(&s2.key1,    "$.key1"));
    EXPECT_TRUE(reader.Get(&s2.key2.a,  "$.key2.a"));
    EXPECT_TRUE(reader.Get(&s2.key2.b,  "$.key2.b"));
    EXPECT_TRUE(reader.Get(&s2.key3[0], "$.key3[0]"));
    EXPECT_TRUE(reader.Get(&s2.key3[1], "$.key3[1]"));
    EXPECT_TRUE(reader.Get(&s2.key3[2], "$.key3[2]"));

    EXPECT_TRUE(std::memcmp(&s1, &s2, sizeof (s1)) == 0);
}

TEST(EnsSimpleMessagePackWriter, Case6)
{
    nn::Bit8 buffer[4096] = {};

    nn::ens::SimpleMessagePackWriter writer(buffer, sizeof (buffer));

    struct
    {
        int key1;
        struct
        {
            int a;
            int b;
        }
        key2[2];
    }
    s1 = {100, {{1, 2}, {3, 4}}}, s2 = {};

    writer.WriteMap(2);
    EXPECT_FALSE(writer.IsValid());
    {
        writer.WriteKeyValue("key1", s1.key1);
        EXPECT_FALSE(writer.IsValid());
        writer.WriteKeyArray("key2", 2);
        EXPECT_FALSE(writer.IsValid());
        {
            writer.WriteMap(2);
            EXPECT_FALSE(writer.IsValid());
            {
                writer.WriteKeyValue("a", s1.key2[0].a);
                EXPECT_FALSE(writer.IsValid());
                writer.WriteKeyValue("b", s1.key2[0].b);
                EXPECT_FALSE(writer.IsValid());
            }
            writer.WriteMap(2);
            EXPECT_FALSE(writer.IsValid());
            {
                writer.WriteKeyValue("a", s1.key2[1].a);
                EXPECT_FALSE(writer.IsValid());
                writer.WriteKeyValue("b", s1.key2[1].b);
                EXPECT_TRUE(writer.IsValid());
            }
        }
    }

    nn::ens::SimpleMessagePackReader reader(buffer, writer.GetWrittenSize());

    EXPECT_TRUE(reader.Get(&s2.key1,      "$.key1"));
    EXPECT_TRUE(reader.Get(&s2.key2[0].a, "$.key2[0].a"));
    EXPECT_TRUE(reader.Get(&s2.key2[0].b, "$.key2[0].b"));
    EXPECT_TRUE(reader.Get(&s2.key2[1].a, "$.key2[1].a"));
    EXPECT_TRUE(reader.Get(&s2.key2[1].b, "$.key2[1].b"));

    EXPECT_TRUE(std::memcmp(&s1, &s2, sizeof (s1)) == 0);
}
