﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../Common/testEns_Common.h"
#include "../../../Common/testEns_AccountUtility.h"
#include "../../../Common/testEns_NetworkUtility.h"
#include "../AcbaaCommon/testEns_ApiForAcbaa.h"

#include <nn/ens/ens_ApiForAcbaa.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>

namespace
{
    const size_t MetadataBufferSize = 4096;
}

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];

    nn::ens::Credential g_Credentials[2] = {nn::ens::InvalidCredential, nn::ens::InvalidCredential};
    bool g_IsGenerated = false;

    nn::ens::MessageHeader g_Headers[2] = {};
    char g_MetadataBuffers[NN_ARRAY_SIZE(g_Headers)][MetadataBufferSize];
    bool g_IsListed = false;

    char g_MetadataFixtureBuffer[MetadataBufferSize];
    nn::ens::ReceiveBuffer g_MetadataFixture = {g_MetadataFixtureBuffer, sizeof (g_MetadataFixtureBuffer)};
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void TestReceiveMessageHeaderList(int expectedCount, int expectedTotalCount,
        nn::ens::MessageHeader pOutHeaderList[], int count, int offset,
        const nn::ens::Credential& credential, const nnt::ens::NsaIdTokenGetter& token) NN_NOEXCEPT
    {
        nn::ens::AsyncContext context;
        int actualCount;
        int actualTotalCount;

        nn::ens::ReceiveMessageHeaderList(&context, &actualCount, &actualTotalCount,
            pOutHeaderList, count, offset, credential, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_EQ(expectedCount, actualCount);
        EXPECT_EQ(expectedTotalCount, actualTotalCount);
    }

    void DumpMessageHeader(const nn::ens::MessageHeader& header) NN_NOEXCEPT
    {
        auto c = nn::time::ToCalendarTimeInUtc(header.sentAt);

        char digest[nn::ens::Digest::StringLength + 1] = {};

        NN_LOG("    MessageId:    %llu\n", header.id.value);
        NN_LOG("    SenderUserId: %llu\n", header.sender.value);
        NN_LOG("    SentAt:       (UTC) %04d/%02d/%02d %02d:%02d:%02d\n", c.year, c.month, c.day, c.hour, c.minute, c.second);
        NN_LOG("    Digest:       %s\n", header.digest.ToString(digest, sizeof (digest)));
        NN_LOG("    MetaSize:     %zu\n", header.metadata.receivedSize);
    }
}

class EnsMessage : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();

        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            g_Headers[i].metadata.pBuffer = g_MetadataBuffers[i];
            g_Headers[i].metadata.bufferSize = sizeof (g_MetadataBuffers[i]);
        }
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsMessage, GenerateCredential)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    bool isGenerated = true;

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &g_Credentials[i], token.Get());

        context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

        // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
        if (context.GetResult().IsFailure() &&
            !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

#endif

        if (context.GetResult().IsSuccess())
        {
            NN_LOG("UserId[%d] = %llu\n", i, g_Credentials[i].userId.value);
        }
        else
        {
            isGenerated = false;
        }
    }

    g_IsGenerated = isGenerated;
}

TEST_F(EnsMessage, GetMetadataFixture)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::ens::AsyncContext context;

    nnt::ens::GetMessageMetadataFixture(&context, &g_MetadataFixture);

    context.GetEvent().Wait();

    if (context.GetResult().IsFailure())
    {
        g_MetadataFixture.receivedSize = 0;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

    // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
    if (context.GetResult().IsFailure() &&
        !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
    {
        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

#endif
}

TEST_F(EnsMessage, ActivateNotificationService)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::account::Uid user = {};

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, &user, 1));
    ASSERT_GE(count, 1);

#endif

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        nn::ens::AsyncContext context;

        nn::ens::ActivateNotificationService(&context, user, g_Credentials[i], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(EnsMessage, Send)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 送信 1
    {
        nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
        nn::ens::SendBuffer body = {"body 1", 6};

        nn::ens::AsyncContext context;

        nn::ens::SendMessage(&context, g_Credentials[1].userId, metadata, body, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    // 次のメッセージの送信時刻をずらすためにスリープを挟む。
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    // 送信 2
    {
        nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
        nn::ens::SendBuffer body = {"body 2", 6};

        nn::ens::AsyncContext context;

        nn::ens::SendMessage(&context, g_Credentials[1].userId, metadata, body, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(EnsMessage, ReceiveHeaderList)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    {
        const int ExpectedCount = 2;
        const int ExpectedTotalCount = 2;
        const int Offset = 0;

        // g_Credentials[1].userId 側で受信
        ASSERT_NO_FATAL_FAILURE(TestReceiveMessageHeaderList(ExpectedCount, ExpectedTotalCount,
            g_Headers, NN_ARRAY_SIZE(g_Headers), Offset, g_Credentials[1], token));

        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            NN_LOG("MessageHeader[%d]\n", i);
            DumpMessageHeader(g_Headers[i]);
        }

        g_IsListed = true;
    }

    // 送信時刻の昇順に取得される。
    EXPECT_LT(g_Headers[0].sentAt, g_Headers[1].sentAt);

    // 送信 1
    EXPECT_NE(g_Headers[0].sentAt.value, 0);
    EXPECT_EQ(g_Headers[0].sender, g_Credentials[0].userId);
    EXPECT_EQ(g_Headers[0].metadata.receivedSize, g_MetadataFixture.receivedSize);
    EXPECT_EQ(std::memcmp(g_Headers[0].metadata.pBuffer, g_MetadataFixture.pBuffer, g_Headers[0].metadata.receivedSize), 0);

    // 送信 2
    EXPECT_NE(g_Headers[1].sentAt.value, 0);
    EXPECT_EQ(g_Headers[1].sender, g_Credentials[0].userId);
    EXPECT_EQ(g_Headers[1].metadata.receivedSize, g_MetadataFixture.receivedSize);
    EXPECT_EQ(std::memcmp(g_Headers[1].metadata.pBuffer, g_MetadataFixture.pBuffer, g_Headers[1].metadata.receivedSize), 0);

    // ヘッダリスト受信（オフセットあり）
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 2;

        nn::ens::MessageHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestReceiveMessageHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 1, g_Credentials[1], token));

        // g_Headers[1] と同じはず
        EXPECT_EQ(header.id, g_Headers[1].id);
        EXPECT_EQ(header.sender, g_Headers[1].sender);
        EXPECT_EQ(header.sentAt, g_Headers[1].sentAt);
        EXPECT_EQ(header.metadata.receivedSize, g_Headers[1].metadata.receivedSize);
        EXPECT_EQ(std::memcmp(header.metadata.pBuffer, g_Headers[1].metadata.pBuffer, header.metadata.receivedSize), 0);
    }
}

TEST_F(EnsMessage, ReceiveBody)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsListed)
    {
        NN_LOG("Headers were not listed. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
    {
        nn::ens::ReceiveBuffer body = {};
        char buffer[512];

        body.pBuffer = buffer;
        body.bufferSize = sizeof (buffer);

        nn::ens::AsyncContext context;

        nn::ens::ReceiveMessageBody(&context, &body, g_Headers[i].id, g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        if (i == 0)
        {
            EXPECT_EQ(body.receivedSize, 6);
            EXPECT_EQ(std::memcmp(body.pBuffer, "body 1", 6), 0);
        }
        else
        {
            EXPECT_EQ(body.receivedSize, 6);
            EXPECT_EQ(std::memcmp(body.pBuffer, "body 2", 6), 0);
        }
    }

    // 他者宛てのメッセージのボディ取得
    {
        nn::ens::ReceiveBuffer body = {};
        char buffer[512];

        body.pBuffer = buffer;
        body.bufferSize = sizeof (buffer);

        nn::ens::AsyncContext context;

        nn::ens::ReceiveMessageBody(&context, &body, g_Headers[0].id, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNotFound, context.GetResult());
    }
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

TEST_F(EnsMessage, GetNotification)
{
    nn::os::SystemEvent& e = nn::ens::GetNotificationEvent();

    nn::ens::NotificationData data = {};
    int count = 0;

    // g_Credentials[1].userId 宛ての通知が 2 通届いているはず。
    while (count < 2)
    {
        ASSERT_TRUE(e.TimedWait(nn::TimeSpan::FromSeconds(10)));
        e.Clear();

        while (nn::ens::PopNotificationData(&data))
        {
            EXPECT_STREQ(data.type, "message_card");
            EXPECT_EQ(data.receiver, g_Credentials[1].userId);

            nn::ens::detail::util::MessagePackReader reader;

            EXPECT_GT(data.payloadSize, 0);
            EXPECT_TRUE(reader.Read(data.payload, data.payloadSize));

            count++;
        }
    }

    EXPECT_EQ(count, 2);
    EXPECT_FALSE(nn::ens::PopNotificationData(&data));
}

#endif

TEST_F(EnsMessage, Delete)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsListed)
    {
        NN_LOG("Headers were not listed. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 自分が送信したメッセージの削除要求
    {
        nn::ens::MessageId messageIds[NN_ARRAY_SIZE(g_Headers)] = {};

        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            messageIds[i] = g_Headers[i].id;
        }

        nn::ens::AsyncContext context;

        nn::ens::DeleteMessage(&context, messageIds, NN_ARRAY_SIZE(messageIds), "RECEIVED", g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        // 成功を返すが、実際は権限がないため削除されない。
        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 2;

        nn::ens::MessageHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestReceiveMessageHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 0, g_Credentials[1], token));
    }

    // 不明な削除理由
    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMessage(&context, &g_Headers[0].id, 1, "HOGE", g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultInvalidParameter, context.GetResult());
    }

    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMessage(&context, &g_Headers[0].id, 1, "RECEIVED", g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 1;

        nn::ens::MessageHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestReceiveMessageHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 0, g_Credentials[1], token));

        EXPECT_EQ(header.id, g_Headers[1].id);
    }
    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMessage(&context, &g_Headers[1].id, 1, "REJECTED", g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        const int ExpectedCount = 0;
        const int ExpectedTotalCount = 0;

        nn::ens::MessageHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestReceiveMessageHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 0, g_Credentials[1], token));
    }

    // 再削除
    {
        nn::ens::MessageId messageIds[NN_ARRAY_SIZE(g_Headers)] = {};

        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            messageIds[i] = g_Headers[i].id;
        }

        nn::ens::AsyncContext context;

        nn::ens::DeleteMessage(&context, messageIds, NN_ARRAY_SIZE(messageIds), "RECEIVED", g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
} // NOLINT(impl/function_size)

TEST_F(EnsMessage, SendToNintendo)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
    nn::ens::SendBuffer body = {"body", 4};

    nn::ens::AsyncContext context;

    nn::ens::SendMessageToNintendo(&context, metadata, body, g_Credentials[0], token.Get());

    context.GetEvent().Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
}

// TORIAEZU: EnsAuthentication に置きたいが、認証トークンを必要とする関数が libnn_ens にまだないため、ここに置く。
//           nn::ens::Report が実装されたら移動する。
TEST_F(EnsMessage, InvalidAuthToken)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    // "ZHVtbXk=" は "dummy" を Base64 エンコードしたもの。
    nn::ens::detail::AuthToken dummyAuthToken = {"ZHVtbXk="};

    nn::ens::detail::core::SetAuthTokenToCache(g_Credentials[0].userId, dummyAuthToken);

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
    nn::ens::SendBuffer body = {"body", 4};

    nn::ens::AsyncContext context;

    nn::ens::SendMessageToNintendo(&context, metadata, body, g_Credentials[0], token.Get());

    context.GetEvent().Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
}
