﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <getopt.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/vi/vi_ProxyName.private.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#if !defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
#include <unistd.h>
#endif
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
    #include <nv/nv_ServiceName.h>
    #include <nvnTool/nvnTool_GlslcInterface.h>
#endif

#include <Common/ProgramOptionParser.h>
#include <Common/GfxJpegViewer.h>
#include <Common/Ui2dSimple.h>
#include <Common/ConfigSwitcher.h>
#include <Common/FilePathViewer.h>
#include "HdmiConfig.h"

namespace {

const int CharBufferLength = 512;
nn::mem::StandardAllocator g_Allocator;
NN_ALIGNAS(4096) char g_FsBuffer[8 * 1024 * 1024];

const char* PresetImageSource[] =
{
    "ui2dsimple",
    "Contents:/Image.jpg",
    "Contents:/CB720p.bmp",
    "Contents:/RMP1080p.bmp",
    "Contents:/oxtonguerapids_T1080.bmp",
};

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
void ModifyFormattedPath(char* dstPath, const char* srcPath)
{
    std::sprintf(dstPath, "Contents:/%s", srcPath);
}
#else
void ModifyFormattedPath(char* dstPath, const char* srcPath)
{
    std::snprintf(dstPath, CharBufferLength, "Contents:/%s", srcPath);
}
#endif

const char* AdaptPathToFileSystem(char* path)
{
    if (!std::strncmp(path, "", CharBufferLength))
    {
        return nullptr;
    }
    char tmpPath[CharBufferLength];
    std::strncpy(tmpPath, path, CharBufferLength);
    ModifyFormattedPath(path, tmpPath);
    return path;
}

}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(768 * 1024 * 1024);
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MallocMemorySize );
    NN_ASSERT( result.IsSuccess() );
    nn::init::InitializeAllocator( reinterpret_cast<void*>( address ), MallocMemorySize );
}
#endif

extern "C" void nnMain()
{
    NN_LOG("=============================================\n");
    NN_LOG("  - Build date:[%s][%s]\n", __DATE__, __TIME__);
    NN_LOG("=============================================\n");

    nns::ProgramOptionParser parser;
    parser.AddOption('c', "video_config", required_argument,
        "set config mode (0 <= <param> < configs) or get config modes (otherwise config_mode).");
    parser.AddOption('d', "hdcp", required_argument,
        "enable hdcp (if <param> > 0, will enable hdcp. otherwise, will disable hdcp.");
    parser.AddOption('e', "edid_vic", required_argument,
        "set edid VIC (if <param> are {0, 1, 2, 3, 4}, {VIC=3, VIC=4, VIC=16, VIC=95, VIC=97} are set respectively).");
    parser.AddOption('i', "image", required_argument,
        "showed image path (<param> is .jpeg path from executable file or image type name (\"noimage\")).");
    parser.AddOption('o', "host", required_argument, "searched host(PC) path.");
    parser.AddOption('r', "rgb_range", required_argument,
        "set rgb range (if <param> is 0, limited range is selected. if <param> > 1, full range is selected).");
    parser.AddOption('t', "content_type", required_argument,
        "set content type (if <param> is 0, graphics content is selected. if <param> > 4, game content is selected).");
    parser.AddOption('q', "font", required_argument, "show setting fonts when booting. (if is_showing > 0, will show setting fonts).");
    parser.AddOption('u', "wave_front_left", required_argument,
        "set wave file for front left (<param> is .wav path from executable file or image type name (\"sin\")).");
    parser.AddOption('v', "wave_front_right", required_argument,
        "set wave file for front right (<param> is .wav path from executable file or image type name (\"sin\")).");
    parser.AddOption('w', "wave_front_center", required_argument,
        "set wave file for front center (<param> is .wav path from executable file or image type name (\"sin\")).");
    parser.AddOption('x', "wave_low_frequency", required_argument,
        "set wave file for low frequency (<param> is .wav path from executable file or image type name (\"sin\")).");
    parser.AddOption('y', "wave_rear_left", required_argument,
        "set wave file for rear left (<param> is .wav path from executable file or image type name (\"sin\")).");
    parser.AddOption('z', "wave_rear_right", required_argument,
        "set wave file for rear right (<param> is .wav path from executable file or image type name (\"sin\")).");
    if (!parser.ParseOption())
    {
        NN_LOG("=============================================\n");
        parser.ShowHelp();
        NN_ABORT("End of %s\n", __FILE__);
    }

    bool isHdcpEnabled = parser.GetBooleanOptionalValue('d', false);
    bool isFontVisible = parser.GetBooleanOptionalValue('q', true);
    int configMode = parser.GetIntegerOptionalValue('c', -1);
    int edidVic = parser.GetIntegerOptionalValue('e', 0);  // VIC=3 が default値
    int rgbRange = parser.GetIntegerOptionalValue('r', DISPLAY_RGB_Full);
    nn::settings::system::HdmiContentType contentType =
        static_cast<nn::settings::system::HdmiContentType>(parser.GetIntegerOptionalValue('t', nn::settings::system::HdmiContentType_Game));
    std::string targetImageName = parser.GetStringOptionalValue('i', "ui2dsimple");
    std::string hostPath = parser.GetStringOptionalValue('o', "C:/image");
    std::vector<std::string> waveFilePath;
    waveFilePath.push_back(parser.GetStringOptionalValue('u', ""));
    waveFilePath.push_back(parser.GetStringOptionalValue('v', ""));
    waveFilePath.push_back(parser.GetStringOptionalValue('w', ""));
    waveFilePath.push_back(parser.GetStringOptionalValue('x', ""));
    waveFilePath.push_back(parser.GetStringOptionalValue('y', ""));
    waveFilePath.push_back(parser.GetStringOptionalValue('z', ""));
    NN_ASSERT_LESS_EQUAL(waveFilePath.size(), nns::audio::ChannelCountMax);

    NN_LOG("  - image path          :[%s]\n", targetImageName.c_str());
    NN_LOG("  - front left wave     :[%s]\n", waveFilePath[0].c_str());
    NN_LOG("  - front right wave    :[%s]\n", waveFilePath[1].c_str());
    NN_LOG("  - front center wave   :[%s]\n", waveFilePath[2].c_str());
    NN_LOG("  - low frequency wave  :[%s]\n", waveFilePath[3].c_str());
    NN_LOG("  - rear left wave      :[%s]\n", waveFilePath[4].c_str());
    NN_LOG("  - rear right wave     :[%s]\n", waveFilePath[5].c_str());
    NN_LOG("=============================================\n");

    // nn::vi と nvnflinger を共存させるための設定
    nn::vi::SetRelayProxyName("dispsys");

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsServiceName("nvdrv:t");
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
        glslcSetAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    }
#endif

    g_Allocator.Initialize(g_FsBuffer, sizeof(g_FsBuffer));
    nn::fs::SetAllocator(Allocate, Deallocate);

    // host PC と SDカード内にある画像ファイルを検索。
    nns::FilePathViewer pathViewer(hostPath.c_str());
    std::vector<std::string> imagePath;
    pathViewer.SearchImageFileFromAttachedDevice(&imagePath);

    nns::HdmiConfig config(isHdcpEnabled, edidVic, configMode, rgbRange, contentType, waveFilePath);
    config.ApplySelectedLabel();
    nns::ConfigSwitcher switcher(&config, isFontVisible);
    switcher.CreateInputListener();

    // 読めなかったらプリセットの画像を突っ込む。
    if (0 == imagePath.size())
    {
        for (auto preset : PresetImageSource)
        {
            imagePath.push_back(preset);
        }
    }

    // 指定したファイルに最初に出す絵をあわせる。
    auto iter = std::find(imagePath.begin(), imagePath.end(), targetImageName);
    auto targetIndex = std::distance(imagePath.begin(), iter) % imagePath.size();

    // 描画するイメージのパスを表示。
    for (int i1=0; i1<imagePath.size(); ++i1)
    {
        NN_LOG("%s [%2d/%2d]:[%s]\n", (i1 == targetIndex) ? "*" : " ", i1, imagePath.size(), imagePath[i1].c_str());
    }
    NN_LOG("=============================================\n");

    nns::SetupGraphics gfxSetup;

    // 描画。入力に合わせて絵を順次切り替え。
    for (int i1=0; ; ++i1)
    {
        const std::string& image = imagePath[targetIndex];
        if ("ui2dsimple" == image)
        {
            nns::ShowMovingPictures(&gfxSetup, &switcher);
        }
        else
        {
            nns::ShowImage(image.c_str(), &gfxSetup, &switcher);
        }
        targetIndex = (targetIndex + 1) % imagePath.size();
    }
    NN_LOG("End of %s\n", __FILE__);
} // NOLINT(impl/function_size)
