﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HdmiConfig.h"
#include <cstdio>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/cec/cec_Api.h>

namespace nns {

namespace {

const char* DefaultStereoAudioContent[2] =
{
    "Contents:/voice2ch_L.wav",
    "Contents:/voice2ch_R.wav",
};

const char* DefaultSurroundAudioContent[6] =
{
    "Contents:/voice6ch_1.wav",
    "Contents:/voice6ch_2.wav",
    "Contents:/voice6ch_3.wav",
    "Contents:/voice6ch_4.wav",
    "Contents:/voice6ch_5.wav",
    "Contents:/voice6ch_6.wav",
};

#define CASE_INT_RETURN_STRING(type) case type: return #type
const char* GetRgbRangeString(int rgbRange)
{
    switch (rgbRange)
    {
    CASE_INT_RETURN_STRING(DISPLAY_RGB_Limited);
    CASE_INT_RETURN_STRING(DISPLAY_RGB_Full);
    default: return "";
    }
    return "";
}

const char* GetContentTypeString(nn::settings::system::HdmiContentType contentType)
{
    switch (contentType)
    {
    CASE_INT_RETURN_STRING(nn::settings::system::HdmiContentType_None);
    CASE_INT_RETURN_STRING(nn::settings::system::HdmiContentType_Graphics);
    CASE_INT_RETURN_STRING(nn::settings::system::HdmiContentType_Cinema);
    CASE_INT_RETURN_STRING(nn::settings::system::HdmiContentType_Photo);
    CASE_INT_RETURN_STRING(nn::settings::system::HdmiContentType_Game);
    default: return "";
    }
    return "";
}
#undef CASE_INT_RETURN_STRING

const Color DefaultBackGroundColor = Color(0.0f, 0.0f, 0.5f, 1.0f);
const Color ProgressBackGroundColor = Color(0.0f, 0.5f, 0.0f, 1.0f);
const Color FailureBackGroundColor = Color(0.5f, 0.0f, 0.0f, 1.0f);

int GetChannelBasedOnInputWaveFile(const std::vector<std::string>& waveFilePath)
{
    for (int i1=(waveFilePath.size() - 1); i1>=0; --i1)
    {
        if (0 < waveFilePath[i1].size())
        {
            return i1 >= 2 ? 6 : 2;
        }
    }
    return 0;
}

}

NN_IMPLICIT HdmiConfig::HdmiConfig(bool isHdcpEnabled, int edidVicId, int configMode, int rgbRange,
                                   int contentType, const std::vector<std::string>& waveFilePath) NN_NOEXCEPT
:
m_State(HdcpState_Unpluged),
m_IsHdcpSetEnabled(false),
m_IsLabelResetting(true),
m_HdcpPollingCount(0),
m_ConfigMode(configMode),
m_PreVideoConfigs(0),
m_VideoConfigSeries(0),
m_HdcpStartMilliSeconds(0LL),
m_Video(android::ISurfaceComposer::eDisplayIdHdmi)
{
    m_Setting.clear();
    m_Setting.reserve(128);
    std::vector<SettingContext::State> settingState;

    int usedConfigMode = (0 > configMode) ? m_Video.FindDisplayConfigModeFromVic(static_cast<nns::VideoConfigUtility::EdidVic>(edidVicId)) : configMode;
    AssignVideoConfigState(&settingState, true);
    int initialIndex = FindSettingStateIndexFromLevel(settingState, usedConfigMode);
    m_Setting.push_back({&HdmiConfig::SwitchVideoConfig, nullptr, "TV Config", std::move(settingState), initialIndex, false});

    settingState.clear();
    settingState.push_back({ "Product", 0 });
    settingState.push_back({ "Debug", 1 });
    initialIndex = FindSettingStateIndexFromLevel(settingState, 0);
    m_Setting.push_back({ &HdmiConfig::SwitchVideoConfigSeries, nullptr, "Config series", std::move(settingState), initialIndex, false });

    settingState.clear();
    settingState.push_back({"Limited", DISPLAY_RGB_Limited});
    settingState.push_back({"Full", DISPLAY_RGB_Full});
    initialIndex = FindSettingStateIndexFromLevel(settingState, rgbRange);
    m_Setting.push_back({&HdmiConfig::SwitchRgbRange, nullptr, "RGB range", std::move(settingState), initialIndex, false});

    settingState.clear();
    settingState.push_back({"Game", nn::settings::system::HdmiContentType_Game});
    settingState.push_back({"None", nn::settings::system::HdmiContentType_None});
    initialIndex = FindSettingStateIndexFromLevel(settingState, contentType);
    m_Setting.push_back({&HdmiConfig::SwitchContentType, nullptr, "Content type", std::move(settingState), initialIndex, false});

    AssignHdcpState(&settingState, isHdcpEnabled);

    settingState.clear();
    settingState.push_back({"Mute", 0});
    settingState.push_back({"Stereo", 2});
    settingState.push_back({"Surround", 6});

    for (int i1=0; i1<2; ++i1)
    {
        std::strncpy(m_StereoAudioContent[i1], DefaultStereoAudioContent[i1], CharBufferLength);
    }
    for (int i1=0; i1<6; ++i1)
    {
        std::strncpy(m_SurroundAudioContent[i1], DefaultSurroundAudioContent[i1], CharBufferLength);
    }
    int defaultChannels = GetChannelBasedOnInputWaveFile(waveFilePath);
    for (int i1=0; i1<defaultChannels; ++i1)
    {
        if (0 < waveFilePath[i1].size())
        {
            if (6 > defaultChannels)
            {
                std::strncpy(m_StereoAudioContent[i1], "Contents:/", CharBufferLength);
                std::strncat(m_StereoAudioContent[i1], waveFilePath[i1].c_str(), CharBufferLength);
            }
            else
            {
                std::strncpy(m_SurroundAudioContent[i1], "Contents:/", CharBufferLength);
                std::strncat(m_SurroundAudioContent[i1], waveFilePath[i1].c_str(), CharBufferLength);
            }
        }
    }
    initialIndex = FindSettingStateIndexFromLevel(settingState, defaultChannels);
    m_Setting.push_back({&HdmiConfig::SwitchAudio, nullptr, "Audio", std::move(settingState), initialIndex, false});

    if (m_Cec.SetCommandListener())
    {
        settingState.clear();
        settingState.push_back({"----", 0});
        settingState.push_back({"Standby", 1});
        settingState.push_back({"OneTouch", 2});
        m_Setting.push_back({&HdmiConfig::SendCecCommand, &HdmiConfig::GetCecStatus, "CEC", std::move(settingState), 0, false});

        settingState.clear();
        settingState.push_back({ "----", 0 });
        settingState.push_back({ "Up", 1 });
        settingState.push_back({ "Mute", 2 });
        settingState.push_back({ "Down", 3 });
        m_Setting.push_back({ &HdmiConfig::SendCecVolumeCommand, nullptr, "CEC volume", std::move(settingState), 0, false});
    }

    settingState.clear();
    settingState.push_back({"OFF", 0});
    settingState.push_back({"ON", 1});
    m_Setting.push_back({&HdmiConfig::SwitchLabelResetting, nullptr, "Resetting", std::move(settingState), 0, false});

    m_BackGroundColor = DefaultBackGroundColor;
}

HdmiConfig::~HdmiConfig() NN_NOEXCEPT
{
    SwitchAudio(0);
}

void HdmiConfig::ApplySelectedLabel() NN_NOEXCEPT
{
    for (auto& setting : m_Setting)
    {
        if (m_IsLabelResetting || setting.isStateIndexChanged)
        {
            if (0 < setting.state.size() && nullptr != setting.switchState)
            {
                (this->*(setting.switchState))(setting.state[setting.stateIndex].level);
            }
        }
        setting.isStateIndexChanged = false;
    }
}

int HdmiConfig::GetConfigCount() NN_NOEXCEPT
{
    return static_cast<int>(m_Setting.size());
}

const char* HdmiConfig::GetConfigCaption(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    return target.caption;
}

const char* HdmiConfig::GetConfigStatus(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    if (nullptr == target.getStatus)
    {
        return "";
    }
    return (this->*(target.getStatus))();
}

const char* HdmiConfig::GetConfigLabel(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    if (0 < target.state.size())
    {
        std::string label = target.state[target.stateIndex].label;
        if ("" != label)
        {
            std::snprintf(m_LabelString, LabelStringLength, "%s", label.c_str());
        }
        else
        {
            std::snprintf(m_LabelString, LabelStringLength, "%d", target.state[target.stateIndex].level);
        }
        return m_LabelString;
    }
    return "None";
}

void HdmiConfig::SetConfig(int index, bool isAscending) NN_NOEXCEPT
{
    SettingContext& target = m_Setting[index];
    if (0 < target.state.size() && nullptr != target.switchState)
    {
        if (isAscending)
        {
            target.stateIndex = (target.stateIndex + 1) % target.state.size();
        }
        else
        {
            target.stateIndex = (target.stateIndex <= 0) ? target.state.size() - 1 : target.stateIndex - 1;
        }
        (this->*(target.switchState))(target.state[target.stateIndex].level);
        target.isStateIndexChanged = true;
    }
}

void HdmiConfig::UpdateConfig() NN_NOEXCEPT
{
    int configs = m_Video.GetDisplayInfoCount();
    // ドックイン or ドックアウトした直後に configs が 1 になる問題へのWAR.
    if (1 == configs) {
        return;
    }

    bool isChanged = m_PreVideoConfigs != configs;
    m_PreVideoConfigs = configs;
    if (isChanged && 0 == m_VideoConfigSeries)
    {
        int stateIndex = GetSelectedConfigModeStateIndex();
        std::vector<SettingContext::State> settingState;
        bool isInserted = 0 < configs;
        // HDMIが接続されたら、EDIDを再検索して対応している解像度のみ表示する。
        // HDMIが接続されなくなっても、解像度の設定は 480p, 720p, 1080p から選択できるようにしておく。
        AssignVideoConfigState(&settingState, !isInserted);
        ResetVideoConfigsState(settingState, stateIndex, isInserted);
        // 再設定する。
        if (isInserted)
        {
            ApplySelectedLabel();
        }
    }

    //// hdcp_status を呼ぶ度にログが出るので、呼ぶ頻度を下げている。
    //if ((++m_HdcpPollingCount % 100) == 0 && m_IsHdcpSetEnabled)
    //{
    //    m_IsHdcpReady = m_Hdcp.IsHdcpReady();
    //}
    //// ドックインしているかつHDCPを有効にしている過程の間だけ音声をミュートする。
    //nns::audio::MuteAudio(HdcpState_Going == m_State && m_IsHdcpSetEnabled && 0 < configs);
}

bool HdmiConfig::IsSceneSwitched() NN_NOEXCEPT
{
    return false;
}

const Color& HdmiConfig::GetBackGroundColor() NN_NOEXCEPT
{
    return m_BackGroundColor;
}

void HdmiConfig::SetBackGroundColor(float rRatio, float gRatio, float bRatio, float aRatio) NN_NOEXCEPT
{
    m_BackGroundColor.rRatio = rRatio;
    m_BackGroundColor.gRatio = gRatio;
    m_BackGroundColor.bRatio = bRatio;
    m_BackGroundColor.aRatio = aRatio;
}

void HdmiConfig::SwitchVideoConfig(int configMode) NN_NOEXCEPT
{
    if (0 <= configMode)
    {
        // 解像度の設定と取得
        auto result = m_Video.SetDisplayInfo(configMode);
        NN_LOG("%s:[%d]\n", result ? "Set the config mode" : "Config mode is illegal", configMode);
        NN_LOG("=============================================\n");
    }
    m_Video.ShowDisplayInfo();
}

void HdmiConfig::SwitchVideoConfigSeries(int series) NN_NOEXCEPT
{
    m_VideoConfigSeries = series;
    int configs = m_Video.GetDisplayInfoCount();
    std::vector<SettingContext::State> settingState;
    bool isInserted = 0 < configs;
    // HDMIが接続されたら、EDIDを再検索して対応している解像度のみ表示する。
    AssignVideoConfigState(&settingState, !isInserted);
    ResetVideoConfigsState(settingState, 0, isInserted);
}

void HdmiConfig::SwitchRgbRange(int rgbRange) NN_NOEXCEPT
{
    // RGBレンジの設定と取得
    int previousRgbRange = -1;
    if (m_Video.GetRgbRange(&previousRgbRange))
    {
        if (m_Video.SetRgbRange(rgbRange))
        {
            NN_LOG("Changed RGB range from [%s] to [%s].\n", GetRgbRangeString(previousRgbRange), GetRgbRangeString(rgbRange));
            NN_LOG("===============================================================================================\n");
        }
        else
        {
            NN_LOG("Warning: Set RGB range is illegal:[%d]\n", rgbRange);
        }
    }
    else
    {
        NN_LOG("Warning: Cannot read current RGB range.\n");
    }
}

void HdmiConfig::SwitchContentType(int contentType) NN_NOEXCEPT
{
    // コンテンツタイプの設定と取得
    nn::settings::system::HdmiContentType currentContentType = static_cast<nn::settings::system::HdmiContentType>(contentType);
    nn::settings::system::HdmiContentType previousContentType;
    if (m_Video.GetContentType(&previousContentType))
    {
        if (m_Video.SetContentType(currentContentType))
        {
            NN_LOG("Changed content type from [%s] to [%s].\n", GetContentTypeString(previousContentType), GetContentTypeString(currentContentType));
            NN_LOG("===============================================================================================\n");
        }
        else
        {
            NN_LOG("Warning: Set content type is illegal:[%d]\n", contentType);
        }
    }
    else
    {
        NN_LOG("Warning: Cannot read current content type.\n");
    }
}

void HdmiConfig::SwitchHdcp(int isEnabled) NN_NOEXCEPT
{
    // HDCP の on/off を切り替える。
    if (m_Hdcp.SwitchHdcp(isEnabled))
    {
        // NSBG-7193 のような怪しい状態遷移があるため、
        // hdcp_status() から返ってくる値を用いない。
        if (0 < m_Video.GetDisplayInfoCount())
        {
            m_State = HdcpState_Going;
            m_HdcpStartMilliSeconds = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        }
        else
        {
            m_State = HdcpState_Unpluged;
        }
        m_IsHdcpSetEnabled = isEnabled;
        NN_LOG("Switched hdcp %s.\n", isEnabled ? "enabled" : "disabled");
        NN_LOG("===============================================================================================\n");
    }
    else
    {
        NN_LOG("Warning: Cannot switch hdcp %s.\n", isEnabled ? "enabled" : "disabled");
    }
}

void HdmiConfig::SetHdcpAuthenticationTimeout(int msec) NN_NOEXCEPT
{
    NN_LOG("Set HDCP authentication timeout:[%d]\n", msec);
    m_Hdcp.SetAuthenticationTimeout(msec);

}

void HdmiConfig::SwitchHdcpStubEmulation(int isEnabled) NN_NOEXCEPT
{
    NN_LOG("Enabled stub emulation in HDCP library:[%d]\n", isEnabled);
    m_Hdcp.SetIsStubEmulation(static_cast<bool>(isEnabled));
}

void HdmiConfig::SwitchAudio(int channels) NN_NOEXCEPT
{
    // 既に音が鳴っている場合は止める。
    if (nns::audio::IsInitialized())
    {
        nns::audio::FinalizeAudioRenderer();
    }
    if (0 < channels)
    {
        // 音を鳴らす。
        nns::audio::InitializeAudioRenderer(channels);
        for (int i1=0; i1<channels; ++i1)
        {
            // "sin" という waveFilePath が入力されれば、sin 波が再生される。
            nns::audio::SetWaveFile((6 > channels) ? m_StereoAudioContent[i1] : m_SurroundAudioContent[i1], i1);
        }
        nns::audio::StartAudioRenderer();
    }
}

void HdmiConfig::SendCecCommand(int cecCommandType) NN_NOEXCEPT
{
    if (0 < cecCommandType)
    {
        bool result = false;
        if (1 == cecCommandType)
        {
            NN_LOG("Send CEC command -> TV go standby.\n");
            result = m_Cec.SendTvStandby();
            NN_LOG("%s\n", result ? "Sending command, \"TV go standby\", is successful."
                : "Warning: Sending command, \"TV go standby\", failed");
        }
        else if (2 == cecCommandType)
        {
            NN_LOG("Send CEC command -> One touch play.\n");
            result = m_Cec.SendOneTouchPlay();
            NN_LOG("%s\n", result ? "Sending command, \"One touch play\", is successful."
                : "Warning: Sending command, \"One touch play\", failed");
        }
    }
    else
    {
        NN_LOG("Not send CEC command...\n");
    }
    NN_LOG("===============================================================================================\n");
}

void HdmiConfig::SendCecVolumeCommand(int cecCommandType) NN_NOEXCEPT
{
    if (0 < cecCommandType)
    {
        bool result = false;
        if (1 == cecCommandType)
        {
            NN_LOG("Send CEC command -> Volume up.\n");
            result = m_Cec.SendTvVolumeUp();
            NN_LOG("%s\n", result ? "Sending command, \"Volume up\", is successful."
                : "Warning: Sending command, \"Volume up\", failed");
        }
        else if (2 == cecCommandType)
        {
            NN_LOG("Send CEC command -> Volume mute.\n");
            result = m_Cec.SendTvVolumeMute();
            NN_LOG("%s\n", result ? "Sending command, \"Volume mute\", is successful."
                : "Warning: Sending command, \"Volume mute\", failed");
        }
        else if (3 == cecCommandType)
        {
            NN_LOG("Send CEC command -> Volume down.\n");
            result = m_Cec.SendTvVolumeDown();
            NN_LOG("%s\n", result ? "Sending command, \"Volume down\", is successful."
                : "Warning: Sending command, \"Volume down\", failed");
        }
    }
    else
    {
        NN_LOG("Not send CEC command...\n");
    }
    NN_LOG("===============================================================================================\n");
}

void HdmiConfig::SwitchLabelResetting(int isEnabled) NN_NOEXCEPT
{
    NN_LOG("Switched label resetting %s.\n", isEnabled ? "enabled" : "disabled");
    NN_LOG("===============================================================================================\n");
    m_IsLabelResetting = isEnabled;
}

const char* HdmiConfig::GetHdcpStatus() NN_NOEXCEPT
{
    if (m_IsHdcpSetEnabled)
    {
        if (HdcpState_Done == m_State)
        {
            int sec = static_cast<int>(m_HdcpElapsedMilliSeconds / 1000);
            int msec = static_cast<int>((m_HdcpElapsedMilliSeconds / 10) % 100);
            std::snprintf(m_LabelString, LabelStringLength, "(Done)%d.%02ds", sec, msec);
            m_BackGroundColor = DefaultBackGroundColor;
            return m_LabelString;
        }
        else if (HdcpState_Going == m_State)
        {
            m_BackGroundColor = ProgressBackGroundColor;
            return "(Going)";
        }
        else if (HdcpState_Failed == m_State)
        {
            m_BackGroundColor = FailureBackGroundColor;
            return "(Failed)";
        }
        else if (HdcpState_Unpluged == m_State)
        {
            m_BackGroundColor = DefaultBackGroundColor;
            return "(Unpluged)";
        }
        else
        {
            m_BackGroundColor = DefaultBackGroundColor;
        }
    }
    else
    {
        m_BackGroundColor = DefaultBackGroundColor;
    }
    return "";
}

const char* HdmiConfig::GetCecStatus() NN_NOEXCEPT
{
    switch (m_Cec.GetStoredBusEventType())
    {
    case nn::cec::BusEventType_Ignore: return "(Ignore)";
    case nn::cec::BusEventType_ActiveSourceChangedToActive: return "(Active)";
    case nn::cec::BusEventType_ActiveSourceChangedToInactive: return "(Inactive)";
    case nn::cec::BusEventType_GoStandby: return "(Standby)";
    case nn::cec::BusEventType_Suspending: return "(Suspend)";
    case nn::cec::BusEventType_ConnectionChange: return "(Change)";
    default: return "";
    }
    return "";
}

int HdmiConfig::GetSelectedConfigModeStateIndex() NN_NOEXCEPT
{
    for (std::vector<SettingContext>::iterator itr=m_Setting.begin(); itr!=m_Setting.end(); ++itr)
    {
        if (&HdmiConfig::SwitchVideoConfig == itr->switchState && itr->stateIndex < itr->state.size())
        {
            return itr->stateIndex;
        }
    }
    return -1;
}

void HdmiConfig::AssignVideoConfigState(std::vector<SettingContext::State>* pSettingState, bool isListUpAll) NN_NOEXCEPT
{
    pSettingState->clear();
    struct VideoConfig
    {
        const char* caption;
        const nns::VideoConfigUtility::EdidVic vic;
    };
    if (0 == m_VideoConfigSeries || isListUpAll)
    {
        const VideoConfig videoConfig[] =
        {
            { "720x480p", nns::VideoConfigUtility::EdidVic_Vic3 },
            { "1280x720p", nns::VideoConfigUtility::EdidVic_Vic4 },
            { "1920x1080p", nns::VideoConfigUtility::EdidVic_Vic16 },
            { "4K 30fps", nns::VideoConfigUtility::EdidVic_Vic95 },
            { "4K 60fps", nns::VideoConfigUtility::EdidVic_Vic97 },
        };
        for (auto& config : videoConfig)
        {
            int mode = m_Video.FindDisplayConfigModeFromVic(config.vic);
            if (0 <= mode || isListUpAll)
            {
                pSettingState->push_back({ config.caption, mode });
            }
        }
    }
    else
    {
        int configs = m_Video.GetDisplayInfoCount();
        for (int i1 = 0; i1 < configs; ++i1)
        {
            android::DisplayInfo config = m_Video.GetDisplayInfo(i1);
            if (59.0f <= config.fps && config.fps <= 60.1f)
            {
                char buffer[CharBufferLength];
                std::snprintf(buffer, CharBufferLength, "%dx%d[%d/%d]", config.w, config.h, i1, configs);
                pSettingState->push_back({ buffer, i1 });
            }
        }

    }
    // 任意の config mode が指定されていた場合、それを選択できるようにする。
    if (0 <= m_ConfigMode)
    {
        pSettingState->push_back({"Specific\nconfig mode", m_ConfigMode});
    }
}

void HdmiConfig::AssignHdcpState(std::vector<SettingContext::State>* pSettingState, bool isHdcpEnabled) NN_NOEXCEPT
{
    m_IsStubEmulationEnabled = m_Hdcp.GetIsStubEmulation();
    if (m_IsStubEmulationEnabled)
    {
        pSettingState->clear();
        pSettingState->push_back({ "OFF", 0 });
        pSettingState->push_back({ "ON", 1 });
        int initialIndex = FindSettingStateIndexFromLevel(*pSettingState, static_cast<int>(isHdcpEnabled));
        m_Setting.push_back({ &HdmiConfig::SwitchHdcp, &HdmiConfig::GetHdcpStatus, "HDCP", std::move(*pSettingState), initialIndex, false });

        m_HdcpAuthenticationTimeout = m_Hdcp.GetAuthenticationTimeout();
        pSettingState->clear();
        for (int i1 = 1; i1 <= 20; ++i1)
        {
            pSettingState->push_back({ "", i1 * 1000 });
        }
        pSettingState->push_back({ "", 0x7fffffff });
        initialIndex = FindSettingStateIndexFromLevel(*pSettingState, m_HdcpAuthenticationTimeout);
        m_Setting.push_back({ &HdmiConfig::SetHdcpAuthenticationTimeout, nullptr, "HDCP timeout", std::move(*pSettingState), initialIndex, false });
    }

    pSettingState->clear();
    pSettingState->push_back({ "Disabled", 0 });
    pSettingState->push_back({ "Enabled", 1 });
    int initialIndex = FindSettingStateIndexFromLevel(*pSettingState, m_IsStubEmulationEnabled);
    m_Setting.push_back({ &HdmiConfig::SwitchHdcpStubEmulation, nullptr, "HDCP access", std::move(*pSettingState), initialIndex, false });

    static HdcpManager::StateTransitionType type =
    {
        [](void* arg)
    {
        auto config = reinterpret_cast<HdmiConfig*>(arg);
        config->m_State = HdcpState_Done;
        config->m_HdcpElapsedMilliSeconds =
            nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - config->m_HdcpStartMilliSeconds;
    },
        [](void* arg)
    {
        auto config = reinterpret_cast<HdmiConfig*>(arg);
        config->m_State = HdcpState_Going;
        config->m_HdcpStartMilliSeconds = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    },
        [](void* arg) { reinterpret_cast<HdmiConfig*>(arg)->m_State = HdcpState_Failed; },
        [](void* arg) { reinterpret_cast<HdmiConfig*>(arg)->m_State = HdcpState_Unpluged; },
        this,
    };
    m_Hdcp.SetHdcpListener(&type);
}

void HdmiConfig::ResetVideoConfigsState(std::vector<SettingContext::State>& settingState, int stateIndex, bool isInserted) NN_NOEXCEPT
{
    for (std::vector<SettingContext>::iterator itr=m_Setting.begin(); itr!=m_Setting.end(); ++itr)
    {
        if (&HdmiConfig::SwitchVideoConfig == itr->switchState)
        {
            // 既に登録されている場合はここへ。
            if (!IsSameVideoConfigState(itr->state, settingState) && 0 < settingState.size())
            {
                if (!(itr->isStateIndexChanged || m_IsLabelResetting) && isInserted)
                {
                    // nvdisp 側でドックイン時に解像度が設定されるので、再設定する場合はここで上書きする。
                    stateIndex = FindSettingStateIndexFromLevel(settingState, m_Video.GetDisplayInfo());
                }
                // 解像度のIDの状況が変わっていたら上書きする。
                itr->state = std::move(settingState);
                itr->stateIndex = (stateIndex < itr->state.size()) ? stateIndex : itr->state.size() - 1;
            }
            break;
        }
    }
}

bool HdmiConfig::IsSameVideoConfigState(const std::vector<SettingContext::State>& lhs, const std::vector<SettingContext::State>& rhs) NN_NOEXCEPT
{
    if (lhs.size() != rhs.size())
    {
        return false;
    }
    for (int i1=0; i1<lhs.size(); ++i1)
    {
        if (lhs[i1].level != rhs[i1].level)
        {
            return false;
        }
    }
    return true;
}

int HdmiConfig::FindSettingStateIndexFromLevel(std::vector<SettingContext::State>& settingState, int level) NN_NOEXCEPT
{
    for (int i1=0; i1<settingState.size(); ++i1)
    {
        if (level == settingState[i1].level)
        {
            return i1;
        }
    }
    // 見つからなかったら便宜上0を返す。
    return 0;
}

}
