﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/viUtil/testVi_PowerFixture.h>
#include <nn/nn_Assert.h>
#include <nn/vi/vi_DisplayMode.private.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include <nn/vi/vi_Result.h>
#include "testDisplay_AndroidDisplayFixture.h"
#include "master/detail/visrv_HdmiDisplay.h"
#include "master/detail/visrv_LcdDisplay.h"
#include "master/detail/visrv_DpDisplay.h"
#include "testDisplay_MockFilter.h"
#include "settings/visrv_InterfaceType.h"

// These are mostly stability tests since the AndroidDisplay has a direct dependency
// on SurfaceFlinger.

namespace
{
    // Note:  Avoiding calls to query HPD to avoid opening another nvdc handle
    class LcdDisplay : public nn::visrv::master::detail::LcdDisplay
    {
    public:
        explicit LcdDisplay(nn::visrv::master::detail::IModeFilter* pFilter) NN_NOEXCEPT
            : nn::visrv::master::detail::LcdDisplay(pFilter)
        {
        }

        virtual nn::Result GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT_NOT_NULL(pOutState);
            *pOutState = nn::vi::HotplugState_Connected;
            NN_RESULT_SUCCESS;
        }
    };

    class HdmiDisplay : public nn::visrv::master::detail::HdmiDisplay
    {
    public:
        explicit HdmiDisplay(nn::visrv::master::detail::IModeFilter* pFilter) NN_NOEXCEPT
            : nn::visrv::master::detail::HdmiDisplay(pFilter)
        {
        }

        virtual nn::Result GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT_NOT_NULL(pOutState);
            *pOutState = nn::vi::HotplugState_Connected;
            NN_RESULT_SUCCESS;
        }
    };

    class DpDisplay : public nn::visrv::master::detail::DpDisplay
    {
    public:
        explicit DpDisplay(nn::visrv::master::detail::IModeFilter* pFilter) NN_NOEXCEPT
            : nn::visrv::master::detail::DpDisplay(pFilter)
        {
        }

        virtual nn::Result GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT_NOT_NULL(pOutState);
            *pOutState = nn::vi::HotplugState_Connected;
            NN_RESULT_SUCCESS;
        }
    };

    typedef ::testing::Types<LcdDisplay,
                             HdmiDisplay,
                             DpDisplay> AndroidDisplayTypes;
    TYPED_TEST_CASE(AndroidDisplayFixture, AndroidDisplayTypes);

    template <typename T>
    class ExternalDisplay : public ::testing::Test
    {
    };

    typedef ::testing::Types<HdmiDisplay,
                             DpDisplay> ExternalDisplayTypes;
    TYPED_TEST_CASE(ExternalDisplay, ExternalDisplayTypes);

    class LcdDisplayTest : public nnt::vi::util::PowerFixture
    {
    };
}

TYPED_TEST(ExternalDisplay, SetUnderscan_InvalidRange_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, display.SetUnderscan(-1));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, display.SetUnderscan(21));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetAlpha_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_SUCCESS(display.SetAlpha(0.f));
    NNT_EXPECT_RESULT_SUCCESS(display.SetAlpha(1.f));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetLayerStack_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_SUCCESS(display.SetLayerStack(nn::vi::LayerStack_Lcd));
    NNT_EXPECT_RESULT_SUCCESS(display.SetLayerStack(nn::vi::LayerStack_Default));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, IsVsyncEventSupported_Any)
{
    MockFilter filter;
    TypeParam display(&filter);

    EXPECT_TRUE(display.IsVsyncEventSupported());
}

TYPED_TEST(ExternalDisplay, Underscan_IcosaCopper)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    for( int i = 0; i <= 20; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(display.SetUnderscan(i));

        int underscan;
        NNT_ASSERT_RESULT_SUCCESS(display.GetUnderscan(&underscan));
        EXPECT_EQ(underscan, i);
    }

    display.Close();
}

TEST_F(LcdDisplayTest, Underscan_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);
    NNT_ASSERT_RESULT_SUCCESS(display.Open());

    for( int i = 0; i <= 20; ++i )
    {
        NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, display.SetUnderscan(i));

        int underscan;
        NNT_ASSERT_RESULT_SUCCESS(display.GetUnderscan(&underscan));
        EXPECT_EQ(0, underscan);
    }

    display.Close();
}

TEST_F(LcdDisplayTest, GetMode_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);
    NNT_ASSERT_RESULT_SUCCESS(display.Open());
    NNT_ASSERT_RESULT_SUCCESS(display.SetPowerState(nn::vi::PowerState_On));

    nn::vi::DisplayModeInfo mode;
    memset(&mode, 0, sizeof(mode));

    NNT_EXPECT_RESULT_SUCCESS(display.GetMode(&mode));
    EXPECT_EQ(1280, mode.width);
    EXPECT_EQ(720, mode.height);
    EXPECT_FLOAT_EQ(60.f, mode.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, mode.mode);

    display.Close();
}

TEST_F(LcdDisplayTest, ListModes_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);
    NNT_ASSERT_RESULT_SUCCESS(display.Open());
    NNT_ASSERT_RESULT_SUCCESS(display.SetPowerState(nn::vi::PowerState_On));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    memset(modes, 0, sizeof(modes));

    EXPECT_EQ(1, display.ListModes(modes, nn::vi::DisplayModeCountMax));
    EXPECT_EQ(1280, modes->width);
    EXPECT_EQ(720, modes->height);
    EXPECT_FLOAT_EQ(60.f, modes->refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, modes->mode);
    // make sure the filter is being used
    EXPECT_TRUE(filter.IsFilterUsed());
}

TEST_F(LcdDisplayTest, SetMode_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);
    NNT_ASSERT_RESULT_SUCCESS(display.Open());
    NNT_ASSERT_RESULT_SUCCESS(display.SetPowerState(nn::vi::PowerState_On));

    nn::vi::DisplayModeInfo oldMode;
    NNT_EXPECT_RESULT_SUCCESS(display.GetMode(&oldMode));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    memset(modes, 0, sizeof(modes));

    int modeCount = display.ListModes(modes, nn::vi::DisplayModeCountMax);
    EXPECT_NE(0, modeCount);

    for( int i = 0; i < modeCount; ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetMode(modes[i]));
        // make sure the filter is being used
        EXPECT_TRUE(filter.IsFilterUsed());
        filter.Reset();

        nn::vi::DisplayModeInfo mode;
        NNT_EXPECT_RESULT_SUCCESS(display.GetMode(&mode));
        EXPECT_EQ(modes[i].width, mode.width);
        EXPECT_EQ(modes[i].height, mode.height);
        EXPECT_FLOAT_EQ(modes[i].refreshRate, mode.refreshRate);
        // 3D not supported yet
        EXPECT_EQ(modes[i].mode, mode.mode);
    }

    NNT_EXPECT_RESULT_SUCCESS(display.SetMode(oldMode));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, Ctor_Any)
{
    MockFilter filter;
    TypeParam display(&filter);

    nn::vi::PowerStateType state;
    NNT_EXPECT_RESULT_SUCCESS(display.GetPowerState(&state));
    EXPECT_EQ(nn::vi::PowerState_Off, state);
}

TYPED_TEST(AndroidDisplayFixture, SetCmuLuma_InvalidRange_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, display.SetCmuLuma(-1.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, display.SetCmuLuma(1.1f));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, CmuLuma_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_ASSERT_RESULT_SUCCESS(display.SetCmuLuma(-1.f));
    float luma = 0.f;
    NNT_ASSERT_RESULT_SUCCESS(display.GetCmuLuma(&luma));
    EXPECT_EQ(-1.f, luma);

    NNT_ASSERT_RESULT_SUCCESS(display.SetCmuLuma(1.f));
    NNT_ASSERT_RESULT_SUCCESS(display.GetCmuLuma(&luma));
    EXPECT_EQ(1.f, luma);
}

TEST_F(LcdDisplayTest, IsHotplugEventSupported_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);
    NNT_ASSERT_RESULT_SUCCESS(display.Open());

    EXPECT_FALSE(display.IsHotplugEventSupported());

    display.Close();
}

TYPED_TEST(ExternalDisplay, IsHotplugEventSupported_IcosaCopper)
{
    MockFilter filter;
    TypeParam display(&filter);

    EXPECT_TRUE(display.IsHotplugEventSupported());
}

TYPED_TEST(AndroidDisplayFixture, SetCmuMode_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    nn::vi::CmuMode modes[] =
    {
        nn::vi::CmuMode_Default,
        nn::vi::CmuMode_HighContrast,
        nn::vi::CmuMode_InvertColor,
        nn::vi::CmuMode_Grayscale,
    };

    for( int i = 0; i < sizeof(modes) / sizeof(modes[0]); ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetCmuMode(modes[i]));
        nn::vi::CmuModeType mode;
        NNT_EXPECT_RESULT_SUCCESS(display.GetCmuMode(&mode));
        EXPECT_EQ(modes[i], mode);
    }

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetCmuMode_NotSupported_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    nn::vi::CmuMode modes[] =
    {
        nn::vi::CmuMode_Disabled,
    };

    for( int i = 0; i < sizeof(modes) / sizeof(modes[0]); ++i )
    {
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, display.SetCmuMode(modes[i]));
    }

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetContrastRatio_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_SUCCESS(display.SetCmuMode(nn::vi::CmuMode_HighContrast));
    NNT_EXPECT_RESULT_SUCCESS(display.SetContrastRatio(0.f));

    float ratio;
    NNT_EXPECT_RESULT_SUCCESS(display.GetContrastRatio(&ratio));
    EXPECT_EQ(0.f, ratio);

    NNT_EXPECT_RESULT_SUCCESS(display.SetContrastRatio(1.f));
    NNT_EXPECT_RESULT_SUCCESS(display.GetContrastRatio(&ratio));
    EXPECT_EQ(1.f, ratio);

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetContrastRatio_NotSupported_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    NNT_EXPECT_RESULT_SUCCESS(display.SetCmuMode(nn::vi::CmuMode_Default));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, display.SetContrastRatio(0.f));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetViewport_ViewportPastLogicalResolution_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(display.GetLogicalResolution(&width, &height));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, display.SetViewport(1, 0, width + 1, height));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, display.SetViewport(0, 1, width, height + 1));

    display.Close();
}

TYPED_TEST(AndroidDisplayFixture, SetViewport_Any)
{
    MockFilter filter;
    TypeParam display(&filter);
    // Only call superclass's Open to avoid opening too many nvdcHandles
    NNT_ASSERT_RESULT_SUCCESS(display.AndroidDisplay::Open());

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(display.GetLogicalResolution(&width, &height));

    const int ViewportWidth = 1280;
    const int ViewportHeight = 720;

    for( int x = 0; x + ViewportWidth <= width; ++x )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(x, 0, ViewportWidth, ViewportHeight));
    }

    for( int x = 0; x + ViewportWidth <= width; ++x )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(x, height - ViewportHeight, ViewportWidth, ViewportHeight));
    }

    for( int y = 0; y + ViewportHeight <= height; ++y )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(0, y, ViewportWidth, ViewportHeight));
    }

    for( int y = 0; y + ViewportHeight <= height; ++y )
    {
        NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(width - ViewportWidth, y, ViewportWidth, ViewportHeight));
    }

    NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(0, 0, 1, 1));
    NNT_EXPECT_RESULT_SUCCESS(display.SetViewport(0, 0, width, height));
}

TYPED_TEST(ExternalDisplay, IsModeChangedEventSupported_IcosaCopper)
{
    MockFilter filter;
    TypeParam display(&filter);

    EXPECT_TRUE(display.IsModeChangedEventSupported());
}

TEST_F(LcdDisplayTest, IsModeChangedEventSupported_IcosaHoag)
{
    MockFilter filter;
    LcdDisplay display(&filter);

    EXPECT_FALSE(display.IsModeChangedEventSupported());
}

// TODO: Gamma tests.  Not supported in nvnflinger currently.
