﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>
#include "testDisplay_LayerFixture.h"
#include "testDisplay_ScalingModeFixture.h"
#include "testDisplay_PixelFormatFixture.h"
#include "testDisplay_LayerVisibilityFixture.h"

typedef LayerFixture Layer;
typedef ScalingModeFixture Layer_SupportedScalingModes;
typedef ScalingModeFixture Layer_UnsupportedScalingModes;

// TODO: move to process with only vi:u
TEST_F(Layer, DISABLED_CreateLayer_TooManyLayers_Any)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    nn::vi::Layer* pLayer2;
    nn::Result result = nn::vi::CreateLayer(&pLayer2, GetDisplay());
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultDenied, result);
    EXPECT_EQ(nullptr, pLayer2);

    nn::vi::DestroyLayer(pLayer);
}

TEST_P(Layer_SupportedScalingModes, SetLayerScalingMode)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerScalingMode(pLayer, GetParam()));

    nn::vi::DestroyLayer(pLayer);
}

INSTANTIATE_TEST_CASE_P(ScalingModes_Any,
                        Layer_SupportedScalingModes,
                        ::testing::Values(nn::vi::ScalingMode_FitToLayer,
                                          nn::vi::ScalingMode_PreserveAspectRatio));

TEST_P(Layer_UnsupportedScalingModes, SetLayerScalingMode)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    nn::Result result = nn::vi::SetLayerScalingMode(pLayer, GetParam());
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, result);

    nn::vi::DestroyLayer(pLayer);
}

INSTANTIATE_TEST_CASE_P(ScalingModes_Any,
                        Layer_UnsupportedScalingModes,
                        ::testing::Values(nn::vi::ScalingMode_Exact,
                                          nn::vi::ScalingMode_ScaleAndCrop,
                                          nn::vi::ScalingMode_None));

TEST_F(Layer, SetLayerSize_Any)
{
    nn::vi::Layer* pLayer;

    nn::vi::LayerSettings settings;
    nn::vi::SetLayerSettingsDefaults(&settings);
    settings.Reset<nn::vi::LayerFlags::Fullscreen>();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay(), settings));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, GetDisplay()));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, 0, height));

    // reset size
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, 0, 0));
    // set to non-trivial position
    const int x = 1;
    const int y = 2;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, x, y));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width - x, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, 0, height - y));

    nn::vi::DestroyLayer(pLayer);
}

TEST_F(Layer, SetLayerSize_InvalidRange_Any)
{
    nn::vi::Layer* pLayer;

    nn::vi::LayerSettings settings;
    nn::vi::SetLayerSettingsDefaults(&settings);
    settings.Reset<nn::vi::LayerFlags::Fullscreen>();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay(), settings));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, GetDisplay()));

    // set to non-trivial position
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 1, 2));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, width, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, 0, height - 1));
}

TEST_F(Layer, SetLayerAlpha_Any)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    // default value is 1.f -- reduce alpha
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerAlpha(pLayer, 0.f));
    // increase alpha to max
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerAlpha(pLayer, 1.f));

    nn::vi::DestroyLayer(pLayer);
}

TEST_F(Layer, SetLayerAlpha_InvalidRange_Any)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    nn::Result result = nn::vi::SetLayerAlpha(pLayer, -1.f);
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, result);

    result = nn::vi::SetLayerAlpha(pLayer, 2.f);
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, result);

    nn::vi::DestroyLayer(pLayer);
}

TEST_F(Layer, GetNativeWindow_Any)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    nn::vi::NativeWindowHandle handle = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetNativeWindow(&handle, pLayer));
    EXPECT_NE(nullptr, handle);

    nn::vi::DestroyLayer(pLayer);
}

TEST_F(Layer, SetLayerPosition_Any)
{
    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, GetDisplay()));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    // default position (should be nop)
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 0.f, 0.f));

    nn::vi::DestroyLayer(pLayer);
}

TEST_F(Layer, SetLayerPosition_Boundaries_Any)
{
    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, GetDisplay()));

    nn::vi::Layer* pLayer;

    nn::vi::LayerSettings settings;
    nn::vi::SetLayerSettingsDefaults(&settings);
    settings.Reset<nn::vi::LayerFlags::Fullscreen>();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay(), settings));

    // move to top-right
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, width - 1, 0.f));
    // move to bottom-left
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 0.f, height - 2));
    // move to bottom-right
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, width - 1, height - 2));
}

TEST_F(Layer, SetLayerPosition_InvalidRange_Any)
{
    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, GetDisplay()));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    // beyond left edge
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, -1.f, 0.f));
    // beyond top edge
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, -1.f));
    // beyond right edge
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, width, 0.f));
    // beyond bottom edge
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, height));
}

TEST_F(Layer, SetLayerZ_Any)
{
    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, GetDisplay()));

    int z;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetLayerZ(&z, pLayer));
    // should always be set to minimum on creation
    EXPECT_EQ(nn::vi::GetZOrderCountMin(GetDisplay()), z);

    // no change - perfectly legal
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerZ(pLayer, nn::vi::GetZOrderCountMin(GetDisplay())));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetLayerZ(&z, pLayer));
    ASSERT_EQ(nn::vi::GetZOrderCountMin(GetDisplay()), z);

    // increase to max z-order
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerZ(pLayer, nn::vi::GetZOrderCountMax(GetDisplay())));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetLayerZ(&z, pLayer));
    ASSERT_EQ(nn::vi::GetZOrderCountMax(GetDisplay()), z);

    // move back to minimum z-order
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerZ(pLayer, nn::vi::GetZOrderCountMin(GetDisplay())));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetLayerZ(&z, pLayer));
    ASSERT_EQ(nn::vi::GetZOrderCountMin(GetDisplay()), z);

    nn::vi::DestroyLayer(pLayer);
}
