﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/os.h>
#include <nn/vi.private.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_CmuLuma.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nnt/viUtil/testVi_PowerFixture.h>

namespace
{
    class ExternalDisplay : public nnt::vi::util::Fixture,
                            public nnt::vi::util::PowerFixture
    {
    };
}

TEST_F(ExternalDisplay, Underscan_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    for( int i = 0; i <= 20; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayUnderscan(pDisplay, i));

        int underscan;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayUnderscan(&underscan, pDisplay));
        EXPECT_EQ(i, underscan);
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetUnderscan_InvalidRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayUnderscan(pDisplay, -1));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayUnderscan(pDisplay, 21));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetDisplayAlpha_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayAlpha(pDisplay, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayAlpha(pDisplay, 1.f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetDisplayAlpha_InvalidRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayAlpha(pDisplay, -.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayAlpha(pDisplay, 1.1f));

    nn::vi::CloseDisplay(pDisplay);
}

// Disabled until we figure out the final design
TEST_F(ExternalDisplay, DISABLED_SetDisplayPowerState_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_Off));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_On));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, GetDisplayHotplugEvent_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::os::SystemEventType hotplugEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayHotplugEvent(&hotplugEvent, pDisplay));
    // best effort to see if event is initialized
    EXPECT_NE(nn::os::SystemEventType::State_NotInitialized, hotplugEvent._state);

    {
        nn::os::SystemEventType hotplugEvent;
        // only one event per client
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultDenied, nn::vi::GetDisplayHotplugEvent(&hotplugEvent, pDisplay));
    }

    nn::os::DestroySystemEvent(&hotplugEvent);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetDisplayLayerStack_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Lcd));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Default));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetDisplayCmuLuma_Icosa)
{
    nn::vi::Initialize();

    nn::vi::Display* pExternal;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternal, "External"));
    nn::vi::Display* pInternal;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pInternal, "Internal"));

    float internalLuma;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&internalLuma, pInternal));

    float values[] = { -1.f, 1.f, 0.f };

    for( float value : values )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuLuma(pExternal, value));
        float luma;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&luma, pExternal));
        EXPECT_EQ(value, luma);

        // make sure only one pipeline is affected
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&luma, pInternal));
        EXPECT_EQ(internalLuma, luma);
    }

    nn::vi::CloseDisplay(pExternal);
    nn::vi::CloseDisplay(pInternal);
}

TEST_F(ExternalDisplay, SetDisplayCmuLuma_InvalidRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, -1.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, 1.1f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, GetDisplayVsyncEvent_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType vsync;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));
    // best effort to see if event is initialized
    EXPECT_NE(nn::os::SystemEventType::State_NotInitialized, vsync._state);

    EXPECT_TRUE(nn::os::TimedWaitSystemEvent(&vsync, nn::TimeSpan::FromSeconds(1)));

    nn::os::DestroySystemEvent(&vsync);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, GetDisplayLogicalResolution_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    EXPECT_EQ(1920, width);
    EXPECT_EQ(1080, height);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetRgbRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::vi::HotplugState hotplug;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayHotplugState(&hotplug, pDisplay));

    if( hotplug == nn::vi::HotplugState_Connected )
    {
        nn::vi::RgbRange ranges[nn::vi::RgbRange_Max];
        int rgbCount = nn::vi::ListDisplayRgbRanges(ranges, nn::vi::RgbRange_Max, pDisplay);
        ASSERT_NE(0, rgbCount);

        bool hasFullRangeSupport = false;
        for( int i = 0; i < rgbCount; ++i )
        {
            hasFullRangeSupport = hasFullRangeSupport || ranges[i] == nn::vi::RgbRange_Full;

            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, ranges[i]));
            nn::vi::RgbRange actualRange;
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));
            EXPECT_EQ(ranges[i], actualRange);
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Auto));
        nn::vi::RgbRange actualRange;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));

        if( hasFullRangeSupport )
        {
            EXPECT_EQ(nn::vi::RgbRange_Full, actualRange);
        }
        else
        {
            EXPECT_EQ(nn::vi::RgbRange_Limited, actualRange);
        }
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, SetDisplayMode_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::vi::HotplugState hotplug;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayHotplugState(&hotplug, pDisplay));

    if( hotplug == nn::vi::HotplugState_Connected )
    {
        nn::vi::DisplayModeInfo oldMode;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&oldMode, pDisplay));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, &oldMode));
        };

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
        ASSERT_NE(modeCount, 0);

        nn::os::SystemEvent modeChangedEvent;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayModeChangedEvent(modeChangedEvent.GetBase(), pDisplay));
        ASSERT_NE(nn::os::SystemEventType::State_NotInitialized, modeChangedEvent.GetBase()->_state);

        for( int i = 0; i < modeCount; ++i )
        {
            EXPECT_FALSE(modeChangedEvent.TryWait());
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, modes + i));
            EXPECT_TRUE(modeChangedEvent.TimedWait(nn::TimeSpan::FromSeconds(1)));
        }
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ExternalDisplay, GetDisplayModeChangedEvent_Duplicate_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::os::SystemEvent event;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayModeChangedEvent(event.GetBase(), pDisplay));

    nn::os::SystemEvent dupe;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultDenied, nn::vi::GetDisplayModeChangedEvent(dupe.GetBase(), pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}
