﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_CmuLuma.h>
#include <nn/os/os_SystemEvent.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/viUtil/testVi_EdidFixture.h>
#include <nnt/viUtil/testVi_Fixture.h>

class EdidDisplay : public nnt::vi::util::Fixture,
                    public nnt::vi::util::EdidFixture
{
};

TEST_F(EdidDisplay, GetUnderscan_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    int underscan = 50;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayUnderscan(&underscan, pDisplay));
    // make sure no writes happen
    EXPECT_EQ(50, underscan);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetUnderscan_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayUnderscan(pDisplay, 10));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetAlpha_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayAlpha(pDisplay, 1.f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetDisplayPowerState_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_On));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetDisplayLayerStack_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Default));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayHotplugEvent_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::os::SystemEventType hotplugEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayHotplugEvent(&hotplugEvent, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetDisplayCmuLuma_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayCmuLuma(pDisplay, 1.f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayCmuLuma_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    float luma;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayCmuLuma(&luma, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SIGLONTD_4999_IcosaCopper)
{
    // EDID from Sharp Aquos LC-19A33M-BK
    const std::uint8_t edid[] =
    {
        0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
        0x4d, 0x10, 0x02, 0x10, 0x01, 0x01, 0x01, 0x01,
        0x00, 0x11, 0x01, 0x03, 0x80, 0x2c, 0x19, 0x78,
        0x0a, 0xef, 0xca, 0xa3, 0x55, 0x47, 0x9a, 0x24,
        0x12, 0x47, 0x49, 0x00, 0x00, 0x00, 0x01, 0x01,
        0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
        0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x1d,
        0x80, 0x18, 0x71, 0x1c, 0x16, 0x20, 0x58, 0x2c,
        0x25, 0x00, 0xb8, 0xfa, 0x10, 0x00, 0x00, 0x9e,
        0x01, 0x1d, 0x00, 0x72, 0x51, 0xd0, 0x1e, 0x20,
        0x6e, 0x28, 0x55, 0x00, 0xb8, 0xfa, 0x10, 0x00,
        0x00, 0x1e, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x53,
        0x48, 0x41, 0x52, 0x50, 0x20, 0x48, 0x44, 0x4d,
        0x49, 0x0a, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd,
        0x00, 0x31, 0x3d, 0x1f, 0x2e, 0x08, 0x00, 0x0a,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x02,
        0x02, 0x03, 0x1d, 0x71, 0x49, 0x94, 0x05, 0x13,
        0x04, 0x12, 0x03, 0x11, 0x02, 0x01, 0x23, 0x09,
        0x07, 0x07, 0x83, 0x01, 0x00, 0x00, 0x66, 0x03,
        0x0c, 0x00, 0x10, 0x00, 0x80, 0x01, 0x1d, 0x80,
        0xd0, 0x72, 0x1c, 0x16, 0x20, 0x10, 0x2c, 0x25,
        0x80, 0xb8, 0xfa, 0x10, 0x00, 0x00, 0x9e, 0x01,
        0x1d, 0x00, 0xbc, 0x52, 0xd0, 0x1e, 0x20, 0xb8,
        0x28, 0x55, 0x40, 0xb8, 0xfa, 0x10, 0x00, 0x00,
        0x1e, 0x8c, 0x0a, 0xd0, 0x90, 0x20, 0x40, 0x31,
        0x20, 0x0c, 0x40, 0x55, 0x00, 0xb8, 0xfa, 0x10,
        0x00, 0x00, 0x18, 0x8c, 0x0a, 0xd0, 0x8a, 0x20,
        0xe0, 0x2d, 0x10, 0x10, 0x3e, 0x96, 0x80, 0xb8,
        0xfa, 0x10, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf9,
    };

    nn::vi::Initialize();
    WriteEdid(edid, sizeof(edid));

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    std::memset(modes, 0, sizeof(modes));
    int count = nn::vi::ListDisplayModes(modes, nn::vi::DisplayModeCountMax, pDisplay);

    ASSERT_EQ(2, count);

    ASSERT_EQ(1280, modes[0].width);
    ASSERT_EQ(720, modes[0].height);
    ASSERT_EQ(60.f, modes[0].refreshRate);
    ASSERT_EQ(nn::vi::StereoMode_None, modes[0].mode);

    ASSERT_EQ(720, modes[1].width);
    ASSERT_EQ(480, modes[1].height);
    ASSERT_EQ(60.f, modes[1].refreshRate);
    ASSERT_EQ(nn::vi::StereoMode_None, modes[1].mode);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayVsyncEvent_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::os::SystemEventType vsync;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayLogicalResolution_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    int width;
    int height;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, SetDisplayRgbRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Auto));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Limited));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Full));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayRgbRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::vi::RgbRange range;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultOperationFailed, nn::vi::GetDisplayRgbRange(&range, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(EdidDisplay, GetDisplayModeChangedEvent_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::os::SystemEvent event;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayModeChangedEvent(event.GetBase(), pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}
