<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Verify TV is unaffected by powering on and off the test SDEV

    .DESCRIPTION
        Powers on and off the test SDEV.  Verifies that CEC and nothing else causes the TV to turn on or change active source.
#>

param
(
    [Parameter(Mandatory=$True)]
    [Int]
    # Number of iterations
    $Iterations,
    
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the test target
    $TestAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the test target
    $TestTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the remote control
    $RemoteControlAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the remote control
    $RemoteControlTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The IP Address of the power source for the HDMI splitter
    $HDMIPowerIP,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The offset number of the power source for the HDMI splitter
    $HDMIPowerNum,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the microphone server
    $ServerAddress,
    
    [Parameter(Mandatory=$True)]
    [Boolean]
    # false: TV turns off
    # true: TV stays on
    $PowerOffTV
    
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# Note: Most of the above parameters are used in Common.ps1
. "${scriptDirectoryPath}\Common.ps1"

Initialize -initRemoteControl $True

Write-Host "Powering off test machine..."
& $ControlTargetExe power-off -t $TestAddress > $null

Get-TV-Focus-With-Remote

if ($PowerOffTV)
{
    Power-Off-TV-With-Remote($true)
}

for ($i = 1; $i -le $Iterations; ++$i)
{
    Write-Host "=== Iteration $i ==="
    
    $attemptCounter = 1
    
    while (1)
    {
    
#################################################################
        # Disconnect the HDMI and then reconnect it
        & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" "${HDMIPowerNum}off" status
        
        Start-Sleep -s 5
    
#################################################################
        # Verify that the system can't communicate with the TV
        Write-Host "(attempting OneTouchPlay when disconnected)"
        $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\RemoteControl\Develop\RemoteControl.nsp" -t $TestAddress -- -OneTouchPlay
        Write-Host $ret
        if ($ret -contains "Operation returned 4")
        {
            break
        }
        
        # Only try up to 10 times before giving up
        # At this point, it likely means the connection can't be cut right now
        if ($attemptCounter -ge 10)
        {
            & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" status
            Cleanup
            Write-Error "*** Able to communicate with TV even when not connected (see SIGLO-68819) ***"
            Write-Error "Aborting"
            exit 1
        }
        
        $attemptCounter = $attemptCounter + 1
    }
    
#################################################################
    # Reconnect the HDMI
    & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" "${HDMIPowerNum}on" status
        
#################################################################
    # Verify this had no impact on the TV
    Write-Host "Playing tone..."
    & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\PlayTone\Develop\PlayTone.nsp" --no-wait -t $TestAddress -- -frequency $TestTone -amplitude $TestAmplitude > $null
    
    Write-Host "Listening for tone..."
    
    $attemptCounter = 1
    do
    {
        Write-Host "- Attempt #${attemptCounter} -"
        
        Start-Sleep -s 1

        # Nothing is supposed to happen, so the TV may or may not be on
        if ($PowerOffTV)
        {
            # Fail if we hear the test tone or remote control tone
            & $CheckToneExe -activeTone $TestTone -ipAddr $ServerAddress > $null
            if ($lastexitcode -eq 0)
            {
                & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" status
                Cleanup
                Write-Error "xxx TEST MACHINE TURNED ON TV xxx"
                
                Write-Error "Aborting"
                exit 1
            }
            
            & $CheckToneExe -activeTone $RemoteControlTone -ipAddr $ServerAddress > $null
        }
        else
        {
            # Fail if we don't hear the remote control tone
            & $CheckToneExe -silentTone $RemoteControlTone -ipAddr $ServerAddress > $null
        }
        
        if ($lastexitcode -eq 0)
        {
            & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" status
            Cleanup
            Write-Error "xxx TEST MACHINE TURNED ON TV xxx"
            
            Write-Error "Aborting"
            exit 1
        }
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($attemptCounter -lt 30)
    
    Write-Host "Replaying tone on remote control..."
    & $RunOnTargetExe "${scriptDirectoryPath}\PlayTone.nsp" --no-wait -t $RemoteControlAddress -- -frequency $RemoteControlTone -amplitude $RemoteControlAmplitude > $null
    
    Write-Host "Powering off test machine..."
    & $ControlTargetExe power-off -t $TestAddress > $null
}

Cleanup

Write-Host "Passed!"