<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Resolution Switching Test

    .DESCRIPTION
        Verifies that the resolution switches properly with a given transition
#>

param
(
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the target
    $TargetAddress,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The transition to run
    $TransitionTest
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

. "${scriptDirectoryPath}\Common.ps1"
. "${scriptDirectoryPath}\${TransitionTest}"

function Test-Transition([int]$bank, [int]$resolution, [int]$preResolution, [int]$postResolution)
{
    Write-HDMI-Switcher "#set_edid bank ${bank} input 1"
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${preResolution}_1" "${transitionCounter}p_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${preResolution}_2" "${transitionCounter}q_" $SHORT_WAIT
    
    Do-Transition "${scriptDirectoryPath}" "${outputFile}" "${script:transitionCounter}" "${resolution}" "${preResolution}" "${postResolution}" "${TargetAddress}"
    
    $script:transitionCounter = $script:transitionCounter + 1
}

$transitionCounter = 1
$outputFile = "${scriptDirectoryPath}\out.png"
$startRes = 1080

Init-Transition-Test $scriptDirectoryPath $outputFile $TargetAddress

###############################################################################################

Write-Host "=== ${transitionCounter}: invalid (Auto) ==="
Test-Transition 1 0 480 480

Write-Host "=== ${transitionCounter}: invalid (720p) ==="
Test-Transition 1 720 480 480

Write-Host "=== ${transitionCounter}: invalid (1080p) ==="
Test-Transition 1 1080 480 480

Write-Host "=== ${transitionCounter}: none (Auto) ==="
Test-Transition 2 0 480 480

Write-Host "=== ${transitionCounter}: none (720p) ==="
Test-Transition 2 720 480 480

Write-Host "=== ${transitionCounter}: none (1080p) ==="
Test-Transition 2 1080 480 480

Write-Host "=== ${transitionCounter}: none (480p) ==="
Test-Transition 2 480 480 480

Write-Host "=== ${transitionCounter}: 720p (Auto) ==="
Test-Transition 3 0 480 720

Write-Host "=== ${transitionCounter}: 720p (480p) ==="
Test-Transition 3 480 720 480

Write-Host "=== ${transitionCounter}: 720p (720p) ==="
Test-Transition 3 720 480 720

#################
Write-Host "=== ${transitionCounter}: invalid (480p) ==="
Test-Transition 1 480 480 480

Write-Host "=== ${transitionCounter}: 720p (1080p) ==="
Test-Transition 3 1080 480 720

Write-Host "=== ${transitionCounter}: all (480p) ==="
Test-Transition 5 480 1080 480

Write-Host "=== ${transitionCounter}: 1080p (Auto) ==="
Test-Transition 4 0 480 1080

Write-Host "=== ${transitionCounter}: 1080p (480p) ==="
Test-Transition 4 480 1080 480

Write-Host "=== ${transitionCounter}: 1080p (720p) ==="
Test-Transition 4 720 480 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (480p) ==="
Test-Transition 6 480 480 480

Write-Host "=== ${transitionCounter}: 1080p (1080p) ==="
Test-Transition 4 1080 480 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (480p) ==="
Test-Transition 7 480 480 480

Write-Host "=== ${transitionCounter}: all (Auto) ==="
Test-Transition 5 0 480 1080

#################
Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 720 1080 720

Write-Host "=== ${transitionCounter}: all (1080p) ==="
Test-Transition 5 1080 720 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (Auto) ==="
Test-Transition 6 0 480 480

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (720p) ==="
Test-Transition 6 720 480 480

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (1080p) ==="
Test-Transition 6 1080 480 480

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (Auto) ==="
Test-Transition 7 0 480 480

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (720p) ==="
Test-Transition 7 720 480 480

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (1080p) ==="
Test-Transition 7 1080 480 480

###############################################################################################

Cleanup-Transition-Test $scriptDirectoryPath

###############################################################################################
if ($failure)
{
    Write-Host "FAILURES OCCURRED"
    exit 1
}

Write-Host "Passed!"
exit 0
