﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winsock2.h>
#include <ws2tcpip.h>
#include <windows.h>
#include <cstdlib>
#include <cstdio>

#include "../AudioCaptureServer/ServerConstants.h"

#pragma comment (lib, "Ws2_32.lib")
#pragma comment (lib, "Mswsock.lib")
#pragma comment (lib, "AdvApi32.lib")

SOCKET ConnectSocket(const char* ipAddr)
{
    SOCKET connectSocket = INVALID_SOCKET;

    addrinfo* addrInfo = NULL;
    addrinfo hints;

    int result;

    ZeroMemory(&hints, sizeof(hints));
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = IPPROTO_TCP;

    // Resolve the server address and port
    result = getaddrinfo(ipAddr, s_CommandPort, &hints, &addrInfo);
    if (result != 0)
    {
        printf("getaddrinfo failed with error: %d\n", result);
        return INVALID_SOCKET;
    }

    // Attempt to connect to an address until one succeeds
    for (addrinfo* ptr = addrInfo; ptr != NULL; ptr = ptr->ai_next)
    {
        // Create a socket for connecting to server
        connectSocket = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol);
        if (connectSocket == INVALID_SOCKET)
        {
            printf("socket failed with error: %ld\n", WSAGetLastError());
            return INVALID_SOCKET;
        }

        // Connect to server.
        result = connect(connectSocket, ptr->ai_addr, static_cast<int>(ptr->ai_addrlen));
        if (result != SOCKET_ERROR)
        {
            goto SocketConnected;
        }

        closesocket(connectSocket);
        connectSocket = INVALID_SOCKET;
    }

SocketConnected:
    freeaddrinfo(addrInfo);

    return connectSocket;
}

int main(int argc, char **argv)
{
    const char* ipAddr = NULL;
    char command = COMMAND_TOGGLE_ENABLE;

    // Get the arguments
    for (int i = 1; i < argc; ++i)
    {

#define CHECK_ARG(name)                                     \
if (!_stricmp(argv[i], "-" #name))                          \
{                                                           \
        if (i + 1 >= argc)                                  \
        {                                                   \
            printf("Error: expected additional argument after -" #name "\n");   \
            return -1;                                      \
        }                                                   \
        ++i;                                                \
        name = argv[i];                                     \
        continue;                                           \
}

        CHECK_ARG(ipAddr);

#undef CHECK_ARG

        if (!_stricmp(argv[i], "-disable"))
        {
            command = COMMAND_TOGGLE_DISABLE;
            continue;
        }
    }

    if (!ipAddr)
    {
        printf("Usage:\n");
        printf("toggle_tone.exe -ipAddr ### -tone ### [-tone ###] [-disable]\n");
        printf("        Either enables or disables listening for the listed tones on the server\n");
        printf("    tone:       A tone to check\n");
        printf("    disable:    All tones listed should NOT be checked\n");
        printf("    ipAddr:     The IP address of the server\n");
        return -1;
    }

    WSADATA wsaData;

    int result;

    // Initialize Winsock
    result = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (result != 0)
    {
        printf("WSAStartup failed with error: %d\n", result);
        return -1;
    }

    // Handle each tone
    for (int i = 1; i < argc; ++i)
    {
        if (!_stricmp(argv[i], "-tone"))
        {
            if (i + 1 >= argc)
            {
                printf("Error: expected additional argument after -tone\n");
                return -1;
            }

            SOCKET connectSocket = ConnectSocket(ipAddr);

            if (connectSocket == INVALID_SOCKET)
            {
                printf("Failed to connect socket\n");
                return -1;
            }

            int tone = atoi(argv[i + 1]);
            int size = sizeof(tone) + sizeof(command);

            // The first four bytes are the size in bytes of what's to come
            // The next byte is the command (enable/disable)
            // The next four bytes are the tone

            result = send(connectSocket, reinterpret_cast<char*>(&size), sizeof(size), 0);
            if (result == SOCKET_ERROR)
            {
                printf("send failed with error: %d\n", WSAGetLastError());
                shutdown(connectSocket, SD_BOTH);
                closesocket(connectSocket);
                WSACleanup();
                return -1;
            }

            result = send(connectSocket, &command, sizeof(command), 0);
            if (result == SOCKET_ERROR)
            {
                printf("send failed with error: %d\n", WSAGetLastError());
                shutdown(connectSocket, SD_BOTH);
                closesocket(connectSocket);
                WSACleanup();
                return -1;
            }

            result = send(connectSocket, reinterpret_cast<char*>(&tone), sizeof(tone), 0);
            if (result == SOCKET_ERROR)
            {
                printf("send failed with error: %d\n", WSAGetLastError());
                shutdown(connectSocket, SD_BOTH);
                closesocket(connectSocket);
                WSACleanup();
                return -1;
            }

            printf("Sent tone %d\n", tone);

            shutdown(connectSocket, SD_BOTH);
            closesocket(connectSocket);

            ++i;
        }
    }

    printf("Successfully sent command %d\n", command);
    return 0;
}
