﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/TutorialUtil.h>

#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayEvents.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

static const size_t g_CommandMemorySize = 64 * 1024;
static const size_t g_ControlMemorySize = 64 * 1024;
static const int    g_NumColorBuffers = 2;

static const int    g_NumColors = 3;
static int          g_Frame = 0;

class GettingStartedWithNVN : public TutorialBaseClass
{
    public:
        GettingStartedWithNVN() ;
        virtual ~GettingStartedWithNVN() ;
        virtual void Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow);
        virtual void Shutdown();

        virtual void Draw(uint64_t millisec) ;
        virtual void Resize(int width, int height) ;

    private:
        static void NVNAPIENTRY DebugLayerCallback(
            NVNdebugCallbackSource source,
            NVNdebugCallbackType type,
            int id,
            NVNdebugCallbackSeverity severity,
            const char* message,
            void* pUser
            ) ;

        int UpdateRenderTargets() ;

        NVNdevice         m_Device;
        NVNqueue          m_Queue;

        MemoryPool*       m_pCommandMemoryPool;

        ptrdiff_t         m_CommandPoolOffset;
        void*             m_pControlPool;
        NVNcommandBuffer  m_CommandBuffer;
        NVNcommandHandle  m_CommandHandle[g_NumColors];

        ptrdiff_t         m_RenderTargetCommandPoolOffset;
        void*             m_pRenderTargetControlPool;
        NVNcommandBuffer  m_RenderTargetCommandBuffer;
        NVNcommandHandle  m_RenderTargetCommandHandle;

        NVNtextureBuilder m_RenderTargetBuilder;
        NVNtexture*       m_RenderTargets[g_NumColorBuffers];

        NVNsync           m_CommandBufferSync;

        MemoryPool*       m_pRenderTargetMemoryPool;

        NVNwindow*        m_pWindow;
        NVNwindowBuilder  m_WindowBuilder;
        size_t            m_ColorTargetSize;
};

GettingStartedWithNVN::GettingStartedWithNVN() :
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pWindow(NULL)
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }

    for (int i = 0; i < g_NumColors; ++i)
    {
        m_CommandHandle[i] = NULL;
    }
}

GettingStartedWithNVN::~GettingStartedWithNVN()
{
}

void GettingStartedWithNVN::Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(loader != NULL, "Bootstrap loader function pointer is NULL\n");

    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*loader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*loader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

    int deviceFlags = 0;
#ifdef _DEBUG
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT | NVN_DEVICE_FLAG_DEBUG_SKIP_CALLS_ON_ERROR_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    if(nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

    if(!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    if(!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(
        NULL,
        Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    if(!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    if(!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "nvnSyncInitialize");
    }

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
}

void GettingStartedWithNVN::Shutdown()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_CommandBufferSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if(m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);
    nvnCommandBufferFinalize(&m_CommandBuffer);

    if(m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if(m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i])
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    nvnQueueFinalize(&m_Queue);

    nvnDeviceFinalize(&m_Device);
}

void GettingStartedWithNVN::Resize(int width, int height)
{
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);

    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (!m_RenderTargets[i])
        {
            m_RenderTargets[i] = new NVNtexture;
        }
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

    const int TileSize = 100;

    const float   Red[4] = { 1.0f, 0.0f, 0.0f, 1.0f };
    const float Green[4] = { 0.0f, 1.0f, 0.0f, 1.0f };
    const float  Blue[4] = { 0.0f, 0.0f, 1.0f, 1.0f };

    const float* ColorList[g_NumColors] = { Red, Green, Blue };

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    for (int i = 0; i < g_NumColors; ++i)
    {
        nvnCommandBufferBeginRecording(&m_CommandBuffer);

        for (int y = 0; y < height; y += TileSize)
        {
            for (int x = 0; x < width; x += TileSize)
            {
                int subWidth = (x + TileSize > width ? width - x : TileSize);
                int subHeight = (y + TileSize > height ? height - y : TileSize);

                nvnCommandBufferSetScissor(&m_CommandBuffer, x, y, subWidth, subHeight);
                nvnCommandBufferClearColor(&m_CommandBuffer, 0, ColorList[((x + y) / TileSize + i) % g_NumColors], NVN_CLEAR_COLOR_MASK_RGBA);
            }
        }

        m_CommandHandle[i] = nvnCommandBufferEndRecording(&m_CommandBuffer);
    }
}

void GettingStartedWithNVN::Draw(uint64_t)
{
    int index = UpdateRenderTargets();

    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle[(g_Frame / 30) % g_NumColors]);

    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);
}

int GettingStartedWithNVN::UpdateRenderTargets()
{
    int bufferIndex;

    nvnQueueAcquireTexture(&m_Queue, m_pWindow, &bufferIndex);

    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_RenderTargets[bufferIndex], NULL, NULL, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

    return bufferIndex;
}

void GettingStartedWithNVN::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static GettingStartedWithNVN tut;
    return (&tut);
}

void EndFrameCallbackFunction(bool* pOutExitLoop, const CallbackFunctionType::EndFrameCallbackParameter* param, void*)
{
    g_Frame = param->currentFrame;
}

void SetDisplayMode(nn::settings::system::TvResolution resolution)
{
    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::vi::OpenDisplay( &pDisplay, "External" ) );

    nn::vi::DisplayModeInfo modes[ nn::vi::DisplayModeCountMax ];
    int modeCount = nn::vi::ListDisplayModes( modes, sizeof( modes ) / sizeof( modes[ 0 ] ), pDisplay );

    NN_ASSERT(modeCount > 0);

    nn::vi::DisplayModeInfo mode = modes[0];

    switch (resolution)
    {
    case nn::settings::system::TvResolution_480p:
        mode.width = 720;
        mode.height = 480;
        break;
    case nn::settings::system::TvResolution_720p:
        mode.width = 1280;
        mode.height = 720;
        break;
    case nn::settings::system::TvResolution_1080p:
        mode.width = 1920;
        mode.height = 1080;
        break;
    case nn::settings::system::TvResolution_Auto:
        for ( int i = 1; i < modeCount; ++i )
        {
            if ( modes[i].width > mode.width )
            {
                mode = modes[i];
            }
        }
        break;
    default:
        NN_ABORT("Unexpected resolution passed: %i\n", resolution);
        break;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::SetDisplayMode( pDisplay, &mode ));

    nn::vi::CloseDisplay(pDisplay);
}

void SetUnderscan(int underscan)
{
    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::vi::OpenDisplay( &pDisplay, "External" ) );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::SetDisplayUnderscan(pDisplay, underscan));

    nn::vi::CloseDisplay(pDisplay);
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nn::vi::Initialize();

    nn::settings::system::TvSettings currentTvSettings;
    nn::settings::system::GetTvSettings( &currentTvSettings );

    bool updateDisplay = false;
    bool updateUnderscan = false;
    bool updateNand = false;
    bool noRun = false;

    for(int i = 0; i < argc; ++i)
    {
        if (!strcmp(argv[i], "-resolution"))
        {
            NN_ASSERT(i + 1 < argc);

            switch (atoi(argv[i + 1]))
            {
            case 480:
                currentTvSettings.tvResolution = nn::settings::system::TvResolution_480p;
                break;
            case 720:
                currentTvSettings.tvResolution = nn::settings::system::TvResolution_720p;
                break;
            case 1080:
                currentTvSettings.tvResolution = nn::settings::system::TvResolution_1080p;
                break;
            default:
                currentTvSettings.tvResolution = nn::settings::system::TvResolution_Auto;
                break;
            }
            continue;
        }

        if (!strcmp(argv[i], "-underscan"))
        {
            NN_ASSERT(i + 1 < argc);

            currentTvSettings.tvUnderscan = atoi(argv[i + 1]);
            continue;
        }

        if (!strcmp(argv[i], "-update_display"))
        {
            updateDisplay = true;
            continue;
        }

        if (!strcmp(argv[i], "-update_underscan"))
        {
            updateUnderscan = true;
            continue;
        }

        if (!strcmp(argv[i], "-update_nand"))
        {
            updateNand = true;
            continue;
        }

        if (!strcmp(argv[i], "-no_run"))
        {
            noRun = true;
            continue;
        }
    }

    if (updateNand)
    {
        nn::settings::system::SetTvSettings( currentTvSettings );
    }

    if (updateDisplay)
    {
        SetDisplayMode(static_cast<nn::settings::system::TvResolution>(currentTvSettings.tvResolution));
    }

    if (updateUnderscan)
    {
        SetUnderscan(currentTvSettings.tvUnderscan);
    }

    if (noRun)
    {
        return;
    }

    CallbackFunctionType cbFunctions;
    cbFunctions.endFrameCallbackFunction = EndFrameCallbackFunction;
    cbFunctions.endFrameCallbackUserParam = NULL;
    TutorialRun(&cbFunctions);
}
