﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace TargetUtility
{
    class Target
    {
        public string Name;
        public string Hardware;
        public string Connection;
        public string SerialNumber;
        public IPAddress IpAddress;
        public string Status;

        private Target() { }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (Target)obj;
            return SerialNumber.Equals(value.SerialNumber);
        }

        public override int GetHashCode()
        {
            return SerialNumber.GetHashCode();
        }

        public void Run(string fileName, string runnerArguments = "", string applicationArguments = "")
        {
            var arguments = $@"-v -t ""{Name}"" {runnerArguments} ""{fileName}""";
            if (!string.IsNullOrEmpty(applicationArguments))
            {
                arguments += " -- " + applicationArguments;
            }

            var result = Runner.RunApplication(EnvironmentInfo.RunOnTarget, arguments);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to run the program\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"FileName={fileName}\n" +
                    $"Arguments={arguments}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}" +
                    $"Output=\n{result.StandardOutputString}" +
                    $"Output=\n{result.StandardErrorString}");
            }
        }

        public void ImportSettings(string settings)
        {
            Run(EnvironmentInfo.SettingsManager, "", $"Import {settings}");
        }

        public void DevMenuCommand(string arguments)
        {
            Run(EnvironmentInfo.DevMenuCommand, @"--pattern-failure-exit ""\[FAILURE\]""", arguments);
        }

        public void InstallApplication(string fileName, bool force = false)
        {
            DevMenuCommand($@"application install ""{fileName}""{(force ? " --force" : string.Empty)}");
        }

        public void LaunchApplication(Int64 programId)
        {
            DevMenuCommand($"application launch 0x{programId:X16}");
        }

        public void TerminateApplication()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"terminate -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to terminate application\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void PowerOff()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"power-off -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to power off target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void PowerOn()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"power-on -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to power off target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void Reset()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"reset -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to power off target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void Connect()
        {
            var timeout = TimeSpan.FromMinutes(1);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"connect -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to connect target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void Disconnect()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, $"disconnect -t {SerialNumber}", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to connect target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void Initialize()
        {
            if (EnvironmentInfo.IsRepository)
            {
                InitializeForRepository();
            }
            else
            {
                InitializeForPackage();
            }
        }

        public bool IsFirmwareExisting()
        {
            var firmwareKeyTable = new Dictionary<string, int>()
            {
                { "SDEV_00_01_08",      2 },
                { "SDEV_00_01_09_01",   3 },
                { "SDEV_00_02_00_00",   5 }
            };

            if (!firmwareKeyTable.ContainsKey(Hardware))
            {
                throw new Exception(
                    $"Undefined hardware: {Hardware}");
            }

            var key = firmwareKeyTable[Hardware];
            var initimgPath = $@"Programs\Eris\Outputs\NX-NXFP2-a64\InitialImages\NX-K{key}-Hb-Unsigned-Nand-64G-Internal\Develop\NX-K{key}-Hb-Unsigned-Nand-64G-Internal.initimg";
            var recoveryWriterPath = $@"Programs\Eris\Outputs\NX-NXFP2-a64\SystemImages\QspiBootImages\RecoveryWriter-K{key}-Hb-Unsigned\Develop\RecoveryWriter-K{key}-Hb-Unsigned.qspi.img";

            if (!File.Exists(Path.Combine(EnvironmentInfo.NintendoSdkRoot, initimgPath)))
            {
                return false;
            }

            if (!File.Exists(Path.Combine(EnvironmentInfo.NintendoSdkRoot, recoveryWriterPath)))
            {
                return false;
            }

            return true;
        }

        private void InitializeForRepository()
        {
            var timeout = TimeSpan.FromMinutes(5);

            var result = Runner.RunApplication(
                "PowerShell", $"{EnvironmentInfo.InitializeSdevFull} -AddressPattern {IpAddress} -SkipStopTargetManager", timeout);
            if (!(result.ExitCode == 0 && result.StandardOutputString.Contains("[trace] Match the success exit pattern. : Hello, world.")))
            {
                throw new Exception(
                    $"Failed to initialize target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        private void InitializeForPackage()
        {
            var timeout = TimeSpan.FromMinutes(5);

            var result = Runner.RunApplication(
            EnvironmentInfo.InitializeSdevExe, $"--target {SerialNumber} --keep-targetmanager-alive", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to connect target\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void WriteProperBootConfig()
        {
            var result = Runner.RunApplication(
                "PowerShell", $"{EnvironmentInfo.WriteBootConfig} -AddressPattern {IpAddress}", TimeSpan.FromSeconds(120));
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to write boot config\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        public void InvalidateBootConfig()
        {
            var result = Runner.RunApplication(
                "PowerShell", $"{EnvironmentInfo.WriteInvalidBootConfig} -AddressPattern {IpAddress}", TimeSpan.FromSeconds(120));
            if (result.ExitCode != 0)
            {
                throw new Exception(
                    $"Failed to invalidate boot config\n" +
                    $"SerialNumber={SerialNumber}\n" +
                    $"ExitCode={result.ExitCode}\n" +
                    $"Output=\n{result.MixedOutputString}");
            }
        }

        private CancellationTokenSource UartLoggingCancellationTokenSource = null;
        private Task UartLoggingTask = null;
        private object UartLoggingLock = new object();


        public void StartUartLogging()
        {
            lock (UartLoggingLock)
            {
                if (UartLoggingTask != null)
                {
                    throw new InvalidOperationException(
                        $"Uart logging is already started\nTargetName={Name}");
                }

                var logFile = Path.Combine(EnvironmentInfo.LogDirectory,
                    $"Uart_{SerialNumber}_{DateTime.Now.ToString("yyyy-MM-dd_HH-mm-ss")}.log");

                const int uartPort = 10023;
                var client = new TcpClient();
                client.Connect(IpAddress, uartPort);
                UartLoggingTask = Task.Factory.StartNew(() =>
                {
                    UartLoggingCancellationTokenSource = new CancellationTokenSource();
                    using (var sw = new StreamWriter(logFile, false, new UTF8Encoding(false)))
                    using (var sr = new StreamReader(new BufferedStream(client.GetStream())))
                    using (UartLoggingCancellationTokenSource.Token.Register(() => sr.Close()))
                    {
                        try
                        {
                            while (!sr.EndOfStream)
                            {
                                sw.WriteLine(sr.ReadLine());
                            }
                        }
                        catch
                        {
                        }
                    }
                    client.Close();
                });
            }
        }

        public void StopUartLogging()
        {
            lock (UartLoggingLock)
            {
                if (UartLoggingTask == null)
                {
                    throw new InvalidOperationException(
                        $"Uart logging is not started\nTargetName={Name}");
                }

                UartLoggingCancellationTokenSource.Cancel();
                UartLoggingTask.Wait();

                UartLoggingCancellationTokenSource.Dispose();
                UartLoggingCancellationTokenSource = null;
                UartLoggingTask.Dispose();
                UartLoggingTask = null;
            }
        }

        public static Target[] Enumerate()
        {
            var timeout = TimeSpan.FromSeconds(10);

            var result = Runner.RunApplication(
                EnvironmentInfo.ControlTarget, "detect-target --xml", timeout);
            if (result.ExitCode != 0)
            {
                throw new Exception($"Failed to detect target\nOutput=\n{result.MixedOutputString}");
            }

            return
                XElement.Parse(result.StandardOutputString).Elements("Target").Select(target =>
                    new Target()
                    {
                        Name = target.Element("Name").Value,
                        Hardware = target.Element("Hardware").Value,
                        Connection = target.Element("Connection").Value,
                        SerialNumber = target.Element("SerialNumber").Value,
                        IpAddress = new IPAddress(target.Element("IPAddress").Value.Split('.').Select(
                            stringValue => byte.Parse(stringValue)).ToArray()),
                        Status = target.Element("Status").Value,
                    }).ToArray();
        }

        public static Target[] Enumerate(string addressPattern)
        {
            return Enumerate().Where(
                target => Regex.Match(target.IpAddress.ToString(), addressPattern).Success).ToArray();
        }
    }
}
