﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nnt/nntest.h>
#include <nn/diag.h>
#include <cstring>
#include <nn/util/util_FormatString.h>
#include <nn/init.h>

namespace {

NN_ALIGNAS(4096) char g_mallocBuffer[1024 * 1024];

}

extern "C" void nninitStartup()
{
    // malloc 用の領域を確保
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_mallocBuffer), sizeof(g_mallocBuffer) );
}

namespace
{

nn::diag::LogMetaData g_LogMetaData;
const size_t LogMessageBufferLength = 512;
char g_LogMessageBuffer[LogMessageBufferLength];

} // anonymous

extern "C"
{
// test_LinkageC.c にある C 言語リンケージ版ログ出力マクロ呼ぶ関数
void CallNncLog(void);
void CallNncVlog(void);
void CallNncSdkLog(void);
void CallNncSdkVlog(void);

// test_LinkageC.c に定義されたテストの期待値
extern const char* ExpectFileName;
extern const int ExpectLineNumber;
extern const char* ExpectMessage;
extern const int ExpectLineNumberByVlog;
extern const char* ExpectMessageByVlog;
extern const int ExpectLineNumberBySdkLog;
extern const char* ExpectMessageBySdkLog;
extern const int ExpectLineNumberBySdkVlog;
extern const char* ExpectMessageBySdkVlog;

}

TEST(LinkageC, Interface)
{
    // テスト用オブザーバーの登録
    nn::diag::LogObserverHolder logObserverHolder;
    nn::diag::InitializeLogObserverHolder(
        &logObserverHolder,
        []( const nn::diag::LogMetaData& metaData,
            const nn::diag::LogBody& body,
            void* /*argument*/)
        {
            g_LogMetaData = metaData;
            const size_t writeBytes = std::min(LogMessageBufferLength, body.messageBytes);
            std::memcpy(
                g_LogMessageBuffer, body.message, writeBytes);
            g_LogMessageBuffer[writeBytes] = '\0';
        },
        NULL);
    nn::diag::RegisterLogObserver(&logObserverHolder);

    // NNC_LOG() のテスト
    CallNncLog();
    EXPECT_EQ(strcmp(g_LogMetaData.sourceInfo.fileName, ExpectFileName), 0);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, ExpectLineNumber);
    EXPECT_EQ(strcmp(g_LogMessageBuffer, ExpectMessage), 0);

    // NNC_VLOG() のテスト
    CallNncVlog();
    EXPECT_EQ(strcmp(g_LogMetaData.sourceInfo.fileName, ExpectFileName), 0);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, ExpectLineNumberByVlog);
    EXPECT_EQ(strcmp(g_LogMessageBuffer, ExpectMessageByVlog), 0);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    // NNC_SDK_LOG() のテスト
    CallNncSdkLog();
    EXPECT_EQ(strcmp(g_LogMetaData.sourceInfo.fileName, ExpectFileName), 0);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, ExpectLineNumberBySdkLog);
    EXPECT_EQ(strcmp(g_LogMessageBuffer, ExpectMessageBySdkLog), 0);

    // NNC_SDK_VLOG() のテスト
    CallNncSdkVlog();
    EXPECT_EQ(strcmp(g_LogMetaData.sourceInfo.fileName, ExpectFileName), 0);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, ExpectLineNumberBySdkVlog);
    EXPECT_EQ(strcmp(g_LogMessageBuffer, ExpectMessageBySdkVlog), 0);

#endif // #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    // テスト用オブザーバーの登録解除
    nn::diag::UnregisterLogObserver(&logObserverHolder);
}
