﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <cstring>
#include <mutex>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include "../../../../../Programs/Eris/Sources/Libraries/lm/impl/lm_LogBuffer.h"

namespace {

struct PushThreadParameter {
    nn::lm::impl::LogBuffer* pLogBuffer;
    const char* pushString;
    size_t pushCount;
    nn::TimeSpan sleepTime;
    int threadPriority;
};

void PushThreadFunction(void* arg)
{
    auto pParam = reinterpret_cast<PushThreadParameter*>(arg);

    for (size_t i = 0; i < pParam->pushCount; i++)
    {
        pParam->pLogBuffer->Push(pParam->pushString, std::strlen(pParam->pushString));
        nn::os::SleepThread(pParam->sleepTime);
    }
}

struct FlushThreadParameter {
    nn::lm::impl::LogBuffer* pLogBuffer;
    nn::TimeSpan sleepTime;
    int threadPriority;
};

volatile std::atomic_bool g_FlushThreadRunning;

void FlushThreadFunction(void* arg)
{
    auto pParam = reinterpret_cast<FlushThreadParameter*>(arg);

    while (g_FlushThreadRunning)
    {
        pParam->pLogBuffer->TryFlush();
        nn::os::SleepThread(pParam->sleepTime);
    }
}

size_t CountString(const char* begin, const char* end, const char* target)
{
    size_t count = 0u;
    size_t targetLen = std::strlen(target);

    while (begin < end)
    {
        const char* result = std::strstr(begin, target);
        if (result == NULL)
        {
            break;
        }

        begin = result + targetLen;
        count++;
    }

    return count;
}

char g_DestBuffer[64 << 10];
size_t g_DestBufferCount = 0;

bool TestFlushFunction(const uint8_t* message, size_t messageSize)
{
    NN_FUNCTION_LOCAL_STATIC(nn::os::Mutex, m_Mutex, (false));

    size_t offset;
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        offset = g_DestBufferCount;
        g_DestBufferCount += messageSize;
    }

    memcpy(g_DestBuffer + offset, message, messageSize);

    return true;
}

} // anonymous

// 書き込みとフラッシュのスレッドを 4 つずつ立てて、並列してアクセスするテスト
TEST(LogBufferTest, MultiThread)
{
    const char* fooString = "Foo";
    const char* barBazString = "BarBaz";
    const char* numbersString = "1234567890";
    const char* symbolsString = "!\"#$%%&'()-=^~\\|@`[]{};+:*,.<>/?_";

    static uint8_t s_LogBufferStorage[2 << 10]; // 2 KiB
    nn::lm::impl::LogBuffer s_LogBuffer(s_LogBufferStorage, sizeof(s_LogBufferStorage), TestFlushFunction);

    PushThreadParameter pushThreadParameters[] = {
        { &s_LogBuffer, fooString,     1000, nn::TimeSpan::FromMilliSeconds(1), nn::os::DefaultThreadPriority },
        { &s_LogBuffer, barBazString,  1000, nn::TimeSpan::FromMilliSeconds(2), nn::os::DefaultThreadPriority - 1  },
        { &s_LogBuffer, numbersString, 1000, nn::TimeSpan::FromMilliSeconds(3), nn::os::DefaultThreadPriority - 2  },
        { &s_LogBuffer, symbolsString, 1000, nn::TimeSpan::FromMilliSeconds(4), nn::os::DefaultThreadPriority + 1  }
    };
    const int pushThreadCount = sizeof(pushThreadParameters) / sizeof(pushThreadParameters[0]);
    nn::os::ThreadType pushThread[pushThreadCount];
    NN_ALIGNAS(4096) static uint8_t s_PushThreadStack[pushThreadCount][4 << 10];

    FlushThreadParameter flushThreadParameters[] = {
        { &s_LogBuffer, nn::TimeSpan::FromMilliSeconds(1), nn::os::DefaultThreadPriority },
        { &s_LogBuffer, nn::TimeSpan::FromMilliSeconds(2), nn::os::DefaultThreadPriority - 1 },
        { &s_LogBuffer, nn::TimeSpan::FromMilliSeconds(3), nn::os::DefaultThreadPriority - 2 },
        { &s_LogBuffer, nn::TimeSpan::FromMilliSeconds(4), nn::os::DefaultThreadPriority + 1 }
    };
    const int flushThreadCount = sizeof(flushThreadParameters) / sizeof(flushThreadParameters[0]);
    nn::os::ThreadType flushThread[flushThreadCount];
    NN_ALIGNAS(4096) static uint8_t s_FlushThreadStack[flushThreadCount][4 << 10];

    for (auto i = 0; i < pushThreadCount; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &pushThread[i], PushThreadFunction, &pushThreadParameters[i], s_PushThreadStack[i], sizeof(s_PushThreadStack[i]), pushThreadParameters[i].threadPriority));
    }

    for (auto i = 0; i < flushThreadCount; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &flushThread[i], FlushThreadFunction, &flushThreadParameters[i], s_FlushThreadStack[i], sizeof(s_FlushThreadStack[i]), flushThreadParameters[i].threadPriority));
    }

    for (auto& thread : pushThread)
    {
        nn::os::StartThread(&thread);
    }

    g_FlushThreadRunning = true;

    for (auto& thread : flushThread)
    {
        nn::os::StartThread(&thread);
    }

    for (auto& thread : pushThread)
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }

    g_FlushThreadRunning = false;

    for (auto& thread : flushThread)
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }

    g_DestBuffer[g_DestBufferCount] = '\0';

    EXPECT_EQ(1000, CountString(g_DestBuffer, g_DestBuffer + g_DestBufferCount, fooString));
    EXPECT_EQ(1000, CountString(g_DestBuffer, g_DestBuffer + g_DestBufferCount, barBazString));
    EXPECT_EQ(1000, CountString(g_DestBuffer, g_DestBuffer + g_DestBufferCount, numbersString));
    EXPECT_EQ(1000, CountString(g_DestBuffer, g_DestBuffer + g_DestBufferCount, symbolsString));
}
