﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/diag_AssertionFailureHandler.h>

#if NN_BUILD_CONFIG_TOOLCHAIN_VC
// VC において、「C4127: 条件式が定数です」の警告を無効にする。
// 典型的な ASSERT マクロのテストでは、マクロの引数に定数式を与えるため、
// ASSERT マクロ内部で生成される条件式が定数になる。
//
// 通常は不要だと思われるので、ASSERT のテストでのみ定義する。
#define NNT_DIAG_DISABLE_WARNING_STATIC_CONDITION __pragma( warning( disable:4127 ) )
#else
#define NNT_DIAG_DISABLE_WARNING_STATIC_CONDITION
#endif


// AssertionFailureOperation_Continue を返して処理を続行する Assertion 失敗ハンドラ
nn::diag::AssertionFailureOperation ContinuingAssertionFailureHandler(
    const nn::diag::AssertionInfo& assertionInfo);

// // AssertionFailureOperation_Abort を返して処理を停止する Assertion 失敗ハンドラ
nn::diag::AssertionFailureOperation AbortingAssertionFailureHandler(
    const nn::diag::AssertionInfo& assertionInfo);

// IsCustomHandlerCalled が false を返すようにします。
void ClearCustomHandlerCalled();

// ClearCustomHandlerCalled が呼ばれた後、ContinuingAssertionFailureHandler が呼ばれたかどうかを返します。
// 一度 true になったら、ClearCustomHandlerCalled を呼ぶまで true のままです。
bool IsCustomHandlerCalled();

// 最後に ContinuingAssertionFailureHandler が呼ばれたときの AssertionType を返します。
nn::diag::AssertionType GetLastAssertionType();

class ScopedSetAssertionFailureHandler
{
public:
    explicit ScopedSetAssertionFailureHandler(nn::diag::AssertionFailureHandler handler);

    // 元に戻すインターフェースは提供していないので、Assertion 失敗した時に Abort するように
    // AbortingAssertionFailureHandler が Assertion 失敗ハンドラに設定されます。
    ~ScopedSetAssertionFailureHandler();
};

#define EXPECT_ASSERTION_HANDLER_CALLED(expectedHandlerCalled, statement, message) \
    do \
    { \
        ScopedSetAssertionFailureHandler setContinuingHandler(ContinuingAssertionFailureHandler); \
        ClearCustomHandlerCalled(); \
        statement; \
        EXPECT_EQ(expectedHandlerCalled, IsCustomHandlerCalled()) << message; \
    } while(0)

#define EXPECT_ASSERTION_SUCCESS_IMPL(statement, statementStringized) \
    EXPECT_ASSERTION_HANDLER_CALLED(false, statement, statementStringized " must succeed")

#define EXPECT_ASSERTION_FAILURE_IMPL(statement, statementStringized) \
    EXPECT_ASSERTION_HANDLER_CALLED(true, statement, statementStringized " must fail")
