﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/diag_AbortObserver.h>
#include <nn/util/util_FormatString.h>

namespace {

const char* g_ExpectMessage;
nn::diag::AbortInfo g_ExpectAbortInfo;

bool CheckAbortInfo(const nn::diag::AbortInfo& info)
{
    static const int MessageBufferLength = 512;
    static char s_MessageBuffer[MessageBufferLength];

    nn::util::VSNPrintf(s_MessageBuffer, MessageBufferLength,
                            info.message->format, *info.message->args);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    return (g_ExpectAbortInfo.abortReason ==       info.abortReason   &&
            g_ExpectAbortInfo.lineNumber  ==       info.lineNumber    &&
            strcmp(g_ExpectAbortInfo.condition,    info.condition)    == 0 &&
            strcmp(g_ExpectMessage,                s_MessageBuffer)   == 0 &&
            strcmp(g_ExpectAbortInfo.functionName, info.functionName) == 0 &&
            strcmp(g_ExpectAbortInfo.fileName,     info.fileName)     == 0);
#else
    return (g_ExpectAbortInfo.abortReason == info.abortReason &&
            0 == info.lineNumber &&
            strcmp("", info.condition)    == 0 &&
            strcmp("", s_MessageBuffer)   == 0 &&
            strcmp("", info.functionName) == 0 &&
            strcmp("", info.fileName)     == 0);
#endif
}

void AbortObserver0(const nn::diag::AbortInfo& info)
{
    // TORIAEZU: 実機向け SDK は標準エラー出力へのログ出力に対応していない
    fprintf(stderr, "0:%s ", CheckAbortInfo(info) ? "OK" : "NG");
}

void AbortObserver1(const nn::diag::AbortInfo& info)
{
    fprintf(stderr, "1:%s ", CheckAbortInfo(info) ? "OK" : "NG");
}

void AbortObserver2(const nn::diag::AbortInfo& info)
{
    fprintf(stderr, "2:%s ", CheckAbortInfo(info) ? "OK" : "NG");
}

void AbortObserver3(const nn::diag::AbortInfo& info)
{
    fprintf(stderr, "3:%s ", CheckAbortInfo(info) ? "OK" : "NG");
}

// ０と１は static な場所にホルダを置く
nn::diag::AbortObserverHolder g_AbortObserverHolder0;
nn::diag::AbortObserverHolder g_AbortObserverHolder1;

} // anonymous

// 適切な AbortInfo が設定されているか
TEST(AbortObserverTest, AbortInfo)
{
    nn::diag::InitializeAbortObserverHolder(&g_AbortObserverHolder0, AbortObserver0);
    nn::diag::RegisterAbortObserver(&g_AbortObserverHolder0);

    g_ExpectAbortInfo.fileName = __FILE__;
    g_ExpectAbortInfo.functionName = NN_CURRENT_FUNCTION_NAME;

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_Abort;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "NN_ABORT";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "0:OK");

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_Abort;
    g_ExpectMessage = "int:123, string=str\n";
    g_ExpectAbortInfo.condition = "NN_ABORT";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT("int:%d, string=%s\n", 123, "str"), "0:OK");

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_Abort;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "false";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT_UNLESS(false), "0:OK");

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_UnexpectedDefault;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "unexpected value passed to switch statement";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_UNEXPECTED_DEFAULT, "0:OK");

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_SdkAssert;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "false";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_SDK_ASSERT(false), "0:OK");

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_SdkRequires;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "false";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_SDK_REQUIRES(false), "0:OK");

#endif // defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_UserAssert;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "false";
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ASSERT(false), "0:OK");

    nn::diag::UnregisterAbortObserver(&g_AbortObserverHolder0);
}

// 意図したオブザーバが呼ばれるか
TEST(AbortObserverTest, Registration)
{
    // ２と３はスタックにホルダを置く
    nn::diag::AbortObserverHolder abortObserverHolder2;
    nn::diag::AbortObserverHolder abortObserverHolder3;

    nn::diag::InitializeAbortObserverHolder(&g_AbortObserverHolder0, AbortObserver0);
    nn::diag::InitializeAbortObserverHolder(&g_AbortObserverHolder1, AbortObserver1);
    nn::diag::InitializeAbortObserverHolder(&abortObserverHolder2, AbortObserver2);
    nn::diag::InitializeAbortObserverHolder(&abortObserverHolder3, AbortObserver3);

    // AbortInfo の期待値をセット
    g_ExpectAbortInfo.fileName = __FILE__;
    g_ExpectAbortInfo.functionName = NN_CURRENT_FUNCTION_NAME;
    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_Abort;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "NN_ABORT";

    // すべて追加する
    nn::diag::RegisterAbortObserver(&g_AbortObserverHolder0);
    nn::diag::RegisterAbortObserver(&g_AbortObserverHolder1);
    nn::diag::RegisterAbortObserver(&abortObserverHolder2);
    nn::diag::RegisterAbortObserver(&abortObserverHolder3);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1; // 行番号の期待値は毎回セットしないといけない
    EXPECT_DEATH(NN_ABORT(), "0:OK 1:OK 2:OK 3:OK");

    // 途中のオブザーバを取り外す
    nn::diag::UnregisterAbortObserver(&g_AbortObserverHolder1);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "0:OK 2:OK 3:OK");

    // 末尾のオブザーバを取り外す
    nn::diag::UnregisterAbortObserver(&abortObserverHolder3);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "0:OK 2:OK");

    // 先頭のオブザーバを取り外す
    nn::diag::UnregisterAbortObserver(&g_AbortObserverHolder0);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "2:OK");

    // 唯一のオブザーバを取り外す
    nn::diag::UnregisterAbortObserver(&abortObserverHolder2);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "");

    // 再度すべて追加する
    nn::diag::RegisterAbortObserver(&g_AbortObserverHolder0);
    nn::diag::RegisterAbortObserver(&g_AbortObserverHolder1);
    nn::diag::RegisterAbortObserver(&abortObserverHolder2);
    nn::diag::RegisterAbortObserver(&abortObserverHolder3);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "0:OK 1:OK 2:OK 3:OK");

    // すべて削除する
    nn::diag::UnregisterAbortObserver(&g_AbortObserverHolder0);
    nn::diag::UnregisterAbortObserver(&g_AbortObserverHolder1);
    nn::diag::UnregisterAbortObserver(&abortObserverHolder2);
    nn::diag::UnregisterAbortObserver(&abortObserverHolder3);

    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "");
}

// デフォルトオブザーバはテスト用に細工するわけにいかないので、
// EnableDefaultAbortObserver() の自動テストは悩みどころ
// とりあえず、目でログを見て有効化／無効化できているかは確認している
TEST(AbortObserverTest, EnableDefaultAbortObserver)
{
    // AbortInfo の期待値をセット
    g_ExpectAbortInfo.fileName = __FILE__;
    g_ExpectAbortInfo.functionName = NN_CURRENT_FUNCTION_NAME;
    g_ExpectAbortInfo.abortReason = nn::diag::AbortReason_Abort;
    g_ExpectMessage = "";
    g_ExpectAbortInfo.condition = "NN_ABORT";

    // 目でデフォルトオブザーバのログが出力されることを確認
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "");

    // 目でデフォルトオブザーバのログが出力されないことを確認
    nn::diag::EnableDefaultAbortObserver(false);
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "");

    // 目でデフォルトオブザーバのログが出力されることを確認
    nn::diag::EnableDefaultAbortObserver(true);
    g_ExpectAbortInfo.lineNumber = __LINE__ + 1;
    EXPECT_DEATH(NN_ABORT(), "");
}
