﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief   FdtReader クラスの単体テスト
 * @detail  nn::dt::server::FdtReader クラスの単体テストです。
 *          API テストを補助する目的で実装しています。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/dt/dt_Result.h>
#include <nn/dt/dt_ResultPrivate.h>
#include <nn/dt/dt_Types.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include "../../../../../Programs/Chris/Sources/Libraries/dt/server/dt_FdtReader.h"

// Result に対する ASSERT_TRUE 的なもの
#define NNT_DT_ASSERT_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { NN_LOG("%s(%d): Result(module:%d / description:%d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(_r.IsSuccess());} } while (NN_STATIC_CONDITION(false))

#define NNT_DT_ASSERT_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!(_r <= exp)) { NN_LOG("%s(%d): Result(module:%d / description:%d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(_r <= exp);} } while (NN_STATIC_CONDITION(false))

namespace {
#ifdef NN_BUILD_CONFIG_OS_WIN
    const size_t MaxFdtSize = 1024 * 1024;
    nn::Bit8 g_Fdt[MaxFdtSize];
    size_t g_FdtSize;
#endif
}

// 本来は Horizon on Tegra でしか使われない FdtReader クラス内のロジックを、Windows 環境でテスト
TEST(FdtReader, HorizonOnTegra)
{
    nn::Result result;
    nn::dt::Node uartNode, cpuNode;
    nn::dt::server::FdtReader fdtReader;

    size_t count;

    // 準備
    fdtReader.Initialize(g_Fdt, g_FdtSize);

    result = fdtReader.FindNodeByPath(&uartNode, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = fdtReader.FindNodeByPath(&cpuNode, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = fdtReader.GetInterruptCount(&count, &uartNode);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(3, count);

    // 異常系
    result = fdtReader.GetInterruptCount(&count, &cpuNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);
}

// 無効な FDT のテスト
TEST(FdtReader, InvalidFdt)
{
    char invalidFdt[] = "Invalid FDT.";

    nn::Result result;
    nn::dt::server::FdtReader fdtReader;

    fdtReader.Initialize(reinterpret_cast<nn::Bit8*>(invalidFdt), sizeof(invalidFdt));

    // ユーザーが最初に呼びそうな API に絞ってテスト
    nn::dt::Node node;
    size_t count;

    result = fdtReader.FindNodeByPath(&node, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultLibFdtErrorBadMagic(), result);

    result = fdtReader.GetCompatibleNodeCount(&count, "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultLibFdtErrorBadMagic(), result);
}

//-----------------------------------------------------------------------------
// テスト用の FDT を読み込み、API の動作を確認する
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    FILE* fp = fopen(NNT_DETAIL_FDT_PATH, "rb");
    if (fp == NULL)
    {
        NN_LOG("Failed to open fdt file. (%s)\n", NNT_DETAIL_FDT_PATH);
        exit(1);
    }

    g_FdtSize = fread(g_Fdt, sizeof(g_Fdt[0]), MaxFdtSize, fp);
    fclose(fp);
#else
    #error "Unsupported platform."
#endif

    // Start gtest
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
