﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

#include "testDt_Common.h"

namespace nnt { namespace dt {

TEST(RegisterApi, Single)
{
    nn::Result result;
    nn::dt::Node cpu1, uart, i2c;
    uint16_t value16;
    uint32_t value32;
    uint64_t value64;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系 (address)
    result = nn::dt::GetRegisterAddress(&value32, &cpu1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x1, value32);

    result = nn::dt::GetRegisterAddress(&value64, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x180000000ull, value64);

    // 正常系 (size)
    result = nn::dt::GetRegisterSize(&value32, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x4000, value32);

    // 異常系 (address)
    result = nn::dt::GetRegisterAddress(&value32, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterAddress(&value64, &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterAddress(&value16, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterAddress(&value64, &i2c);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);

    // 異常系 (size)
    result = nn::dt::GetRegisterSize(&value32, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterSize(&value32, &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterSize(&value64, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterSize(&value32, &i2c);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);
}

TEST(RegisterApi, Array)
{
    nn::Result result;
    nn::dt::Node cpu1, i2c;
    uint32_t array32[3];
    uint64_t array64[3];

    // 準備
    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系 (address)
    result = nn::dt::GetRegisterAddressArray(array32, 1, &cpu1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x1, array32[0]);

    result = nn::dt::GetRegisterAddressArray(array64, 3, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x180004000ull, array64[0]);
    ASSERT_EQ(0x010000000ull, array64[1]);
    ASSERT_EQ(0x020000000ull, array64[2]);

    // 正常系 (size)
    result = nn::dt::GetRegisterSizeArray(array32, 3, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x4000ull, array32[0]);
    ASSERT_EQ(0x1000ull, array32[1]);
    ASSERT_EQ(0x1000ull, array32[2]);

    // 異常系 (address)
    result = nn::dt::GetRegisterAddressArray(array32, sizeof(array32) / sizeof(array32[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterAddressArray(array64, sizeof(array64) / sizeof(array64[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterAddressArray(array32, 3, &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);

    // 異常系 (size)
    result = nn::dt::GetRegisterSizeArray(array32, sizeof(array32) / sizeof(array32[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterSizeArray(array64, sizeof(array64) / sizeof(array64[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetRegisterSizeArray(array32, 2, &i2c);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);
}

TEST(RegisterApi, List)
{
    nn::Result result;
    nn::dt::Node cpu1, i2c;
    int count;
    uint32_t list32[3];
    uint64_t list64[3];

    // 準備
    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系 (address)
    result = nn::dt::GetRegisterAddressList(list32, &count, sizeof(list32) / sizeof(list32[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x1, list32[0]);
    ASSERT_EQ(1, count);

    result = nn::dt::GetRegisterAddressList(list64, &count, sizeof(list64) / sizeof(list64[0]), &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x180004000ull, list64[0]);
    ASSERT_EQ(0x010000000ull, list64[1]);
    ASSERT_EQ(0x020000000ull, list64[2]);
    ASSERT_EQ(3, count);

    result = nn::dt::GetRegisterAddressList(list64, &count, 1, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x180004000ull, list64[0]);
    ASSERT_EQ(1, count);

    // 正常系 (size)
    result = nn::dt::GetRegisterSizeList(list32, &count, sizeof(list32) / sizeof(list32[0]), &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x4000ull, list32[0]);
    ASSERT_EQ(0x1000ull, list32[1]);
    ASSERT_EQ(0x1000ull, list32[2]);
    ASSERT_EQ(3, count);

    result = nn::dt::GetRegisterSizeList(list32, &count, 0, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0, count);

    // 異常系 (address)
    result = nn::dt::GetRegisterAddressList(list32, nullptr, sizeof(list32) / sizeof(list32[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterAddressList(list32, &count, sizeof(list32) / sizeof(list32[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterAddressList(list64, &count, sizeof(list64) / sizeof(list64[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // 異常系 (size)
    result = nn::dt::GetRegisterSizeList(list32, nullptr, sizeof(list32) / sizeof(list32[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterSizeList(list32, &count, sizeof(list32) / sizeof(list32[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetRegisterSizeList(list64, &count, sizeof(list64) / sizeof(list64[0]), &cpu1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);
}

}}
