﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalApi.h>
#include <nn/dt/server/dt_Server.h>

#include <nn/fs.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

// #define USE_HIPC 1

namespace {
    const size_t MaxFdtSize = 1024 * 1024;
    nn::Bit8 g_Fdt[MaxFdtSize];
}

//-----------------------------------------------------------------------------
// テスト用の FDT を読み込み、API の動作を確認する
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, NNT_DETAIL_FDT_PATH, nn::fs::OpenMode_Read));

    int64_t fdtSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fdtSize, fileHandle));
    if (fdtSize < 0)
    {
        NN_ABORT("nn::fs::GetFileSize() returns invalid value.\n");
    }
    if (fdtSize > MaxFdtSize)
    {
        NN_ABORT("FDT is too large.\n");
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, g_Fdt, static_cast<size_t>(fdtSize)));

    nn::fs::CloseFile(fileHandle);

    // Start libnn_dtServer
    nn::dt::server::InitializeForDebug(g_Fdt, static_cast<size_t>(fdtSize));

#if defined(USE_HIPC)
    nn::dt::server::RegisterHipcServer();

    const size_t IpcServerThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t ipcServerthreadStack[IpcServerThreadStackSize];
    nn::os::ThreadType ipcServerThread;

    nn::os::CreateThread(&ipcServerThread,
        [](void*)
        {
            nn::dt::server::StartHipcServer();
            nn::dt::server::LoopHipcServer();
        }
        , nullptr, ipcServerthreadStack, IpcServerThreadStackSize, nn::os::DefaultThreadPriority);
    nn::os::StartThread(&ipcServerThread);
#else
    nn::dt::detail::InitializeWith(nn::dt::server::GetServiceObject());
#endif

    // Start gtest
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
