﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  DeviceCode の API のテスト
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/nn_DeviceCode.h>
#include <nn/devicecode/devicecode_Builder.h>

#include <nnc/nn_DeviceCode.h>

#include <nnt/nntest.h>

extern "C" bool nntDeviceCodeInnerFormatTest(void);
extern "C" nnDeviceCode nntDeviceCodeMakeDeviceCodeValue(nnDeviceCodeCategory category, nnDeviceCodeInnerId innerId);

namespace nnt { namespace devicecode {

struct InnerFormatTestParam
{
    nn::detail::DeviceCodeType value;
    nn::DeviceCode::Category category;
    nn::DeviceCode::InnerId innerId;
};

class InnerFormat: public ::testing::TestWithParam<InnerFormatTestParam>
{};

TEST_P(InnerFormat, Basic)
{
    const auto& p = GetParam();
    auto dc = nn::DeviceCode(p.value);

    ASSERT_EQ(dc.GetInternalValue(), p.value);
    ASSERT_EQ(dc.GetCategory(), p.category);
    ASSERT_TRUE(dc.IsCategory(p.category));
    ASSERT_EQ(dc.GetInnerId(), p.innerId);
    ASSERT_EQ(
        dc.GetInternalValue(),
        NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(dc.GetCategory(), dc.GetInnerId())
    );
    ASSERT_EQ(
        dc,
        nn::DeviceCode(NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(dc.GetCategory(), dc.GetInnerId()))
    );
}

INSTANTIATE_TEST_CASE_P(
    InnerFormatInst,
    InnerFormat,
    ::testing::Values(
        InnerFormatTestParam({ NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(0x00000000ul, 0x00000000ul), 0x00000000ul, 0x00000000ul }),
        InnerFormatTestParam({ NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(0x00000000ul, 0x0000fffful), 0x00000000ul, 0x0000fffful }),
        InnerFormatTestParam({ NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(0x000000fful, 0x00000000ul), 0x000000fful, 0x00000000ul }),
        InnerFormatTestParam({ NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(0x000000fful, 0x0000fffful), 0x000000fful, 0x0000fffful }),

        // Invalid Code の内部表現の保証
        InnerFormatTestParam({ nn::DeviceCode::GetInvalidCode().GetInternalValue(), 0x00000000ul, 0x00000000ul }),

        // 範囲外の値がマスクされることの保証
        InnerFormatTestParam({ NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL_NO_RANGECHECK(0xfffffffful, 0xfffffffful), 0x000000fful, 0x0000fffful })
    )
);

TEST(InnerFormatC, CLinkage)
{
    ASSERT_TRUE(nntDeviceCodeInnerFormatTest());
}

TEST(Comparison, Basic)
{
    const auto dc1 = nn::DeviceCode(NN_DEVICECODE_MAKE_CODE(TEST, 1));

    ASSERT_TRUE (dc1 == dc1);
    ASSERT_FALSE(dc1 != dc1);

    const auto dc1copy = nn::DeviceCode(dc1);

    ASSERT_TRUE(dc1 == dc1copy);
    ASSERT_FALSE(dc1 != dc1copy);

    const auto dc1moved = nn::DeviceCode(std::move(dc1copy));

    ASSERT_TRUE(dc1 == dc1moved);
    ASSERT_FALSE(dc1 != dc1moved);

    const auto dc2 = nn::DeviceCode(NN_DEVICECODE_MAKE_CODE(TEST, 2));

    ASSERT_FALSE(dc1 == dc2);
    ASSERT_TRUE (dc1 != dc2);
}

TEST(Death, OutOfRange)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::DeviceCode(NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL(0x00000100, 0x00000000)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::DeviceCode(NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL(0x100000ff, 0x00000000)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::DeviceCode(NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL(0x00000000, 0x00010000)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::DeviceCode(NN_DETAIL_DEVICECODE_MAKE_CODE_IMPL(0x00000000, 0x100000ff)), "");

    // TODO: NNC_SDK_ASSERT 対応したら対応する
#if 0
    EXPECT_DEATH_IF_SUPPORTED(nntDeviceCodeMakeDeviceCodeValue(0x00000100, 0x00000000), "");
    EXPECT_DEATH_IF_SUPPORTED(nntDeviceCodeMakeDeviceCodeValue(0x100000ff, 0x00000000), "");
    EXPECT_DEATH_IF_SUPPORTED(nntDeviceCodeMakeDeviceCodeValue(0x00000000, 0x00010000), "");
    EXPECT_DEATH_IF_SUPPORTED(nntDeviceCodeMakeDeviceCodeValue(0x00000000, 0x100000ff), "");
#else
    // テストとしての意味はないが呼び出しだけしておく
    volatile auto dc1 = nntDeviceCodeMakeDeviceCodeValue(0x00000100, 0x00000000);
    volatile auto dc2 = nntDeviceCodeMakeDeviceCodeValue(0x000001ff, 0x00000000);
    ASSERT_FALSE(dc1._value == dc2._value);
#endif
}

}} // nnt::devicecode
