﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_RsaPssVerifier.h>
#include <nn/crypto/crypto_RsaPssSigner.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nnt/nntest.h>
#include "ProfileCheck.h"
#include "OctetString.h"

TEST(RsaTest, 2048Pss_Profile)
{
    // Test vectors from resources/SigVerPSS_186-3.rsp#1695
    OctetString n(
        "c6e0ed537a2d85cf1c4effad6419884d824ceabf5200e755691cb7328acd6a755fe85798502ccaec9e55d47afd0cf3258ebe920b50c5fd9d72897462bd0e459bbdf902b63d17195b2ef54908980be12aa7489f8af274b92c0cbc16aed2fa46f782d5517b666edfb2e5e5efeaff7e24965e26472e51980b0cfe457d297e6aa5dacb8e728dc6f58130f925a13275c3cace62f820db1e13cc5274c58ff4d7837671a1bf5f80d6ad8699c568df8d24dd0f152ded36ef4861f59b354bba96a076913a25facf4722737e6deed95b69a00fb2bced0feeedea4ff01a92605cfe26a6b39553d0c74e5650eb3779705e135c4b2fa518a8d4339c53efab4bb0058238def555");
    OctetString e("010001");
    OctetString d(
        "03ac73787e325992a96749d5ef8500e2ccf99e96214dbc22df2c6fde3538aaa8578e1b3cc871af5f940ed4a6df46438bdf240f896478fd2090fffa2af9c034a7cb684e5fc491f3940987c537d80128d6b37230ba4314c60d3580ad9aeb46ed6929dcf1629f6784667c547fec48c3112a1d9144f1802c82bb1476544e757e9853819185949352b92411adabd0f76efafe72c3b3fce88c5895b0bc4ac1ad36ec8d5be4adb89e72519850c6fc8c4076b658a2e554a37b5aa76aef7293a1ec256ccdc0c93c60aa528596a44ad72c76ed55726206d4bfd2f431745cc1c7dc399213051275fcfd2757552cef855be7bf23a5480688032bb4f322669a3e7d2fbff31c91");

    OctetString msg(
        "81eaf473d40896dbf4deac0f35c63bd1e129147c76e7aa8d0ef921631f55a7436411079f1bcc7b98714ac2c13b5e7326e60d918db1f05ffb19da767a95bb141a84c4b73664ccebf844f3601f7c853f009b21becba11af3106f1de5827b14e9fac84b2cbf16d18c045622acb260024768e8acc4c0ae2c0bd5f60a98023828cdec");
    OctetString sig(
        "40d59ebc6cb7b960cbda0db353f9b85d77e7c03f84447fb8e91b96a5a7377abc329d1f55c85e0dbedbc2886ce191d9e2cf3be05b33d6bbd2ba92b85eee2ff89cd6ee29cd531e42016e6aba1d620fe55e44480c033e8a59c0852dd1caffbc2ce82969e3a9f44ceff79f89993b9ebf3741b2ccab0b9516f2e128656a5b2ad5251e20c6ce0c26a14eef7ee86458942ddbe95ccc1f67b253e43e72117f49595dab5ba423496ece12825435661112666dbae71aaffd5a8f1d58db9dc02e0d70fe3ac36a87b8eeed4f20c00fd4303f9f767d03bca1a619bbe4b08e4e53b5cb69d2ba0235063e04ca392334d9979a41c42a66ca8b9721edcf76989ba89f3a170bb2e485");

    const size_t signatureSize = 2048 / 8;
    char signBuf[signatureSize];
    static uint8_t workBuf[8192];

    uint8_t random[] = {
        0x11,0x22,0x33,0x44,0x55,0x54,0x32,0x16,0x78,0x90
    };

    bool ok = true;
    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PSS/SHA256-Sign (Stack)", 27500, -1, 38 * 1024, 65, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PSS/SHA256-Sign (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PSS/SHA256-Sign (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPssSigner<2048 / 8,nn::crypto::Sha256Generator>::Sign(signBuf,
                                                                                 sizeof(signBuf),
                                                                                 n,
                                                                                 n.size(), d,
                                                                                 d.size(), msg,
                                                                                 msg.size(), random,
                                                                                 sizeof(random));
        }
    }
    ASSERT_TRUE(ok);
    ASSERT_EQ(sig,std::string(signBuf,signatureSize));

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PSS/SHA256-Sign (Heap)", 27500, -1, 38 * 1024, 65, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PSS/SHA256-Sign (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PSS/SHA256-Sign (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPssSigner<2048 / 8,nn::crypto::Sha256Generator>::Sign(signBuf,
                                                                                 sizeof(signBuf),
                                                                                 n,
                                                                                 n.size(), d,
                                                                                 d.size(), msg,
                                                                                 msg.size(), random,
                                                                                 sizeof(random),
                                                                                 workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);
    ASSERT_EQ(sig,std::string(signBuf,signatureSize));

    memcpy(signBuf,sig,sizeof(signBuf));
    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PSS/SHA256-Verify (Stack)", 300, -1, 6500, 32, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PSS/SHA256-Verify (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PSS/SHA256-Verify (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            ok &= nn::crypto::RsaPssVerifier<2048 / 8,nn::crypto::Sha256Generator>::Verify(signBuf,
                                                                                           sizeof(signBuf),
                                                                                           n,
                                                                                           n.size(), e,
                                                                                           e.size(), msg,
                                                                                           msg.size());
        }
    }
    ASSERT_TRUE(ok);

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PSS/SHA256-Verify (Heap)", 300, -1, 6500, 32, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PSS/SHA256-Verify (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PSS/SHA256-Verify (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            ok &= nn::crypto::RsaPssVerifier<2048 / 8,nn::crypto::Sha256Generator>::Verify(signBuf,
                                                                                           sizeof(signBuf),
                                                                                           n,
                                                                                           n.size(), e,
                                                                                           e.size(), msg,
                                                                                           msg.size(),
                                                                                           workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);
}
