﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>

/*
Generate and dump keys with :

openssl genrsa -out key2048.pem 2048
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 2p | xxd -p -r | xxd -i -c 16
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 4p | xxd -p -r | xxd -i -c 16
*/
static const unsigned char gKeyPublicN[] =
{
    0xdf, 0xa6, 0x14, 0x44, 0x94, 0x4a, 0xc2, 0xa6, 0xa8, 0x61, 0x7b, 0x09, 0x1e, 0x36, 0x35, 0xfd,
    0x09, 0x23, 0x6b, 0x3d, 0xf1, 0xd0, 0x4d, 0xcf, 0xf7, 0x60, 0x63, 0xa9, 0xdb, 0xa6, 0xa2, 0x74,
    0x28, 0x9b, 0x2a, 0xd7, 0xb3, 0xf8, 0x9d, 0x74, 0xea, 0x3f, 0x85, 0xba, 0x20, 0x94, 0xf6, 0xa5,
    0x24, 0xc2, 0xff, 0x0d, 0x32, 0xaa, 0xa6, 0xe1, 0xf9, 0x91, 0x32, 0xe3, 0x6e, 0xa8, 0x76, 0x0b,
    0xda, 0x9c, 0xef, 0xe4, 0x52, 0x94, 0x1c, 0x72, 0xc1, 0x5d, 0x9e, 0x10, 0x3c, 0x3e, 0x0c, 0x6b,
    0x3c, 0xe4, 0x12, 0x88, 0x5f, 0xb6, 0xce, 0xad, 0x3d, 0xe1, 0xc2, 0xa4, 0xe1, 0x28, 0xd3, 0x3d,
    0x05, 0x42, 0xdb, 0x64, 0x7d, 0x6e, 0x82, 0x43, 0xc1, 0x6e, 0x93, 0x31, 0x0b, 0xa5, 0x99, 0x13,
    0x5f, 0x72, 0xe3, 0x9c, 0x02, 0x05, 0x59, 0x77, 0xf1, 0x79, 0x54, 0x24, 0x65, 0x73, 0xd1, 0xf5,
    0x37, 0x2b, 0xdc, 0xd1, 0x78, 0x77, 0x04, 0xb8, 0xd5, 0x83, 0x0b, 0xe0, 0x1b, 0x4c, 0xa5, 0x1b,
    0x27, 0x82, 0x38, 0x94, 0x9c, 0x3b, 0xcf, 0x06, 0x7a, 0x33, 0xfb, 0x1b, 0xc7, 0x86, 0xcf, 0x44,
    0x3c, 0x78, 0x84, 0x10, 0x41, 0x22, 0x11, 0x1f, 0x7b, 0x37, 0xad, 0x66, 0xfd, 0x56, 0xd7, 0x61,
    0xa5, 0x32, 0xe5, 0x87, 0xd2, 0x16, 0x64, 0x12, 0xde, 0xc7, 0xbf, 0x6d, 0xb5, 0x1f, 0x6f, 0xcb,
    0xc4, 0xa1, 0x9a, 0xd3, 0xcf, 0x21, 0xa9, 0xe9, 0x02, 0x50, 0x6b, 0x13, 0x34, 0x20, 0xf0, 0x26,
    0x4a, 0x2e, 0x3e, 0x38, 0xcb, 0x4e, 0x77, 0x3b, 0x02, 0xaa, 0xfa, 0x0c, 0x82, 0xc2, 0x6c, 0x97,
    0x7e, 0x7c, 0x6f, 0x55, 0x59, 0xc9, 0xb4, 0x2b, 0x1c, 0xf7, 0xcd, 0x99, 0x15, 0xd7, 0xa8, 0x5a,
    0xe1, 0xeb, 0x67, 0x61, 0x33, 0x8e, 0xc7, 0x75, 0x28, 0xc5, 0xa9, 0x20, 0x45, 0x62, 0x7b, 0x19
};
static const unsigned char gKeyPublicE[] =
{
    0x01,0x00,0x01
};

static const unsigned char gKeyPrivateD[] =
{
    0xc9, 0x41, 0x6b, 0x95, 0xb7, 0x18, 0x7b, 0x87, 0x7b, 0x98, 0x95, 0x1e, 0x44, 0x68, 0xa6, 0x36,
    0xb8, 0xe5, 0x7d, 0x98, 0xf5, 0xa4, 0xec, 0x90, 0xcb, 0xdc, 0x9a, 0x72, 0xfc, 0xed, 0x3b, 0xdd,
    0x69, 0xb9, 0x10, 0x59, 0xa1, 0x63, 0x85, 0x55, 0xf3, 0x82, 0x0c, 0x35, 0xcd, 0x22, 0x1f, 0xbd,
    0xc6, 0x92, 0x0e, 0xf0, 0x72, 0x7a, 0xac, 0xc8, 0x47, 0x4b, 0xc9, 0x4e, 0x6e, 0x82, 0xfc, 0x0f,
    0x78, 0xbf, 0xa7, 0x04, 0x80, 0xe7, 0x35, 0x9e, 0x72, 0xb1, 0x6b, 0x0e, 0xd3, 0x17, 0x47, 0x17,
    0xad, 0x68, 0xd3, 0xe0, 0x92, 0xb9, 0x2f, 0xcd, 0x28, 0x15, 0xa3, 0x91, 0xb9, 0xe8, 0x82, 0x06,
    0xb4, 0x58, 0x50, 0x93, 0x62, 0x68, 0x72, 0xa2, 0xc4, 0x4b, 0x5b, 0xfb, 0x53, 0xae, 0xc6, 0x4c,
    0x6f, 0xa5, 0x34, 0x7b, 0x6c, 0x3a, 0x22, 0x23, 0xd4, 0x51, 0xc9, 0x4d, 0x23, 0x5a, 0x91, 0x40,
    0xf9, 0x89, 0x82, 0x91, 0x49, 0x89, 0x36, 0xdb, 0x2a, 0x45, 0xfb, 0x09, 0xa9, 0xf9, 0x00, 0xa6,
    0x4b, 0x90, 0xbf, 0xb0, 0xff, 0xa0, 0xce, 0x6b, 0xfd, 0xaa, 0x8c, 0x34, 0x5e, 0x2d, 0x6c, 0xc0,
    0xe9, 0xb5, 0x32, 0x6b, 0xc5, 0xa9, 0x77, 0xe7, 0x8f, 0xaa, 0x08, 0x0c, 0x03, 0xf1, 0x1a, 0xb1,
    0x98, 0x19, 0xc2, 0xf6, 0x39, 0x64, 0xc0, 0xfa, 0x39, 0x3e, 0xfd, 0x0a, 0x95, 0x4d, 0x7f, 0x04,
    0x80, 0xb6, 0x10, 0x49, 0xa9, 0x0d, 0x75, 0x8b, 0x46, 0xd9, 0x2d, 0xf0, 0x88, 0x1e, 0x15, 0x39,
    0xf0, 0xf1, 0x8a, 0x49, 0x49, 0xcf, 0xb1, 0x5e, 0xce, 0xb9, 0x0c, 0x93, 0x17, 0x36, 0x29, 0x14,
    0xf8, 0x49, 0xc4, 0xee, 0x93, 0xc1, 0x9d, 0xda, 0xf2, 0xfa, 0x5a, 0x73, 0xa6, 0xe4, 0xa6, 0x86,
    0x04, 0x0d, 0xe1, 0xbd, 0x48, 0x78, 0x4e, 0x71, 0x2d, 0x2b, 0x9a, 0x3a, 0x65, 0x39, 0xb5, 0x31
};

/*
Create a message and sign a SHA256 appendix with RSA-PKCS#1 :

echo -ne "I owe you 1 beer\0000" >msg.txt
openssl sha256 -binary -out msg.sha256 msg.txt
openssl pkeyutl -sign -inkey key2048.pem -pkeyopt rsa_padding_mode:pkcs1 -pkeyopt digest:sha256 -in msg.sha256 -out msg.pkcs1
xxd -i -c 16 msg.pkcs1

Verify it :
openssl rsa -in key2048.pem -pubout -out key2048-public.pem
openssl pkeyutl -verify -pubin -inkey key2048-public.pem -pkeyopt rsa_padding_mode:pkcs1 -pkeyopt digest:sha256 -in msg.sha256 -sigfile msg.pkcs1
*/
static const unsigned char gMessage[] = {
    "I owe you 1 beer"
};

static const unsigned char gSignature[] = {
    0xc4, 0x9d, 0x5a, 0x20, 0x26, 0x90, 0x79, 0x61, 0x98, 0xe5, 0x95, 0xbc, 0x34, 0xe4, 0x4b, 0x11,
    0x7d, 0x66, 0xba, 0x9e, 0x94, 0x8b, 0x8c, 0x2b, 0xa6, 0xc9, 0x36, 0x5f, 0x4c, 0xa3, 0x21, 0x7f,
    0x12, 0xf0, 0x70, 0x32, 0xca, 0xee, 0x79, 0xe4, 0x33, 0xbb, 0xf9, 0x18, 0x0f, 0x8c, 0x2e, 0x06,
    0xc7, 0xe2, 0xad, 0xa0, 0x08, 0x50, 0x89, 0xd9, 0x3f, 0xd8, 0x8c, 0x93, 0xd6, 0x16, 0x65, 0x1c,
    0xeb, 0x2f, 0xd6, 0x31, 0x72, 0x83, 0x9e, 0xfa, 0x53, 0x14, 0x2a, 0xad, 0xf4, 0x6e, 0x24, 0xdf,
    0x30, 0x63, 0xcd, 0x86, 0xab, 0x7e, 0xa3, 0xd3, 0x12, 0xfa, 0xc5, 0x9d, 0xe1, 0x3d, 0x77, 0x31,
    0x5a, 0x7c, 0xfd, 0x58, 0x2a, 0xf9, 0x55, 0x69, 0x89, 0x27, 0x2b, 0xf3, 0xbd, 0x89, 0x4e, 0xb6,
    0x1a, 0xd4, 0x13, 0x5e, 0x71, 0xf2, 0x4a, 0xdb, 0x87, 0xb0, 0x09, 0x6e, 0x9d, 0x5f, 0x10, 0xaa,
    0x7f, 0xb6, 0x69, 0xe9, 0x85, 0x07, 0x2c, 0x12, 0xa7, 0x40, 0xc9, 0x90, 0xde, 0xbd, 0x4a, 0x6b,
    0xd8, 0x5f, 0xb3, 0x7d, 0xb8, 0x0e, 0xb5, 0xdb, 0x62, 0x42, 0x09, 0x22, 0xb8, 0xd0, 0x8d, 0x63,
    0x9c, 0x04, 0xcc, 0x7b, 0x29, 0x69, 0x48, 0xa9, 0x88, 0xdf, 0xae, 0x64, 0xa4, 0x59, 0x07, 0x9d,
    0x19, 0x56, 0x87, 0x11, 0xcb, 0x48, 0x85, 0x49, 0x3a, 0x60, 0x22, 0x4e, 0x13, 0x1d, 0xc2, 0x03,
    0xc2, 0x48, 0x0a, 0x07, 0xde, 0x4a, 0x70, 0x38, 0x2f, 0x65, 0x4c, 0xab, 0x06, 0xd3, 0xdc, 0xc7,
    0xd7, 0x68, 0x11, 0x03, 0x84, 0x15, 0x00, 0x3d, 0xbb, 0xc1, 0x6a, 0xf1, 0x64, 0x3a, 0x31, 0x20,
    0xc7, 0x29, 0xf2, 0xc6, 0x11, 0xf8, 0x72, 0x5e, 0x23, 0x63, 0x8a, 0x1d, 0x07, 0xb1, 0x81, 0x7f,
    0xb9, 0x42, 0x29, 0xfb, 0x59, 0xde, 0x59, 0x65, 0xc5, 0x98, 0x1c, 0x21, 0x0a, 0x1c, 0x51, 0xfd
};

TEST(RsaTest, Pkcs1Verify1)
{
    nn::crypto::Rsa2048Pkcs1Sha256Verifier ver;
    ver.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPublicE, sizeof(gKeyPublicE) );
    ver.Update( gMessage, sizeof(gMessage) );
    EXPECT_TRUE( ver.Verify( gSignature, sizeof(gSignature) ) );

    EXPECT_TRUE( nn::crypto::VerifyRsa2048Pkcs1Sha256( gSignature, sizeof(gSignature),
                                                       gKeyPublicN, sizeof(gKeyPublicN),
                                                       gKeyPublicE, sizeof(gKeyPublicE),
                                                       gMessage, sizeof(gMessage) ) );
}

TEST(RsaTest, Pkcs1Sign1)
{
    nn::crypto::Rsa2048Pkcs1Sha256Signer gen;
    gen.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPrivateD, sizeof(gKeyPrivateD) );
    gen.Update( gMessage, sizeof(gMessage) );

    char signature[sizeof(gKeyPublicN)];

    std::memset( signature, 0, sizeof(signature) );
    gen.Sign( signature, sizeof(signature) );
    EXPECT_EQ( 0, std::memcmp( signature, gSignature, sizeof(signature) ) );

    std::memset( signature, 0, sizeof(signature) );
    nn::crypto::SignRsa2048Pkcs1Sha256( signature, sizeof(signature),
                                        gKeyPublicN, sizeof(gKeyPublicN),
                                        gKeyPrivateD, sizeof(gKeyPrivateD),
                                        gMessage, sizeof(gMessage) );
    EXPECT_EQ( 0, std::memcmp( signature, gSignature, sizeof(signature) ) );
}

TEST(RsaTest, Pkcs1VerifyFails)
{
    char signature[sizeof(gKeyPublicN)];
    // A signature of all 0's is valid input
    std::memset( signature, 0, sizeof(signature) );
    EXPECT_FALSE( nn::crypto::VerifyRsa2048Pkcs1Sha256( signature, sizeof(signature),
                                                        gKeyPublicN, sizeof(gKeyPublicN),
                                                        gKeyPublicE, sizeof(gKeyPublicE),
                                                        gMessage, sizeof(gMessage) ) );
}

TEST(RsaTest, Pkcs1VerifyStates)
{
    nn::crypto::Rsa2048Pkcs1Sha256Verifier ver;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Verify() without Initizlize() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify( gSignature, sizeof(gSignature) ), "");
#endif

    // Initialize
    ver.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPublicE, sizeof(gKeyPublicE) );

    // Calling Verify() without Update() is OK (verification result itself is false here but does not stop)
    ver.Verify( gSignature, sizeof(gSignature) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() after Verify() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Verify() twice fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify( gSignature, sizeof(gSignature) ), "");
#endif
}

TEST(RsaTest, Pkcs1SignStates)
{
    nn::crypto::Rsa2048Pkcs1Sha256Signer gen;
    char signature[sizeof(gKeyPublicN)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Sign() without Initizlize() fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign( signature, sizeof(signature) ), "");
#endif

    // Initialize
    gen.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPrivateD, sizeof(gKeyPrivateD) );

    // Calling Sign() without Update() is OK
    gen.Sign( signature, sizeof(signature) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() after Verify fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Sign() twice fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign( signature, sizeof(signature) ), "");
#endif
}

static const unsigned char gZero[2048 / 8] =
{
    0
};

TEST(RsaTest, Pkcs1Zeroes)
{
    nn::crypto::Rsa2048Pkcs1Sha256Verifier ver;
    EXPECT_FALSE(ver.Initialize(gZero, sizeof(gZero),
                                gZero, sizeof(gZero)));

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update(gMessage, sizeof(gMessage)), "");

    // Calling Verify() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify(gSignature, sizeof(gSignature)), "");
#endif

    EXPECT_FALSE(nn::crypto::VerifyRsa2048Pkcs1Sha256(gSignature, sizeof(gSignature),
                                                      gZero, sizeof(gZero),
                                                      gZero, sizeof(gZero),
                                                      gMessage, sizeof(gMessage)));

    nn::crypto::Rsa2048Pkcs1Sha256Signer gen;
    EXPECT_FALSE(gen.Initialize(gZero, sizeof(gZero),
                                gZero, sizeof(gZero)));
    char signature[sizeof(gKeyPublicN)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update(gMessage, sizeof(gMessage)), "");

    // Calling Sign() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign(signature, sizeof(signature)), "");
#endif

    std::memset(signature, 0xBD, sizeof(signature));
    EXPECT_FALSE(nn::crypto::SignRsa2048Pkcs1Sha256(signature, sizeof(signature),
                                                    gZero, sizeof(gZero),
                                                    gZero, sizeof(gZero),
                                                    gMessage, sizeof(gMessage)));
}
