﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_RsaPkcs1Verifier.h>
#include <nn/crypto/crypto_RsaPkcs1Signer.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nnt/nntest.h>
#include "ProfileCheck.h"
#include "OctetString.h"

TEST(RsaTest, 2048Pkcs1_Profile)
{
    // Test vectors from resources/SigVerPkcs1_186-3.rsp#1695
    OctetString n(
        "c6e0ed537a2d85cf1c4effad6419884d824ceabf5200e755691cb7328acd6a755fe85798502ccaec9e55d47afd0cf3258ebe920b50c5fd9d72897462bd0e459bbdf902b63d17195b2ef54908980be12aa7489f8af274b92c0cbc16aed2fa46f782d5517b666edfb2e5e5efeaff7e24965e26472e51980b0cfe457d297e6aa5dacb8e728dc6f58130f925a13275c3cace62f820db1e13cc5274c58ff4d7837671a1bf5f80d6ad8699c568df8d24dd0f152ded36ef4861f59b354bba96a076913a25facf4722737e6deed95b69a00fb2bced0feeedea4ff01a92605cfe26a6b39553d0c74e5650eb3779705e135c4b2fa518a8d4339c53efab4bb0058238def555");
    OctetString e("010001");
    OctetString d(
        "03ac73787e325992a96749d5ef8500e2ccf99e96214dbc22df2c6fde3538aaa8578e1b3cc871af5f940ed4a6df46438bdf240f896478fd2090fffa2af9c034a7cb684e5fc491f3940987c537d80128d6b37230ba4314c60d3580ad9aeb46ed6929dcf1629f6784667c547fec48c3112a1d9144f1802c82bb1476544e757e9853819185949352b92411adabd0f76efafe72c3b3fce88c5895b0bc4ac1ad36ec8d5be4adb89e72519850c6fc8c4076b658a2e554a37b5aa76aef7293a1ec256ccdc0c93c60aa528596a44ad72c76ed55726206d4bfd2f431745cc1c7dc399213051275fcfd2757552cef855be7bf23a5480688032bb4f322669a3e7d2fbff31c91");

    OctetString msg(
        "81eaf473d40896dbf4deac0f35c63bd1e129147c76e7aa8d0ef921631f55a7436411079f1bcc7b98714ac2c13b5e7326e60d918db1f05ffb19da767a95bb141a84c4b73664ccebf844f3601f7c853f009b21becba11af3106f1de5827b14e9fac84b2cbf16d18c045622acb260024768e8acc4c0ae2c0bd5f60a98023828cdec");

    const size_t signatureSize = 2048 / 8;
    char signBuf[signatureSize];
    static uint8_t workBuf[8192];

    bool ok = true;

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PKCS1/SHA1-Sign (Stack)", 27500, -1, 38 * 1024, 64, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PKCS1/SHA1-Sign (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PKCS1/SHA1-Sign (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Signer<2048 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), d, d.size() );
            gen.Update( msg, msg.size() );
            gen.Sign(signBuf, sizeof(signBuf));
        }
    }
    ASSERT_TRUE(ok);

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PKCS1/SHA1-Sign (Heap)", 27500, -1, 38 * 1024, 64, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PKCS1/SHA1-Sign (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PKCS1/SHA1-Sign (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Signer<2048 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), d, d.size() );
            gen.Update( msg, msg.size() );
            gen.Sign(signBuf, sizeof(signBuf), workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PKCS1/SHA1-Verify (Stack)", 300, -1, 6500, 31, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PKCS1/SHA1-Verify (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PKCS1/SHA1-Verify (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Verifier<2048 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), e, e.size() );
            gen.Update( msg, msg.size() );
            ok &= gen.Verify(signBuf, sizeof(signBuf));
        }
    }
    ASSERT_TRUE(ok);

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-2048-PKCS1/SHA1-Verify (Heap)", 300, -1, 6500, 31, Repeat );
        PROFILE_CHECK_Win("RSA-2048-PKCS1/SHA1-Verify (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-2048-PKCS1/SHA1-Verify (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Verifier<2048 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), e, e.size() );
            gen.Update( msg, msg.size() );
            ok &= gen.Verify(signBuf, sizeof(signBuf), workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);
}

TEST(RsaTest, 4096Pkcs1_Profile)
{
    OctetString n(
        "b10935650754c1a27e54f9ee525c77045d1a056baedc8c1ba05a3d66322c8e2e41689ca02b8e6ec55d3331cd714e161b35c774311540c6410cc3b130e65cec325377ccf18a5d77054afaa737e65b68662327627f5d1092df98c08b60a3ba67599b1406986f441528d2f1f5d6fe5ec6b8aa797dbdb3162876b08298237af81c7004863576af572ddd66c487cf32a82d0f9fe260f8f2681613208aec55723622f50d3cb3bd33b1226d5b5b9ea2c97a2c59054e31536849ec2f0f2597c077e621ddea0a22ddfb1df925c9f1941cd431afce0398d28e736cfeab3dd7f062a65a2a4c1ddca1f32afa03b4ddfd8b7ca40cc0ce2ed42229d753f03cc6d8bcdca76ab7c2703c5ddad08aa9bf9a27740a8b23168c6b55917bd3d5c1c057c34d1efb1411da51ab745519bd92b9432fea8fadcb9a70e5dc3adcd3081d3333507b7b998886e988296628dd7265e64eab557712556cc492377f15a078dcb620a6e7f051a1fb8754efdca3de253dd0aad4205b032659e4a4fb307f074fbde340702b5911acb34737d7834600cf2ac77f62f83ccc9ba78ff911021b9f8ad1bf421430ae0d64916944b504542c0f6263c48e5e233ef1869df1efc2ccf6f4a95e4db2de7b4cb1a992bef6573511bab4bf2cd58e72a0c235f56dfc182df4dfffb69433a5677415f35d84f5125f5200eb57868cce6b74c5833417990e318372c9239a36dca1a0b28809");
    OctetString e("010001");
    OctetString d(
        "25e0b881e32da938611b4156525ce24216c168837fa84479ecb72207e9984adb6eb7393bb3d607b1469d9b7c3f4fdbbefaa4b0218850919a7d66a954b315129c39eb99f7dc08df5c4c8c90968f3ce37b66ee184ef3d485f83d308521aa2649d28c319eafa2aec87031a1ff5d7e933ca56a2410593425fb865981b7976fca021b9d7c3198312fcfea5d0093a62b4a7c49a985c005c3a7ad816e270b25c507fc36be1c4cc0a07cb7c6fa130240062793b18047189aa5e79b16fe80a6955191f5910b701bc1aee6dcd5cefd57194bf54d8e208ae41202744190d5ec8bcc2f977f11461a5cb4306fc9b73afff2863a7b580d454bb1fb8dccb1cbef27945109a8f5a3d2b1a32566aa4e8c01a62a1d735117ff5a6a3cc7e70756e7df10e97ee75c5f8ef89fb0a97d7a35698069f59a9f365d4396ab9a764c2fbefb840faf5f57737801b73ac6f9f524167b4f3a567aff999a0db10d55d82155720a5ddc63c35b6a632a3da59c16bd0c143c437661dcd339652ae42f54f8b2d8e52672613772abfe8cf0d6ebbbccc764b51b3eda2ae28d4ba8747a430ccc32c73baea63d050b86210c485ac9554606070764cd06b423efdda4059fc45ffd7193f7123d14014d5bbe5b542476e7bfdc4905731a0d9987fe4a68cf6077c3df63778af08a1f4eb8f00a4a8e03aa8726f43fa829d87c2d0a32e16b47a3f0472a6368ec50a144234c802e6919");

    OctetString msg(
        "200ee1828829d9fd27576e253ea700245c38c5ae78fc76e33dbc4877a55d1e10c961fdeff8b1c0d306990d1ae614872a7b4a450fac578465b74e9879c77d29abc8f39b177a40ce74c47c083b4c8b2f0c449e3c4f87fdec17e405b84bb96c0807c4cda44037606ba70a0847d0959460945e3e90b4307818be6de99135a4b225ec");
    OctetString sig(
        "33d2c45a355afbb8eb5f64fbdc4bd3719afbcba36d5d4bbc697887ecb7e7ecaf99bb31798977e3385544fd4c44efe1b05f2a34119bafcd6377c24f57c030498f6d96148677079ffa05a253e9499d6b13d3c02d5347dd3263045919f1a7169f4297cb4ead2340e6706269a8607b1575044e75adcb94cc7db8ed80a776ff1e56d1ac13ed7d82439750d51904337c63bea9a059a056f30bbb8c1c721a0d666cb843b1a8223b197a9f48e3941a9f6d8cf022dc4edca612d057b2548986698f2a53266f49e7995640eebe929fceda0d33eb24437113edaea93e8d7892ab14b25e851b88808b470a90bdcff021e798ffbed003b3b9c8d53e4a1cf77aa7b5016a2ca41d4da22ebe498c73bb3d0239cf41ef7f404fd609d390c8c1a0d2f0a6817cd3ef966196d64c89524032b6ddfdc6f9d6876d6b9e1c55010969238af5f2ab616bbc9234445d07f2462aa907b31a08677fcb9236206187e00888b53e925c334f4993d3f18ad6db81ced54b666fc6513da7a4e8a8f1c0eaeb6819cce7cbd27de9f9c9718d900297247e41a704b7613221fa114cf145a1cdabf4217eee25678a24420c4a75f8d444069c976ef95d61e5abed512c01cff4b864038ec8e4aa877ac501664d48be5aba39a35df9b3b1ef01a25ebede122b1797494442420e0f0d0d4b7c49d85f9ffdb102a7a1e0ce6e4a99d9690a80958ed548e5beadec583c192316fc7311");

    const size_t signatureSize = 4096 / 8;
    char signBuf[signatureSize];
    static uint8_t workBuf[8192];

    bool ok = true;
    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-4096-PKCS1/SHA1-Sign (Stack)", 190000, -1, 74 * 1024, 65, Repeat );
        PROFILE_CHECK_Win("RSA-4096-PKCS1/SHA1-Sign (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-4096-PKCS1/SHA1-Sign (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Signer<4096 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), d, d.size() );
            gen.Update( msg, msg.size() );
            gen.Sign(signBuf, sizeof(signBuf));
        }
    }
    ASSERT_TRUE(ok);
    ASSERT_EQ(sig,std::string(signBuf,signatureSize));

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-4096-PKCS1/SHA1-Sign (Heap)", 190000, -1, 74 * 1024, 65, Repeat );
        PROFILE_CHECK_Win("RSA-4096-PKCS1/SHA1-Sign (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-4096-PKCS1/SHA1-Sign (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Signer<4096 / 8, nn::crypto::Sha1Generator> gen;
            gen.Initialize( n, n.size(), d, d.size() );
            gen.Update( msg, msg.size() );
            gen.Sign(signBuf, sizeof(signBuf), workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);
    ASSERT_EQ(sig,std::string(signBuf,signatureSize));

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-4096-PKCS1/SHA1-Verify (Stack)", 882, -1, 12 * 1024, 32, Repeat );
        PROFILE_CHECK_Win("RSA-4096-PKCS1/SHA1-Verify (Stack)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-4096-PKCS1/SHA1-Verify (Stack)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Verifier<4096 / 8, nn::crypto::Sha1Generator> ver;
            ver.Initialize( n, n.size(), e, e.size() );
            ver.Update( msg, msg.size() );
            ok &= ver.Verify(signBuf, sizeof(signBuf));
        }
    }
    ASSERT_TRUE(ok);

    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("RSA-4096-PKCS1/SHA1-Verify (Heap)", 882, -1, 12 * 1024, 32, Repeat );
        PROFILE_CHECK_Win("RSA-4096-PKCS1/SHA1-Verify (Heap)", -1, -1, -1, -1, Repeat );
        PROFILE_CHECK_OsHorizon("RSA-4096-PKCS1/SHA1-Verify (Heap)", -1, -1, -1, -1, Repeat );
        for ( int i = Repeat; i--; )
        {
            nn::crypto::RsaPkcs1Verifier<4096 / 8, nn::crypto::Sha1Generator> ver;
            ver.Initialize( n, n.size(), e, e.size() );
            ver.Update( msg, msg.size() );
            ok &= ver.Verify(signBuf, sizeof(signBuf), workBuf, sizeof(workBuf));
        }
    }
    ASSERT_TRUE(ok);
}
