﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <map>
#include <string>
#include <vector>
#include <set>

#include <nnt/nntest.h>
#include <nn/nn_Macro.h>

/**
    This is a utility base class to process text files from the NIST
    Cryptographic Algorithms Validation Suite (http://csrc.nist.gov/groups/STM/cavp/)
    and run the test units it contains.
*/
class CavsParser
{
public:
    CavsParser();
    void TestFile(const char* filename, int expectCompletedUnits);
    void TestString(const std::string& str, int expectCompletedUnits = 1);

    const char* m_FileName;
    int m_SkipUntilLine;
    int m_Completed;
    int m_LineNumber;

protected:
    typedef int TokenId;

    struct AllowedToken
    {
        std::string m_Prefix;
        bool m_BytesFromHex;//should we do conversion hex -> binary
        bool m_StackAll;

        AllowedToken()  // needed for std::map
        {
        }

        AllowedToken(const std::string& format, bool needH2B, bool needStack = false)
        {
            m_Prefix = format;
            m_BytesFromHex = needH2B;
            m_StackAll = needStack;
        }
    };

    std::map<TokenId, AllowedToken> m_AllowedTokens;
    bool ExtractToken(const std::string& line);
    TokenId m_TokenForUnitComplete;

    std::map<TokenId, std::string> m_UnitTokens;
    std::map<TokenId, std::vector<std::string> > m_StackedTokens;
    bool Has( TokenId i )
    {
        auto it = m_UnitTokens.find( i );
        if ( it == m_UnitTokens.end() )
        {
            return false;
        }
        return true;
    }
    void ClearUnit();

    virtual void TestUnit() = 0;
    bool m_Skipped;
};
