﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_RsaPssSigner.h>
#include <nn/crypto/crypto_RsaPssVerifier.h>
#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/crypto/crypto_RsaPssSha256Encoder.h>

#include "testCrypto_Util.h"

/*
Generate and dump keys with :

openssl genrsa -out key2048.pem 2048
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 2p | xxd -p -r | xxd -i -c 16
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 4p | xxd -p -r | xxd -i -c 16
*/
static const unsigned char gKeyPublicN[] =
{
    0xb4, 0xd1, 0x13, 0xc9, 0x51, 0x42, 0xe6, 0x76, 0x62, 0xf4, 0xf6, 0x04, 0xb6, 0x6e, 0x7b, 0x2b,
    0x3b, 0xe8, 0xe3, 0x2f, 0x43, 0x97, 0x08, 0x82, 0x70, 0xda, 0x5e, 0xae, 0x6f, 0x78, 0x48, 0xd5,
    0x0d, 0x83, 0xf3, 0x5e, 0x8c, 0xa5, 0x43, 0xdf, 0xd2, 0xdc, 0x42, 0x1c, 0xf6, 0x97, 0xba, 0xd2,
    0x8b, 0xf1, 0x28, 0x49, 0x11, 0x06, 0x14, 0x42, 0xc0, 0x5b, 0xf9, 0x03, 0xdf, 0x5d, 0x15, 0xde,
    0x04, 0x0d, 0xdf, 0x33, 0xbd, 0x83, 0xe9, 0x3c, 0x8f, 0x2e, 0xc4, 0x17, 0x21, 0x92, 0x25, 0xc7,
    0xe2, 0x09, 0xe8, 0x75, 0x4e, 0x1f, 0x87, 0xb9, 0x2e, 0x41, 0x0a, 0x80, 0xf9, 0xad, 0x73, 0x71,
    0x30, 0xba, 0xc8, 0x87, 0x8b, 0x9b, 0x2d, 0x20, 0x4a, 0xa0, 0x44, 0xa4, 0xb4, 0x99, 0xdd, 0xd8,
    0x73, 0xfd, 0xae, 0xe8, 0xbb, 0x9d, 0x40, 0x7a, 0x8c, 0x7c, 0xad, 0x5e, 0xa4, 0x03, 0x10, 0x3e,
    0x55, 0x23, 0x97, 0xb9, 0x7c, 0x93, 0x6a, 0x74, 0x3b, 0xe8, 0x2f, 0x70, 0x03, 0xb6, 0xad, 0x1c,
    0x5b, 0x4c, 0x81, 0x60, 0xfd, 0xef, 0xeb, 0xb0, 0x3d, 0x2b, 0x1d, 0xe3, 0xb7, 0xd5, 0xea, 0x30,
    0x9c, 0x8e, 0xc9, 0x53, 0x26, 0x42, 0x05, 0xd6, 0x84, 0x03, 0xb1, 0x45, 0x28, 0xff, 0x67, 0xe6,
    0x2b, 0x82, 0x91, 0xb4, 0xe2, 0x18, 0xad, 0xa5, 0x25, 0xb1, 0x7f, 0x79, 0xd8, 0x07, 0xd4, 0x2a,
    0x61, 0x1e, 0xbf, 0x45, 0x40, 0xa6, 0xc3, 0xff, 0x30, 0xbd, 0x04, 0x3f, 0x32, 0xf9, 0x08, 0xd9,
    0xb1, 0x4a, 0x5f, 0x4e, 0x44, 0x80, 0x06, 0x94, 0x26, 0xcf, 0xa7, 0x88, 0xb1, 0xeb, 0x03, 0x29,
    0xf3, 0x54, 0x11, 0xbb, 0x0f, 0x22, 0xf9, 0x5d, 0x50, 0xf4, 0x4e, 0x5f, 0x16, 0x08, 0xbb, 0x2c,
    0x61, 0xae, 0xee, 0x20, 0xb7, 0x90, 0xdc, 0x51, 0xc1, 0x86, 0xd7, 0x55, 0xfa, 0xab, 0x8c, 0x11
};
static const unsigned char gKeyPublicE[] =
{
    0x01,0x00,0x01
};

static const unsigned char gKeyPrivateD[] =
{
    0x44, 0xa1, 0x52, 0xfc, 0x47, 0x9e, 0xb7, 0xf0, 0xd9, 0x70, 0x96, 0xec, 0x84, 0xfd, 0xfc, 0x3c,
    0x15, 0xba, 0x38, 0x78, 0xd9, 0x13, 0xbb, 0x84, 0x1f, 0x27, 0x93, 0x66, 0x6c, 0x22, 0xe0, 0xaa,
    0x15, 0x3e, 0x2f, 0x22, 0x85, 0x09, 0xb9, 0x70, 0x7c, 0x0d, 0x63, 0xfe, 0xb0, 0x4e, 0x38, 0xf7,
    0xef, 0x6b, 0xf2, 0x59, 0x81, 0x5f, 0xa8, 0x53, 0x09, 0xc6, 0x1f, 0x07, 0x2c, 0x41, 0x9d, 0xaa,
    0x2e, 0x29, 0xc1, 0xbd, 0x56, 0xc2, 0x79, 0xc5, 0xa5, 0xa7, 0x1f, 0x80, 0x94, 0x1a, 0x6f, 0xc2,
    0xa5, 0x76, 0x13, 0x48, 0x6b, 0x05, 0xa6, 0xb3, 0x4f, 0xee, 0x19, 0xf9, 0x88, 0x4b, 0x20, 0x95,
    0x26, 0xe4, 0xce, 0x76, 0x3c, 0x34, 0x02, 0xf5, 0xcd, 0xbd, 0x75, 0x30, 0x7a, 0xe8, 0x27, 0xff,
    0x1f, 0x4f, 0x7b, 0x93, 0x14, 0x9b, 0x04, 0x3d, 0x3f, 0x93, 0x97, 0x0e, 0x63, 0xfc, 0x51, 0xae,
    0x62, 0xc8, 0xe4, 0x69, 0xb4, 0x82, 0x8b, 0xd9, 0x77, 0x8e, 0x8b, 0x11, 0x01, 0x83, 0x5f, 0xa6,
    0xbc, 0x81, 0xfe, 0xd1, 0x1c, 0x98, 0xd0, 0x60, 0x4b, 0x65, 0xa8, 0x8f, 0xf9, 0x4f, 0x8e, 0xad,
    0xb9, 0x45, 0xe0, 0xbf, 0x64, 0xbc, 0x5d, 0xf1, 0x30, 0x7f, 0x1f, 0xf9, 0x10, 0x7e, 0x09, 0x65,
    0x3d, 0x0c, 0xc7, 0x21, 0x03, 0xbe, 0x9c, 0x28, 0x8f, 0x2b, 0x84, 0x13, 0xfc, 0x5a, 0x14, 0x2c,
    0x4d, 0x40, 0x06, 0x61, 0x49, 0xee, 0x2f, 0x1a, 0x00, 0xc4, 0xc6, 0xa5, 0xb6, 0x97, 0x55, 0x74,
    0xbd, 0x3c, 0x13, 0xcd, 0xd9, 0x49, 0xcb, 0x7a, 0x3a, 0x9d, 0x9f, 0x3b, 0xdd, 0xcd, 0x92, 0x46,
    0xda, 0x8e, 0xed, 0x83, 0x96, 0x80, 0x93, 0x65, 0x51, 0xfb, 0x67, 0x67, 0x9e, 0x5d, 0xd6, 0xd6,
    0x75, 0x11, 0x79, 0xc8, 0x22, 0x00, 0x16, 0xcc, 0x83, 0x74, 0x87, 0xb4, 0x8f, 0x8e, 0x0a, 0xbd
};

/*
Create a message and sign a SHA256 appendix with RSA-PSS :

echo -ne "I owe you 1 beer\0000" >msg.txt
openssl sha256 -binary -out msg.sha256 msg.txt
openssl pkeyutl -sign -inkey key2048.pem -pkeyopt rsa_padding_mode:pss -pkeyopt rsa_pss_saltlen:-1 -pkeyopt digest:sha256 -in msg.sha256 -out msg.pss
xxd -i -c 16 msg.pss

Verify it :
openssl rsa -in key2048.pem -pubout -out key2048-public.pem
openssl pkeyutl -verify -pubin -inkey key2048-public.pem -pkeyopt rsa_padding_mode:pss -pkeyopt digest:sha256 -in msg.sha256 -sigfile msg.pss
*/
static const unsigned char gMessage[] = {
    "I owe you 1 beer"
};

static const unsigned char gSignature[] = {
    0x84, 0x26, 0xb4, 0xbf, 0xcd, 0x80, 0x59, 0x2f, 0x6e, 0x03, 0x1e, 0x30, 0xe3, 0xe3, 0xc9, 0x80,
    0xd5, 0x72, 0x62, 0xbd, 0x60, 0x36, 0x1e, 0x7f, 0x77, 0xb4, 0xc1, 0x19, 0x29, 0x7c, 0x47, 0x35,
    0x29, 0xd8, 0x18, 0xc1, 0x92, 0x31, 0xc3, 0x8d, 0x33, 0x6e, 0x43, 0xb4, 0xf3, 0x7c, 0x2a, 0xb7,
    0x72, 0x34, 0x31, 0xfd, 0xd6, 0xbc, 0x5b, 0xb5, 0xdf, 0x1c, 0xd6, 0x57, 0x7c, 0xa5, 0x8e, 0xe9,
    0xec, 0x9a, 0x38, 0x8c, 0x8a, 0x08, 0xcb, 0xc4, 0xd5, 0x72, 0x81, 0xb1, 0x82, 0x68, 0xd4, 0x7c,
    0x8e, 0x27, 0x08, 0xa8, 0xc4, 0x7d, 0x0e, 0x88, 0xf9, 0xfe, 0xb8, 0x06, 0x87, 0x90, 0x25, 0xf7,
    0x54, 0x9d, 0x8c, 0xbb, 0x3a, 0x48, 0xa7, 0x9b, 0x85, 0x53, 0x68, 0x9b, 0x0a, 0x11, 0x0f, 0x8c,
    0x53, 0x67, 0x98, 0x6e, 0x7d, 0x06, 0x4a, 0x33, 0x05, 0x9a, 0x95, 0x1c, 0xe4, 0xf9, 0xe9, 0x64,
    0xe5, 0xf0, 0x10, 0xdf, 0x0f, 0x34, 0xc1, 0xcb, 0x98, 0x4b, 0x89, 0x3f, 0xdc, 0xd6, 0x5b, 0xa7,
    0xf0, 0x7d, 0x60, 0xaa, 0xa2, 0x0a, 0x4c, 0x66, 0x0f, 0x9e, 0xed, 0xd5, 0xdd, 0x10, 0xf7, 0xee,
    0x9b, 0x93, 0x7d, 0xff, 0xa6, 0x62, 0xb1, 0x99, 0x90, 0xd3, 0x28, 0x10, 0x8f, 0x29, 0x0c, 0xc9,
    0x1b, 0x04, 0xef, 0xd5, 0x0d, 0x58, 0xcf, 0x0b, 0xe1, 0x24, 0x4b, 0x7e, 0xbb, 0x42, 0xb7, 0x56,
    0x3e, 0x4b, 0x46, 0xae, 0xb5, 0x7d, 0xcf, 0x80, 0x86, 0x7c, 0xba, 0x15, 0x6f, 0x3a, 0xd6, 0x22,
    0xbd, 0x16, 0xaf, 0x7c, 0x31, 0x33, 0xc8, 0x2a, 0xa0, 0xbd, 0x42, 0x3c, 0xae, 0xe4, 0x41, 0xc2,
    0x7b, 0xf8, 0xb0, 0x91, 0x7f, 0xd3, 0xcf, 0x8c, 0xb2, 0xc5, 0xc8, 0x27, 0xdb, 0x89, 0x15, 0xa9,
    0x9e, 0x57, 0xf2, 0x16, 0xe2, 0xfc, 0x3f, 0xf6, 0x7c, 0xe9, 0x4c, 0x12, 0x14, 0x4a, 0xd6, 0x90
};

TEST(RsaTest, PssVerify1)
{
    nn::crypto::Rsa2048PssSha256Verifier ver;

    EXPECT_LE(sizeof(gSignature), (nn::crypto::RsaPssVerifier<2048 / 8, nn::crypto::Sha256Generator>::SignatureSize));

    ver.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPublicE, sizeof(gKeyPublicE) );
    ver.Update( gMessage, sizeof(gMessage) );
    EXPECT_TRUE( ver.Verify( gSignature, sizeof(gSignature) ) );

    EXPECT_TRUE( nn::crypto::VerifyRsa2048PssSha256( gSignature, sizeof(gSignature),
                                                     gKeyPublicN, sizeof(gKeyPublicN),
                                                     gKeyPublicE, sizeof(gKeyPublicE),
                                                     gMessage, sizeof(gMessage) ) );

    uint8_t workBuffer[nn::crypto::Rsa2048PssSha256Verifier::RequiredWorkBufferSize];
    EXPECT_TRUE( nn::crypto::VerifyRsa2048PssSha256( gSignature, sizeof(gSignature),
                                                     gKeyPublicN, sizeof(gKeyPublicN),
                                                     gKeyPublicE, sizeof(gKeyPublicE),
                                                     gMessage, sizeof(gMessage),
                                                     workBuffer, sizeof(workBuffer)) );
}

// Same random as used by OpenSSL above, to get matching signatures
static const unsigned char gSalt[] = {
    0x90, 0xbf, 0x9b, 0x23, 0xa9, 0xc3, 0x86, 0x2d, 0xc9, 0x26, 0x76, 0x62, 0xa5, 0x6d, 0x49, 0xcc,
    0x7f, 0x00, 0x24, 0x5b, 0xd6, 0x7f, 0xf1, 0x8c, 0xfd, 0x0f, 0xcd, 0x11, 0xf0, 0x61, 0x40, 0x3b,
};

TEST(RsaTest, PssSign1)
{
    nn::crypto::Rsa2048PssSha256Signer gen;
    gen.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPrivateD, sizeof(gKeyPrivateD) );
    gen.Update( gMessage, sizeof(gMessage) );

    char signature[sizeof(gKeyPublicN)];

    EXPECT_LE(sizeof(signature), (nn::crypto::RsaPssSigner<2048 / 8, nn::crypto::Sha256Generator>::SignatureSize));

    std::memset( signature, 0, sizeof(signature) );
    gen.Sign( signature, sizeof(signature), gSalt, sizeof(gSalt) );
    EXPECT_ARRAY_EQ(signature, gSignature, sizeof(signature));

    std::memset( signature, 0, sizeof(signature) );
    nn::crypto::SignRsa2048PssSha256( signature, sizeof(signature),
                                      gKeyPublicN, sizeof(gKeyPublicN),
                                      gKeyPrivateD, sizeof(gKeyPrivateD),
                                      gMessage, sizeof(gMessage),
                                      gSalt, sizeof(gSalt) );
    EXPECT_ARRAY_EQ(signature, gSignature, sizeof(signature));

    uint8_t workBuffer[nn::crypto::Rsa2048PssSha256Signer::RequiredWorkBufferSize];
    std::memset( signature, 0, sizeof(signature) );
    nn::crypto::SignRsa2048PssSha256( signature, sizeof(signature),
                                      gKeyPublicN, sizeof(gKeyPublicN),
                                      gKeyPrivateD, sizeof(gKeyPrivateD),
                                      gMessage, sizeof(gMessage),
                                      gSalt, sizeof(gSalt),
                                      workBuffer, sizeof(workBuffer) );
    EXPECT_ARRAY_EQ(signature, gSignature, sizeof(signature));

}

TEST(RsaTest, PssEncode)
{
    char em[sizeof(gKeyPublicN)];

    // Encode
    nn::crypto::EncodeRsa2048PssSha256(em, sizeof(em), gMessage, sizeof(gMessage), gSalt, sizeof(gSalt));

    // Generate signature by RsaCalculator
    nn::crypto::RsaCalculator<256, 256> rsa;
    EXPECT_TRUE(rsa.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD)));
    EXPECT_TRUE(rsa.ModExp(em, sizeof(em), em, sizeof(em)));

    EXPECT_ARRAY_EQ(em, gSignature, sizeof(gSignature));
}

TEST(RsaTest, PssVerifyFails)
{
    char signature[sizeof(gKeyPublicN)];
    // A signature of all 0's is valid input
    std::memset( signature, 0, sizeof(signature) );
    EXPECT_FALSE( nn::crypto::VerifyRsa2048PssSha256( signature, sizeof(signature),
                                                      gKeyPublicN, sizeof(gKeyPublicN),
                                                      gKeyPublicE, sizeof(gKeyPublicE),
                                                      gMessage, sizeof(gMessage) ) );
}

TEST(RsaTest, PssVerifyStates)
{
    nn::crypto::Rsa2048PssSha256Verifier ver;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Verify() without Initizlize() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify( gSignature, sizeof(gSignature) ), "");
#endif

    // Initialize
    ver.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPublicE, sizeof(gKeyPublicE) );

    // Calling Verify() without Update() is OK (verification result itself is false here but does not stop)
    ver.Verify( gSignature, sizeof(gSignature) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() after Verify() fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Verify() twice fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify( gSignature, sizeof(gSignature) ), "");
#endif
}

TEST(RsaTest, PssSignStates)
{
    nn::crypto::Rsa2048PssSha256Signer gen;
    char signature[sizeof(gKeyPublicN)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Sign() without Initizlize() fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign( signature, sizeof(signature), gSalt, sizeof(gSalt) ), "");
#endif

    // Initialize
    gen.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPrivateD, sizeof(gKeyPrivateD) );

    // Calling Sign() without Update() is OK
    gen.Sign( signature, sizeof(signature), gSalt, sizeof(gSalt) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() after Verify fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update( gMessage, sizeof(gMessage) ), "");

    // Calling Sign() twice fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign( signature, sizeof(signature), gSalt, sizeof(gSalt) ), "");
#endif
}

static const unsigned char gZero[2048 / 8] =
{
    0
};

TEST(RsaTest, PssZeroes)
{
    nn::crypto::Rsa2048PssSha256Verifier ver;
    EXPECT_FALSE(ver.Initialize(gZero, sizeof(gZero),
                                gZero, sizeof(gZero)));
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Update(gMessage, sizeof(gMessage)), "");

    // Calling Verify() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(ver.Verify(gSignature, sizeof(gSignature)), "");
#endif

    EXPECT_FALSE(nn::crypto::VerifyRsa2048PssSha256(gSignature, sizeof(gSignature),
                                                    gZero, sizeof(gZero),
                                                    gZero, sizeof(gZero),
                                                    gMessage, sizeof(gMessage)));

    nn::crypto::Rsa2048PssSha256Signer gen;
    EXPECT_FALSE(gen.Initialize(gZero, sizeof(gZero),
                                gZero, sizeof(gZero)));
    char signature[sizeof(gKeyPublicN)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Update() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Update(gMessage, sizeof(gMessage)), "");

    // Calling Sign() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(gen.Sign(signature, sizeof(signature), gSalt, sizeof(gSalt)), "");
#endif

    std::memset(signature, 0xBD, sizeof(signature));
    EXPECT_FALSE(nn::crypto::SignRsa2048PssSha256(signature, sizeof(signature),
                                                  gZero, sizeof(gZero),
                                                  gZero, sizeof(gZero),
                                                  gMessage, sizeof(gMessage),
                                                  gSalt, sizeof(gSalt)));
}

TEST(RsaTest, PssMemoryShortage)
{
    uint8_t workBuffer[nn::crypto::Rsa2048PssSha256Signer::RequiredWorkBufferSize];

    // Sign
    {
        char signature[sizeof(gKeyPublicN)];

        // 0 バイトから 1KB 刻みくらいで複数パターンチェックしておく
        for (size_t bufferSize = 0; bufferSize < sizeof(workBuffer); bufferSize += 1024)
        {
            // ワークバッファが足りないと Develop ビルドでは事前条件チェックで ASSERT され、
            // Release ビルドでは false が返される
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            EXPECT_DEATH_IF_SUPPORTED(
                nn::crypto::SignRsa2048PssSha256( signature, sizeof(signature),
                                                  gKeyPublicN, sizeof(gKeyPublicN),
                                                  gKeyPrivateD, sizeof(gKeyPrivateD),
                                                  gMessage, sizeof(gMessage),
                                                  gSalt, sizeof(gSalt),
                                                  workBuffer, bufferSize), ""
            );
#else
            EXPECT_FALSE(
                nn::crypto::SignRsa2048PssSha256( signature, sizeof(signature),
                                                  gKeyPublicN, sizeof(gKeyPublicN),
                                                  gKeyPrivateD, sizeof(gKeyPrivateD),
                                                  gMessage, sizeof(gMessage),
                                                  gSalt, sizeof(gSalt),
                                                  workBuffer, bufferSize)
            );
#endif
        }
    }

    // Verify
    {
        // 0 バイトから 1KB 刻みくらいで複数パターンチェックしておく
        for (size_t bufferSize = 0; bufferSize < sizeof(workBuffer); bufferSize += 1024)
        {
            // 呼び出し時にワークバッファが足りないと Develop ビルドでは事前条件チェックで ASSERT され、
            // Release ビルドでは false が返される
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            EXPECT_DEATH_IF_SUPPORTED(
                nn::crypto::VerifyRsa2048PssSha256( gSignature, sizeof(gSignature),
                                                    gKeyPublicN, sizeof(gKeyPublicN),
                                                    gKeyPublicE, sizeof(gKeyPublicE),
                                                    gMessage, sizeof(gMessage),
                                                    workBuffer, bufferSize), ""
            );
#else
            EXPECT_FALSE(
                nn::crypto::VerifyRsa2048PssSha256( gSignature, sizeof(gSignature),
                                                    gKeyPublicN, sizeof(gKeyPublicN),
                                                    gKeyPublicE, sizeof(gKeyPublicE),
                                                    gMessage, sizeof(gMessage),
                                                    workBuffer, bufferSize)
            );
#endif
        }
    }
}
