﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_HmacSha1Generator.h>

#include "testCrypto_Util.h"

struct HmacSha1TestVector
{
    char       data[nn::crypto::HmacSha1Generator::BlockSize];
    size_t     dataSize;
    nn::Bit8   key[nn::crypto::HmacSha1Generator::BlockSize * 2];
    size_t     keySize;
    nn::Bit8   expectedMac[nn::crypto::HmacSha1Generator::MacSize];
    size_t     expectedMacSize;
};

/* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/HMAC_SHA1.pdf
   に記載されているテストベクトル */
const HmacSha1TestVector hmacSha1TestVectors[] =
{
    // #1) 鍵サイズ = ブロックサイズ
    {
        "Sample message for keylen=blocklen",
        34,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,},
        64,
        {0x5f, 0xd5, 0x96, 0xee, 0x78, 0xd5, 0x55, 0x3c, 0x8f, 0xf4, 0xe7, 0x2d, 0x26, 0x6d, 0xfd, 0x19,
         0x23, 0x66, 0xda, 0x29,},
        20,
    },

    // #2) 鍵サイズ < ブロックサイズ
    {
        "Sample message for keylen<blocklen",
        34,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13,},
        20,
        {0x4c, 0x99, 0xff, 0x0c, 0xb1, 0xb3, 0x1b, 0xd3, 0x3f, 0x84, 0x31, 0xdb, 0xaf, 0x4d, 0x17, 0xfc,
         0xd3, 0x56, 0xa8, 0x07,},
        20,
    },

    // #3) 鍵サイズ > ブロックサイズ
    {
        "Sample message for keylen=blocklen",
        34,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63,},
        100,
        {0x2d, 0x51, 0xb2, 0xf7, 0x75, 0x0e, 0x41, 0x05, 0x84, 0x66, 0x2e, 0x38, 0xf1, 0x33, 0x43, 0x5f,
         0x4c, 0x4f, 0xd4, 0x2a,},
        20,
    },

    // #4) 出力を truncate する
    {
        "Sample message for keylen<blocklen, with truncated tag",
        54,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30,},
        49,
        {0xfe, 0x35, 0x29, 0x56, 0x5c, 0xd8, 0xe2, 0x8c, 0x5f, 0xa7, 0x9e, 0xac,},
        12,
    },

};

const int TestVectorCount = sizeof(hmacSha1TestVectors) / sizeof(hmacSha1TestVectors[0]);

void HmacSha1BasicTest(const HmacSha1TestVector& testVector)
{
    nn::crypto::HmacSha1Generator hmacSha1;
    nn::Bit8                      mac[nn::crypto::HmacSha1Generator::MacSize];

    EXPECT_GE(sizeof(mac), nn::crypto::HmacSha1Generator::MacSize);

    hmacSha1.Initialize(testVector.key, testVector.keySize);
    hmacSha1.Update(testVector.data, testVector.dataSize);
    hmacSha1.GetMac(mac, sizeof(mac));

    EXPECT_ARRAY_EQ(mac, testVector.expectedMac, testVector.expectedMacSize);
}

void HmacSha1FunctionTest(const HmacSha1TestVector& testVector)
{
    nn::Bit8  mac[nn::crypto::HmacSha1Generator::MacSize];

    if (testVector.dataSize <= nn::crypto::HmacSha1Generator::BlockSize)
    {
        nn::crypto::GenerateHmacSha1Mac(mac,
                                        sizeof(mac),
                                        testVector.data,
                                        testVector.dataSize,
                                        testVector.key,
                                        testVector.keySize);

        EXPECT_ARRAY_EQ(mac, testVector.expectedMac, testVector.expectedMacSize);
    }
}

/**
  @brief   HmacSha1Generator クラスによる MAC 計算をテストします。

  @details
  NIST が公表しているテストベクトルを用いて、
  正しい MAC 値が計算されることを確認します。
 */
TEST(HmacSha1Test, Basic)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        HmacSha1BasicTest(hmacSha1TestVectors[i]);
    }
}

/**
  @brief   GenerateHmacSha1Mac 関数によるハッシュ計算をテストします。

  @details
  NIST が公表しているテストベクトルを用いて、
  正しいハッシュ値が計算されることを確認します。
 */
TEST(HmacSha1Test, FunctionInterface)
{
    // 短いテストベクトルについてのみテストする
    for (int i = 0; i < TestVectorCount; i++)
    {
        HmacSha1FunctionTest(hmacSha1TestVectors[i]);
    }
}

/**
  @brief   HmacSha1Generator クラスの状態遷移をテストします。
 */
TEST(HmacSha1Test, StateTransition)
{
    nn::crypto::HmacSha1Generator hmacSha1;
    nn::Bit8                      mac[nn::crypto::HmacSha1Generator::MacSize];
    nn::Bit8                      mac2[nn::crypto::HmacSha1Generator::MacSize];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化せずに Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(hmacSha1.Update(hmacSha1TestVectors[0].data, hmacSha1TestVectors[0].dataSize), "");

    // 初期化せずに GetMac が呼ばれても NG
    EXPECT_DEATH_IF_SUPPORTED(hmacSha1.GetMac(mac, sizeof(mac)), "");
#endif

    // 初期化
    hmacSha1.Initialize(hmacSha1TestVectors[0].key, hmacSha1TestVectors[0].keySize);

    // Update を呼ばずに GetMac を呼ぶのは大丈夫
    hmacSha1.GetMac(mac, sizeof(mac));

    // GetMac は連続で呼んでも大丈夫で、同じ値が出力されるはず
    hmacSha1.GetMac(mac2, sizeof(mac2));

    EXPECT_ARRAY_EQ(mac, mac2, nn::crypto::HmacSha1Generator::MacSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetMac の後に Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(hmacSha1.Update(hmacSha1TestVectors[0].data, hmacSha1TestVectors[0].dataSize), "");
#endif
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
TEST(HmacSha1Test, Destructor)
{
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };

    nn::crypto::HmacSha1Generator hmac;
    hmac.Initialize(key, sizeof(key));

    // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
    // Note:
    // nn::crypto::Sha1Generator がデストラクタでメモリクリアを行っているのでこのテストが有効
    // (ユーザーが指定する任意のハッシュ関数クラスがメモリクリアを行うとは限らない)
    EXPECT_ARRAY_NONZERO(&hmac, sizeof(hmac));
    hmac.~HmacGenerator();
    EXPECT_ARRAY_ZERO(&hmac, sizeof(hmac));
}
