﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/graphics/testGraphics_PerformanceProfileData.h>
#include <nn/util/util_FormatString.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

// Siglo 環境で rapidjson の警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/writer.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

namespace nnt{ namespace graphics{

//------------------------------------------------------------------------------------------
PerformanceProfileData::PerformanceProfileData()
    : m_pAllocator(NULL)
    , m_pDeallocator(NULL)
    , m_pName(NULL)
    , m_pDate(NULL)
    , m_pEnvironment(NULL)
    , m_pComment(NULL)
    , m_InitializationLoad(0)
{
    for (int i = 0; i < FrameValueTypeMax; ++i)
    {
        m_pFrameValues[i] = NULL;
        m_FrameValueCount[i] = 0;
        m_AvaiableElementCountMax[i] = 0;
        m_AverageValue[i] = 0;
    }
}

//------------------------------------------------------------------------------------------
PerformanceProfileData::~PerformanceProfileData()
{
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::Initialize(int maxDataCount, nn::AllocateFunction pAlloc, nn::DeallocateFunction pDealloc)
{
    m_pAllocator = pAlloc;
    m_pDeallocator = pDealloc;

    m_pName = static_cast<char*>((*m_pAllocator)(sizeof(char) * 1));
    m_pName[0] = '\0';
    m_pDate = static_cast<char*>((*m_pAllocator)(sizeof(char) * 1));
    m_pDate[0] = '\0';
    m_pEnvironment = static_cast<char*>((*m_pAllocator)(sizeof(char) * 1));
    m_pEnvironment[0] = '\0';
    m_pComment = static_cast<char*>((*m_pAllocator)(sizeof(char) * 1));
    m_pComment[0] = '\0';

    for (int i = 0; i < FrameValueTypeMax; ++i)
    {
        m_pFrameValues[i] = static_cast<FrameValue*>((*m_pAllocator)(sizeof(FrameValue) * maxDataCount));
        m_AvaiableElementCountMax[i] = maxDataCount;
    }
}

//-------------------------------------------------------------------------------------
bool PerformanceProfileData::Initialize(const char* pJsonText, nn::AllocateFunction pAlloc, nn::DeallocateFunction pDealloc)
{
    m_pAllocator = pAlloc;
    m_pDeallocator = pDealloc;

    rapidjson::Document doc;
    if (doc.Parse<0>(pJsonText).HasParseError())
    {
        NN_LOG("Parse json text failed.\n");
        return false;
    }

    // ヘッダを取得
    if (!doc.HasMember("header"))
    {
        return false;
    }

    // ヘッダ情報取得
    rapidjson::Value& header = doc["header"];

    ReadString(&m_pName, header, "name");
    ReadString(&m_pDate, header, "date");
    ReadString(&m_pEnvironment, header, "environment");
    ReadString(&m_pComment, header, "comment");

    ReadInt64(&m_AverageValue[FrameValueType_CpuLoad], header, "cpu_load_average");
    ReadInt64(&m_AverageValue[FrameValueType_GpuLoad], header, "gpu_load_average");
    ReadInt64(&m_AverageValue[FrameValueType_MemoryUsage], header, "memory_usage_average");
    ReadInt64(&m_InitializationLoad, header, "initialization_load");

    // フレームごとの情報を取得
    ReadFrameValue(&m_pFrameValues[FrameValueType_CpuLoad], &m_FrameValueCount[FrameValueType_CpuLoad], doc, "cpu_load");
    ReadFrameValue(&m_pFrameValues[FrameValueType_GpuLoad], &m_FrameValueCount[FrameValueType_GpuLoad], doc, "gpu_load");
    ReadFrameValue(&m_pFrameValues[FrameValueType_MemoryUsage], &m_FrameValueCount[FrameValueType_MemoryUsage], doc, "memory_max_usage");

    for (int i = 0; i < FrameValueTypeMax; ++i)
    {
        m_AvaiableElementCountMax[i] = m_FrameValueCount[i];
    }

    return true;
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::Finalize()
{
    NN_SDK_ASSERT_NOT_NULL(m_pDeallocator);
    for (int i = 0; i < FrameValueTypeMax; ++i)
    {
        (*m_pDeallocator)(m_pFrameValues[i], sizeof(FrameValue) * m_AvaiableElementCountMax[i]);
    }

    for (int i = 0; i < FrameValueTypeMax; ++i)
    {
        m_pFrameValues[i] = NULL;
        m_FrameValueCount[i] = 0;
    }

    if (m_pName != NULL)
    {
        (*m_pDeallocator)(m_pName, strlen(m_pName));
        m_pName = NULL;
    }
    if (m_pDate != NULL)
    {
        (*m_pDeallocator)(m_pDate, strlen(m_pDate));
        m_pDate = NULL;
    }
    if (m_pEnvironment != NULL)
    {
        (*m_pDeallocator)(m_pEnvironment, strlen(m_pEnvironment));
        m_pEnvironment = NULL;
    }
    if (m_pComment != NULL)
    {
        (*m_pDeallocator)(m_pComment, strlen(m_pComment));
        m_pComment = NULL;
    }
    m_InitializationLoad = 0;
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetName(const char* pName)
{
    NN_SDK_ASSERT_NOT_NULL(pName);

    if (m_pName != NULL)
    {
        (*m_pDeallocator)(m_pName, strlen(m_pName));
    }

    size_t length = strlen(pName);
    m_pName = static_cast<char*>((*m_pAllocator)(sizeof(char) * (length + 1)));
    strncpy(m_pName, pName, length);
    m_pName[length] = '\0';
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetDate(const char* pDate)
{
    NN_SDK_ASSERT_NOT_NULL(pDate);

    if (m_pDate != NULL)
    {
        (*m_pDeallocator)(m_pDate, strlen(m_pDate));
    }

    size_t length = strlen(pDate);
    m_pDate = static_cast<char*>((*m_pAllocator)(sizeof(char) * (length + 1)));
    strncpy(m_pDate , pDate, length);
    m_pDate[length] = '\0';
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetEnvironmentText(const char* pText)
{
    NN_SDK_ASSERT_NOT_NULL(pText);

    if (m_pEnvironment != NULL)
    {
        (*m_pDeallocator)(m_pEnvironment, strlen(m_pEnvironment));
    }

    size_t length = strlen(pText);
    m_pEnvironment = static_cast<char*>((*m_pAllocator)(sizeof(char) * (length + 1)));
    strncpy(m_pEnvironment, pText, length);
    m_pEnvironment[length] = '\0';
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetCommentText(const char* pText)
{
    NN_SDK_ASSERT_NOT_NULL(pText);

    if (m_pComment != NULL)
    {
        (*m_pDeallocator)(m_pComment, strlen(m_pComment));
    }

    size_t length = strlen(pText);
    m_pComment= static_cast<char*>((*m_pAllocator)(sizeof(char) * (length + 1)));
    strncpy(m_pComment, pText, length);
    m_pComment[length] = '\0';
}


//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetCpuValue(int frame, int64_t value)
{
    AddElement(FrameValueType_CpuLoad, frame, value);
};

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetGpuValue(int frame, int64_t value)
{
    AddElement(FrameValueType_GpuLoad, frame, value);
};

//------------------------------------------------------------------------------------------
void PerformanceProfileData::SetMemoryUsageValue(int frame, int64_t value)
{
    AddElement(FrameValueType_MemoryUsage, frame, value);
};

//------------------------------------------------------------------------------------------
void PerformanceProfileData::AddElement(FrameValueType type, int frame, int64_t value)
{
    NN_SDK_ASSERT_NOT_NULL(m_pFrameValues[type]);
    NN_SDK_ASSERT(m_FrameValueCount[type] < m_AvaiableElementCountMax[type]);

    m_pFrameValues[type][m_FrameValueCount[type]].SetFrame(frame);
    m_pFrameValues[type][m_FrameValueCount[type]].SetValue(value);
    m_FrameValueCount[type]++;

    // 平均値を更新する
    CalculateAverageValue(type);
}

//------------------------------------------------------------------------------------------
void PerformanceProfileData::CalculateAverageValue(FrameValueType type)
{
    if (m_FrameValueCount[type] > 0)
    {
        m_AverageValue[type] = 0;
        for (uint32_t i = 0; i < m_FrameValueCount[type]; ++i)
        {
            m_AverageValue[type] += m_pFrameValues[type][i].GetValue();
        }
        m_AverageValue[type] /= m_FrameValueCount[type];
    }
}

//------------------------------------------------------------------------------------------
bool PerformanceProfileData::Write(const char* pFilePath)
{
    rapidjson::Document doc;

    doc.SetObject();

    doc.AddMember("header", rapidjson::Value(rapidjson::kObjectType), doc.GetAllocator());

    doc["header"].AddMember("name", rapidjson::Value().SetString(m_pName, doc.GetAllocator()), doc.GetAllocator());
    doc["header"].AddMember("date", rapidjson::Value().SetString(m_pDate, doc.GetAllocator()), doc.GetAllocator());
    doc["header"].AddMember("environment", rapidjson::Value().SetString(m_pEnvironment, doc.GetAllocator()), doc.GetAllocator());
    doc["header"].AddMember("comment", rapidjson::Value().SetString(m_pComment, doc.GetAllocator()), doc.GetAllocator());
    doc["header"].AddMember("cpu_load_average", rapidjson::Value(m_AverageValue[FrameValueType_CpuLoad]), doc.GetAllocator());
    doc["header"].AddMember("gpu_load_average", rapidjson::Value(m_AverageValue[FrameValueType_GpuLoad]), doc.GetAllocator());
    doc["header"].AddMember("memory_usage_average", rapidjson::Value(m_AverageValue[FrameValueType_MemoryUsage]), doc.GetAllocator());
    doc["header"].AddMember("initialization_load", rapidjson::Value(m_InitializationLoad), doc.GetAllocator());

    doc.AddMember("cpu_load", rapidjson::Value().SetObject(), doc.GetAllocator());
    rapidjson::Value&   cpuLoad = doc["cpu_load"];
    cpuLoad.AddMember("values", rapidjson::Value(rapidjson::kArrayType), doc.GetAllocator());
    rapidjson::Value&   cpuLoadArray = cpuLoad["values"];

    if (m_FrameValueCount[FrameValueType_CpuLoad] > 0)
    {
        for (uint32_t i = 0; i < m_FrameValueCount[FrameValueType_CpuLoad]; ++i)
        {
            rapidjson::Value    frame;
            frame.SetObject();
            frame.AddMember("x", rapidjson::Value(m_pFrameValues[FrameValueType_CpuLoad][i].GetFrame()), doc.GetAllocator());
            frame.AddMember("y", rapidjson::Value(m_pFrameValues[FrameValueType_CpuLoad][i].GetValue()), doc.GetAllocator());
            cpuLoadArray.PushBack(frame, doc.GetAllocator());
        }
    }

    doc.AddMember("gpu_load", rapidjson::Value().SetObject(), doc.GetAllocator());
    rapidjson::Value&   gpuLoad = doc["gpu_load"];
    gpuLoad.AddMember("values", rapidjson::Value(rapidjson::kArrayType), doc.GetAllocator());
    rapidjson::Value&   gpuLoadArray = gpuLoad["values"];

    if (m_FrameValueCount[FrameValueType_GpuLoad] > 0)
    {
        for (uint32_t i = 0; i < m_FrameValueCount[FrameValueType_GpuLoad]; ++i)
        {
            rapidjson::Value    frame;
            frame.SetObject();
            frame.AddMember("x", rapidjson::Value(m_pFrameValues[FrameValueType_GpuLoad][i].GetFrame()), doc.GetAllocator());
            frame.AddMember("y", rapidjson::Value(m_pFrameValues[FrameValueType_GpuLoad][i].GetValue()), doc.GetAllocator());
            gpuLoadArray.PushBack(frame, doc.GetAllocator());
        }
    }

    doc.AddMember("memory_max_usage", rapidjson::Value().SetObject(), doc.GetAllocator());
    rapidjson::Value&   memoryUsage = doc["memory_max_usage"];
    memoryUsage.AddMember("values", rapidjson::Value(rapidjson::kArrayType), doc.GetAllocator());
    rapidjson::Value&   memoryUsageArray = memoryUsage["values"];

    if (m_FrameValueCount[FrameValueType_MemoryUsage] > 0)
    {
        for (uint32_t i = 0; i < m_FrameValueCount[FrameValueType_MemoryUsage]; ++i)
        {
            rapidjson::Value    frame;
            frame.SetObject();
            frame.AddMember("x", rapidjson::Value(m_pFrameValues[FrameValueType_MemoryUsage][i].GetFrame()), doc.GetAllocator());
            frame.AddMember("y", rapidjson::Value(m_pFrameValues[FrameValueType_MemoryUsage][i].GetValue()), doc.GetAllocator());
            memoryUsageArray.PushBack(frame, doc.GetAllocator());
        }
    }

    rapidjson::StringBuffer buffer;
    rapidjson::Writer<rapidjson::StringBuffer>  writer(buffer);

    doc.Accept(writer);

    nn::fs::FileHandle  FileHandle;
    nn::Result result;
    result = nn::fs::DeleteFile(pFilePath);
    NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
    result = nn::fs::CreateFile(pFilePath, 0);
    NN_ASSERT(result.IsSuccess());
    if (!result.IsSuccess())
    {
        return false;
    }
    result = nn::fs::OpenFile(&FileHandle, pFilePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    NN_ASSERT(result.IsSuccess());
    if (!result.IsSuccess())
    {
        return false;
    }
    result = nn::fs::WriteFile(FileHandle, 0, buffer.GetString(), buffer.GetSize(), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    NN_ASSERT(result.IsSuccess());
    if (!result.IsSuccess())
    {
        return false;
    }
    nn::fs::CloseFile(FileHandle);

    return true;
}

//-------------------------------------------------------------------------------------
void PerformanceProfileData::ReadString(char** ppWrite, rapidjson::Value& value, const char* pName)
{
    if (value.HasMember(pName))
    {
        size_t  strLen = value[pName].GetStringLength();
        *ppWrite = static_cast<char*>((*m_pAllocator)(strLen + 1));
        strncpy(*ppWrite, value[pName].GetString(), strLen);
        (*ppWrite)[strLen] = '\0';
    }
}

//-------------------------------------------------------------------------------------
void PerformanceProfileData::ReadInt64(int64_t* pWrite, rapidjson::Value& value, const char* pName)
{
    if (value.HasMember(pName))
    {
        *pWrite = value[pName].GetInt64();
    }
}

//-------------------------------------------------------------------------------------
void PerformanceProfileData::ReadFrameValue(FrameValue** ppWrite, uint32_t* pCount, rapidjson::Document& doc, const char* pRootTagName)
{
    if (doc.HasMember(pRootTagName))
    {
        rapidjson::Value&   load_info = doc[pRootTagName];
        if (load_info.HasMember("values") &&
            load_info["values"].IsArray())
        {
            rapidjson::Value&   values = load_info["values"];
            rapidjson::SizeType count = values.Size();

            *ppWrite = static_cast<FrameValue*>((*m_pAllocator)(sizeof(FrameValue) * count));
            *pCount = count;

            for (rapidjson::SizeType i = 0; i < count; ++i)
            {
                if (values[i].HasMember("x"))
                {
                    rapidjson::Value&   x = values[i]["x"];
                    (*ppWrite)[i].SetFrame(x.GetUint());
                }
                if (values[i].HasMember("y"))
                {
                    rapidjson::Value&   y = values[i]["y"];
                    (*ppWrite)[i].SetValue(y.GetInt64());
                }
            }
        }
    }
}


}}

