﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Control/Fx3Control.h"

namespace nnt {
namespace usb {
namespace cdmsc {

const int DataSize = 1024;

extern bool g_DownloadFirmware;

TEST(Fx3MscTestSuite, SoftReset2_1)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    uint16_t countData = 0;
    int countMassStorageResetCommands = 0;
    int logIndex = 0;
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = 0x01 << 8; // Bulk out endpoint
    commandInfo |= Fx3SetConfigOptionsEnableFlag;
    commandInfo |= Fx3SetConfigOptionsStallFlag;
    commandInfo |= Fx3SetConfigOptionsCountFlag;
    countData = 2;

    Fx3SendVendorSpecificRequest(
        NULL,
        0x40,
        Fx3Request_SetConfigOptions,
        commandInfo,
        countData,
        0x00
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // get command logs and check MassStorageReset was called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );
    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_MassStorageReset)
        {
            countMassStorageResetCommands++;
        }
    }

    EXPECT_EQ(countData, countMassStorageResetCommands);
}


TEST(Fx3MscTestSuite, TestUnitReady5_1)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure Fx3 to not be ready for ~10 seconds
    // TestUnitReady completes with sense code NOT READY 2-4-1 every time for ~10 seconds
    // After elasped time complete successfully
    commandInfo = Fx3MscCommand_TestUnitReady << 8;

    testResponse.Type = Fx3MscTestResponseType_NakDataTransportForXMilliseconds;
    testResponse.u.NakTime.Time = 10000;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);
}


TEST(Fx3MscTestSuite, TestUnitReady5_2)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    int logIndex = 0;
    bool hasStartStopUnit = false;
    NN_USB_DMA_ALIGN Fx3TestResponse responseCountData = { 0 };
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure Fx3 to requrie a StartStopUnit
    // TestUnitReady completes with sense code NOT READY 2-4-2
    // Expect a StartStopUnit complete successfully
    // Then a TestUnitReady complete successfully
    commandInfo = Fx3MscCommand_TestUnitReady << 8;

    responseCountData.Type = Fx3MscTestResponseType_StallAndSetSenseCodeTableIndex;
    responseCountData.u.SenseCodeTableIndex.Index = Fx3MscSenseCodesIndex_LOGICAL_DRIVE_NOT_READY_INITIALIZATION_REQUIRED;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&responseCountData,
        0x40,
          Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(responseCountData)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);

    static NN_ALIGNAS(4096) char g_Fx3StringBuf[128] = { 0 };
    sprintf(g_Fx3StringBuf, "\n**Waiting for probe");
    Fx3PrintString((uint8_t*)g_Fx3StringBuf, sizeof(g_Fx3StringBuf));
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    sprintf(g_Fx3StringBuf, "\n**Done waiting for probe");
    Fx3PrintString((uint8_t*)g_Fx3StringBuf, sizeof(g_Fx3StringBuf));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // Get Command log and check for StartStopUnit cmd
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );

    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_StartStopUnit)
        {
            hasStartStopUnit = true;
        }
    }

    EXPECT_TRUE(hasStartStopUnit);
}


TEST(Fx3MscTestSuite, ReadCapacity6_1)
{
    nn::cdmsc::UnitProfile profile;
    nn::Result result = nn::ResultSuccess();
    uint16_t testField = 0;
    NN_USB_DMA_ALIGN uint32_t unsupportedReadCapacity = 0xFF;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure Fx3 to return unsupported block size for ReadCapacity
    testField = Fx3TestField_ReadCapacity;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&unsupportedReadCapacity,
        0x40,
        Fx3Request_SetFieldValue,
        testField,
        0x00,
        sizeof(unsupportedReadCapacity)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    result = Fx3Probe(&profile);
    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultDeviceNotAvailable, result);

    // Expect a hard reset: Reconnect Control interface
    Fx3Reconnect(WAIT_SECONDS_FOR_ATTACH);

}

// Note: ModeSense7_2 has executed before ModSense7_1 - Cdmsc Driver remembers not to issue mode sense cmd.
TEST(Fx3MscTestSuite, ModeSense7_2)
{
    nn::cdmsc::UnitProfile profile;
    nn::Result result = nn::ResultSuccess();
    uint16_t testField = 0;
    uint32_t block = 1;
    NN_USB_DMA_ALIGN uint32_t writeProtectValue = 0;
    NN_USB_DMA_ALIGN uint8_t writeData[DataSize] = "ModeSense 7.2 test!";

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure Fx3 to report device is write protected from ModeSense
    testField = Fx3TestField_WriteProtect;
    writeProtectValue = 0x01;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&writeProtectValue,
        0x40,
        Fx3Request_SetFieldValue,
        testField,
        0x00,
        sizeof(writeProtectValue)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }

    // Expect to return ResultWriteProtected
    result = nn::cdmsc::Write(writeData, profile.handle, 0, block);
    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultWriteProtected, result);
}


TEST(Fx3MscTestSuite, ModeSense7_1)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    uint16_t logType = 0;
    uint32_t block = 1;
    int logIndex = 0;
    int countModeSenseCommands = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t writeData[DataSize] = "ModeSense 7.1 test!";
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure Fx3 to return sense code ILLEGAL Request 5-20-00
    commandInfo = Fx3MscCommand_ModeSense10 << 8;

     testResponse.Type = Fx3MscTestResponseType_StallAndSetSenseCodeTableIndex;
     testResponse.u.SenseCodeTableIndex.Index = Fx3MscSenseCodesIndex_INVALID_COMMAND_OPERATION_CODE;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // clear command logs
    logType = Fx3ClearCommandLog << 8;

    Fx3SendVendorSpecificRequest(
        nullptr,
        0x40,
        Fx3Request_ClearLog,
        logType,
        0x00,
        0x00
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // get command logs and check ModeSense was not called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );

    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        //FX3_LOG("Command: 0x%x\n", readLogData[logIndex]);
        if (readLogData[logIndex] == Fx3MscCommand_ModeSense10)
        {
            countModeSenseCommands++;
        }
    }

    EXPECT_EQ(0, countModeSenseCommands);

    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::cdmsc::Write(writeData, profile.handle, 0, block));
}


TEST(Fx3MscTestSuite, SyncCache8_0)
{
    nn::cdmsc::UnitProfile profile;
    nn::Result result = nn::ResultSuccess();
    uint16_t commandInfo = 0;
    uint16_t logType = 0;
    int countSyncCacheCommands = 0;
    int logIndex = 0;
    uint32_t block = 1;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    // Configure fx3 to return sense code ILLEGAL REQUEST 5-20-00 from SyncCache
    commandInfo = Fx3MscCommand_SyncCache10 << 8;
    testResponse.Type = Fx3MscTestResponseType_StallAndSetSenseCodeTableIndex;
    testResponse.u.SenseCodeTableIndex.Index = Fx3MscSenseCodesIndex_INVALID_COMMAND_OPERATION_CODE;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
         Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // Flush() returns UnsupportedOpperation
    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }
    result = nn::cdmsc::Flush(profile.handle, 0, block);
    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultUnsupportedOperation, result);

    // clear command logs
    logType = Fx3ClearCommandLog << 8;

    Fx3SendVendorSpecificRequest(
        nullptr,
        0x40,
        Fx3Request_ClearLog,
        logType,
        0x00,
        0x00
    );

    // Fx3 Disconnects and reconnects - SyncCache should not be issued
    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);

    // get command logs and check SynCache10 was not called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );

    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_SyncCache10)
        {
            countSyncCacheCommands++;
        }
    }

    EXPECT_EQ(0, countSyncCacheCommands);

    // Flush() returns UnsupportedOpperation
    result = nn::cdmsc::Flush(profile.handle, 0, block);
    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultUnsupportedOperation, result);
}

TEST(Fx3MscTestSuite, DummyRead9_0)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    testResponse.Type = Fx3MscTestResponseType_NakDataTransportForXMilliseconds;
    testResponse.u.NakTime.Time = 10000;

    // Configure Fx3 to NAK the data phase of first read for 10 seconds
    commandInfo = Fx3MscCommand_Read10 << 8;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0x00,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));
    FX3_LOG("Vid: %d, PID:%d\n", profile.vid, profile.pid);
}


} // cdmsc
} // usb
} // nnt
