﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nnt.h>
#include <nnt/usb/testUsb_Fx3Utility.h>

#include <nn/usb/usb_Host.h>
#include <nn/usb/usb_Result.public.h>
#include <nn/cdmsc/cdmsc_Types.h>
#include <nn/cdmsc.h>

#include "../Common/CdmscTestSuiteFx3Firmware.h"

namespace nnt {
namespace usb {
namespace cdmsc {


#define FX3_LOG NN_LOG

///////////////////////////////////////////////////////////////////////////////
// Attach timeout

enum
{
    WAIT_SECONDS_FOR_ATTACH = 60,
};

const int Fx3MscVID                  = 0x04B4;
const int Fx3MscPID                  = 0x000B;
const int Fx3MscMaxControlBufSize    = 0x10000;
const int Fx3MscMaxTransferSize      = 2 * 1024 * 1024;
const int Fx3CommandLogLength        = 64;

const uint8_t Fx3TimedResponseFlag       = 0 << 0;
const uint8_t Fx3CountResponseFlag       = 1 << 0;
const uint8_t Fx3NakResponseFlag         = 0 << 1;
const uint8_t Fx3StallResponseFlag       = 1 << 1;

const uint8_t Fx3SetConfigOptionsEnableFlag     = 1 << 0;
const uint8_t Fx3SetConfigOptionsDisableFlag    = 0 << 0;
const uint8_t Fx3SetConfigOptionsNakFlag        = 0 << 1;
const uint8_t Fx3SetConfigOptionsStallFlag      = 1 << 1;
const uint8_t Fx3SetConfigOptionsTimedFlag      = 0 << 2;
const uint8_t Fx3SetConfigOptionsCountFlag      = 1 << 2;

const uint8_t Fx3ClearCommandLog                = 0 << 0;
const uint8_t Fx3ClearErrorLog                  = 1 << 0;


///////////////////////////////////////////////////////////////////////////////
// Vendor request definitions (commands to FW)
// Documented here: http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=199816802
enum Fx3Request
{
    // USB Test Suite Requests
    Fx3Request_Reset                    = 0x00,
    Fx3Request_PrintString              = 0x03,
    Fx3Request_ColdReset                = 0x07,

    // Mass Storage Class Specifc Requests
    Fx3Request_SetFieldValue            = 0x10,
    Fx3Request_StartDisconnectTest      = 0x15,
    Fx3Request_GetCommandLog            = 0x16,
    Fx3Request_ClearLog                 = 0x17,
    Fx3Request_GetErrorLog              = 0x18,
    Fx3Request_SetConfigOptions         = 0x19,
    Fx3Request_SetTestResponse          = 0x1a,

    Fx3Request_FirmwareVersion          = 0xff,
};

enum Fx3TestField {
    Fx3TestField_GetMaxLun = 0x00,
    Fx3TestField_WriteProtect,
    Fx3TestField_ReadCapacity,
    Fx3TestField_MaxMscTestFieldValues
};

enum Fx3MscCommand {
    Fx3MscCommand_TestUnitReady     = 0x00,
    Fx3MscCommand_RequestSense      = 0x03,
    Fx3MscCommand_ReadCapacity10    = 0x25,
    Fx3MscCommand_ReadCapacity16    = 0x9e,
    Fx3MscCommand_ModeSense10       = 0x5a,
    Fx3MscCommand_StartStopUnit     = 0x1b,
    Fx3MscCommand_SyncCache10       = 0x35,
    Fx3MscCommand_Read10            = 0x28,
    Fx3MscCommand_Write10           = 0x2a,
    Fx3MscCommand_Read16            = 0x88,
    Fx3MscCommand_Write16           = 0x8a,
    Fx3MscCommand_GetMaxLun         = 0xFE, // Ref [BOT r1.0] 3.2
    Fx3MscCommand_MassStorageReset  = 0xFF, // Ref [BOT r1.0] 3.1
};

enum Fx3MscTestResponseType {
    Fx3MscTestResponseType_StallNumberOfTimes = 0,
    Fx3MscTestResponseType_NakDataTransportForXMilliseconds,
    Fx3MscTestResponseType_StallAndSetSenseCodeTableIndex,
    Fx3MscTestResponseType_FailSetStatusResidue,
    Fx3MscTestResponseType_RwBytesMismatch,
};

enum Fx3MscSenseCodesIndex {
    Fx3MscSenseCodesIndex_NO_SENSE = 0,
    Fx3MscSenseCodesIndex_RECOVERED_DATA_WITH_RETRIES,
    Fx3MscSenseCodesIndex_RECOVERED_DATA_WITH_ECC,
    Fx3MscSenseCodesIndex_LOGICAL_DRIVE_NOT_READY_BECOMING_READY,
    Fx3MscSenseCodesIndex_LOGICAL_DRIVE_NOT_READY_INITIALIZATION_REQUIRED,
    Fx3MscSenseCodesIndex_LOGICAL_UNIT_NOT_READY_FORMAT_IN_PROGRESS,
    Fx3MscSenseCodesIndex_LOGICAL_DRIVE_NOT_READY_DEVICE_IS_BUSY,
    Fx3MscSenseCodesIndex_NO_REFERENCE_POSITION_FOUND,
    Fx3MscSenseCodesIndex_LOGICAL_UNIT_COMMUNICATION_FAILURE,
    Fx3MscSenseCodesIndex_LOGICAL_UNIT_COMMUNICATION_TIMEOUT,
    Fx3MscSenseCodesIndex_LOGICAL_UNIT_COMMUNICATION_OVERRUN,
    Fx3MscSenseCodesIndex_MEDIUM_NOT_PRESENT,
    Fx3MscSenseCodesIndex_USB_TO_HOST_SYSTEM_INTERFACE_VALUE,
    Fx3MscSenseCodesIndex_INSUFFICIENT_RESOURCES,
    Fx3MscSenseCodesIndex_UNKNOWN_ERROR,
    Fx3MscSenseCodesIndex_NO_SEEK_COMPLETE,
    Fx3MscSenseCodesIndex_WRITE_FAULT,
    Fx3MscSenseCodesIndex_ID_CRC_ERROR,
    Fx3MscSenseCodesIndex_UNRECOVERED_READ_ERROR,
    Fx3MscSenseCodesIndex_ADDRESS_MARK_NOT_FOUND_FOR_ID_FIELD,
    Fx3MscSenseCodesIndex_ADDRESS_MARK_NOT_FOUND_FOR_DATA_FIELD,
    Fx3MscSenseCodesIndex_RECORDED_ENTITY_NOT_FOUND,
    Fx3MscSenseCodesIndex_CANNOT_READ_MEDIUM_UNKNOWN_FORMAT,
    Fx3MscSenseCodesIndex_FORMAT_COMMAND_FAILED,
    Fx3MscSenseCodesIndex_NN_DIAGNOSTIC_FAILURE_ON_COMPONENT_NN,
    Fx3MscSenseCodesIndex_PARAMETER_LIST_LENGTH_ERROR,
    Fx3MscSenseCodesIndex_INVALID_COMMAND_OPERATION_CODE,
    Fx3MscSenseCodesIndex_LOGICAL_BLOCK_ADDRESS_OUT_OF_RANGE,
    Fx3MscSenseCodesIndex_INVALID_FIELD_IN_COMMAND_PACKET,
    Fx3MscSenseCodesIndex_LOGICAL_UNIT_NOT_SUPPORTED,
    Fx3MscSenseCodesIndex_INVALID_VIELD_IN_PARAMETER_LIST,
    Fx3MscSenseCodesIndex_PARAMETER_NOT_SUPPORTED,
    Fx3MscSenseCodesIndex_PARAMETER_VALUE_INVALID,
    Fx3MscSenseCodesIndex_SAVING_PARAMETERS_NOT_SUPPORT,
    Fx3MscSenseCodesIndex_NOT_READY_TO_READY_TRANSITION_MEDIA_CHANGED,
    Fx3MscSenseCodesIndex_POWER_ON_RESET_OR_BUS_DEVICE_RESET_OCCURRED,
    Fx3MscSenseCodesIndex_COMMANDS_CLEARED_BY_ANOTHER_INITIATOR,
    Fx3MscSenseCodesIndex_WRITE_PROTECTED_MEDIA,
    Fx3MscSenseCodesIndex_OVERLAPPED_COMMAND_ABORTED,
};

#pragma pack(push, 1)
struct Fx3SetCommandResponseData
{
    uint8_t         flags;
    uint32_t        data;
};
#pragma pack(pop)

#pragma pack(push, 1)

// StallNumberOfTimes
struct Fx3StallTimes {
    uint32_t Count;
};

// NakDataTransportForXMilliseconds
struct Fx3NakTime {
    uint32_t Time;
};

//StallAndSetSenseCodeTableIndex
struct Fx3SetSenseCodeTableIndex {
    uint8_t Index;
};

//FailSetStatusResidue
struct Fx3SetStatusAndResidue {
    uint8_t  Status;
    uint32_t Residue;
};

//RwMismatch
struct Fx3RwBytesMismatch {
    uint8_t AddBytesForRead;
    uint32_t NumberOfBytes;
};

struct Fx3TestResponse {
    uint8_t  Type;
    union {
        uint32_t Lba32;
        uint64_t Lba64;
    } a;
    union {
        struct Fx3StallTimes StallCount;
        struct Fx3NakTime NakTime;
        struct Fx3SetSenseCodeTableIndex SenseCodeTableIndex;
        struct Fx3SetStatusAndResidue FailAndSetStatusResidue;
        struct Fx3RwBytesMismatch;
    } u;
};

#pragma pack(pop)


///////////////////////////////////////////////////////////////////////////////
// Initialize HS client
void Fx3Initialize(int seconds, bool installFw);

bool IsFx3Initialized();

///////////////////////////////////////////////////////////////////////////////
// Finalize HS client
void Fx3Finalize();

///////////////////////////////////////////////////////////////////////////////
// Recover and reconncect device from failing test cases
void Fx3Reconnect(int seconds);

bool Fx3Attach(int seconds);

///////////////////////////////////////////////////////////////////////////////
// Reset Cdmsc Fx3's control interface
void Fx3Reset(int seconds, bool installFw);

///////////////////////////////////////////////////////////////////////////////
// Disconnect and re-attach to apply MSC behaviour
void Fx3ActivateMassStorageMode(int seconds);

///////////////////////////////////////////////////////////////////////////////
// Poll for Mass Storage interface to be used
nn::Result Fx3Probe(nn::cdmsc::UnitProfile *pOutProfile);

bool Fx3WaitForDeviceDisconnect(int seconds);

///////////////////////////////////////////////////////////////////////////////
// Configure Fx3 behaviour for testing specific cases
void Fx3SendVendorSpecificRequest(uint8_t *pData, uint8_t bmRequest, Fx3Request request, uint16_t wValue, uint16_t wIndex, uint16_t wLength);


///////////////////////////////////////////////////////////////////////////////
// Print string on FX3 console
void Fx3PrintString(uint8_t *pStringData, uint32_t size);

} // namespace: cdmsc
} // namespace: usb
} // namespace: nnt
