﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Optional.h>
#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ParseMakerNote.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptMakerNote.h"
#include "detail/testCapsrv_TestMakerNote.h"

#define NNT_CAPSRV_TEST_REMOVE_PAREN(...) __VA_ARGS__
#define NNT_CAPSRV_TEST_MAKERNOTEPARSER_SUCCESS(Bytes)  \
    {                                                               \
        nn::capsrv::server::detail::MakerNoteInfo info = {};        \
        uint8_t input[] = { NNT_CAPSRV_TEST_REMOVE_PAREN Bytes };  \
        auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input, sizeof(input));   \
        EXPECT_TRUE(result.IsSuccess());          \
    }
#define NNT_CAPSRV_TEST_MAKERNOTEPARSER(Bytes, ExpectedResultType)  \
    {                                                               \
        nn::capsrv::server::detail::MakerNoteInfo info = {};        \
        uint8_t input[] = { NNT_CAPSRV_TEST_REMOVE_PAREN Bytes };  \
        auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input, sizeof(input));   \
        EXPECT_TRUE(ExpectedResultType::Includes(result));          \
    }

namespace {

#define NNT_CAPSRV_DEFINE_ENTRY_BYTES(NAME) \
    const char g_Name ## NAME [] = NNT_CAPSRV_MAKERNOTE_STRINGIZE(NAME);    \
    const uint8_t g_BytesOk    ## NAME [] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRY_OK(NAME) };   \
    const uint8_t g_BytesShort ## NAME [] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRY_SHORT(NAME) };   \
    const uint8_t g_BytesLong  ## NAME [] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRY_LONG(NAME) };

    NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_ALL( NNT_CAPSRV_DEFINE_ENTRY_BYTES );


    struct EntryVariation
    {
        const char* name;
        std::string bytesOk;
        std::string bytesShort;
        std::string bytesLong;
        bool isOptional;
    };

#define NNT_CAPSRV_DEFINE_MAKEENTRYVARIATION(NAME)              \
    EntryVariation MakeEntryVariation ## NAME () NN_NOEXCEPT    \
    {                                                           \
        EntryVariation v = {};                                  \
        v.name = g_Name ## NAME;                                \
        v.bytesOk = std::string(g_BytesOk ## NAME, g_BytesOk ## NAME + sizeof(g_BytesOk ## NAME));    \
        v.bytesShort = std::string(g_BytesShort ## NAME, g_BytesShort ## NAME + sizeof(g_BytesShort ## NAME));    \
        v.bytesLong = std::string(g_BytesLong ## NAME, g_BytesLong ## NAME + sizeof(g_BytesLong ## NAME));    \
        v.isOptional = NNT_CAPSRV_MAKERNOTE_IS_OPTIONAL_ ## NAME;    \
        return v;                                               \
    }

#define NNT_CAPSRV_MAKE_VARIATIONLISTENTRY(NAME) MakeEntryVariation ## NAME(),

    NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_ALL( NNT_CAPSRV_DEFINE_MAKEENTRYVARIATION );

    struct TestEntryOption
    {
        // 暗号化する場合、開始オフセット。
        nn::util::optional<int> encryptionOffset;
        nn::util::optional<int> encryptionUnitSize;
        // 入力の長さを固定する場合、その長さ。
        nn::util::optional<size_t> fixedInputLength;

        void Apply(std::string& buf) const NN_NOEXCEPT
        {
            if(this->fixedInputLength)
            {
                buf.resize(*this->fixedInputLength, 0);
            }

            if(this->encryptionOffset && this->encryptionUnitSize)
            {
                int offset = *this->encryptionOffset;
                int unit = *this->encryptionUnitSize;
                size_t newSize = ((buf.size() - *this->encryptionOffset + unit - 1) / unit) * unit + offset;
                buf.resize(newSize, 0);
            }

            if(this->encryptionOffset)
            {
                nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(&buf[0], buf.size(), *this->encryptionOffset);
            }
        }
    };

    void TestEntryVariation(const std::string& bytesVersion, const EntryVariation* pEntryVariationList, int entryVariationCount, const TestEntryOption& opt) NN_NOEXCEPT
    {
        // 正常
        // この関数自体のテストも兼ねている
        NN_LOG("Checking ok EntryList ...\n");
        {
            std::string input;
            input += bytesVersion;
            for(int k = 0; k < entryVariationCount; k++)
            {
                auto& e = pEntryVariationList[k];
                input += e.bytesOk;
            }
            opt.Apply(input);
            NN_LOG("    input %d bytes\n", input.size());
            nn::capsrv::server::detail::MakerNoteInfo info = {};
            auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size());
            EXPECT_TRUE(result.IsSuccess());
        }

        // エントリーが短い
        NN_LOG("Checking short entry ...\n");
        for(int i = 0; i < entryVariationCount; i++)
        {
            auto& target = pEntryVariationList[i];
            NN_LOG("  Checking %s\n", target.name);

            std::string input;
            input += bytesVersion;
            for(int k = 0; k < entryVariationCount; k++)
            {
                auto& e = pEntryVariationList[k];
                if(i == k)
                {
                    input += target.bytesShort;
                }
                else
                {
                    input += e.bytesOk;
                }
            }
            opt.Apply(input);
            NN_LOG("    input %d bytes\n", input.size());
            nn::capsrv::server::detail::MakerNoteInfo info = {};
            auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size());
            EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
        }

        // エントリーが長い
        NN_LOG("Checking long entry ...\n");
        for(int i = 0; i < entryVariationCount; i++)
        {
            auto& target = pEntryVariationList[i];
            NN_LOG("  Checking %s\n", target.name);

            std::string input;
            input += bytesVersion;
            for(int k = 0; k < entryVariationCount; k++)
            {
                auto& e = pEntryVariationList[k];
                if(i == k)
                {
                    input += target.bytesLong;
                }
                else
                {
                    input += e.bytesOk;
                }
            }
            opt.Apply(input);
            NN_LOG("    input %d bytes\n", input.size());
            nn::capsrv::server::detail::MakerNoteInfo info = {};
            auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size());
            EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
        }

        // エントリーの欠損
        NN_LOG("Checking lacking entry ...\n");
        for(int i = 0; i < entryVariationCount; i++)
        {
            auto& target = pEntryVariationList[i];

            NN_LOG("  Checking %s\n", target.name);

            std::string input;
            input += bytesVersion;
            for(int k = 0; k < entryVariationCount; k++)
            {
                auto& e = pEntryVariationList[k];
                if(i != k)
                {
                    input += e.bytesOk;
                }
            }
            opt.Apply(input);
            NN_LOG("    input %d bytes\n", input.size());
            nn::capsrv::server::detail::MakerNoteInfo info = {};
            if(!target.isOptional)
            {
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                    nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size())
                );
            }
            else
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size()));
            }
        }

        // エントリーの重複
        NN_LOG("Checking duplicated entry ...\n");
        for(int i = 0; i < entryVariationCount; i++)
        {
            auto& target = pEntryVariationList[i];
            NN_LOG("  Checking %s\n", target.name);

            std::string input;
            input += bytesVersion;
            for(int k = 0; k < entryVariationCount; k++)
            {
                auto& e = pEntryVariationList[k];
                if(i == k)
                {
                    input += e.bytesOk;
                }
                input += e.bytesOk;
            }
            opt.Apply(input);
            NN_LOG("    input %d bytes\n", input.size());
            nn::capsrv::server::detail::MakerNoteInfo info = {};
            auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size());
            EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
        }

        // エントリーの順番入れ替え
        NN_LOG("Checking swapped entry ...\n");
        for(int i = 0; i < entryVariationCount; i++)
        {
            auto& target0 = pEntryVariationList[i];
            for(int j = i + 1; j < entryVariationCount; j++)
            {
                auto& target1 = pEntryVariationList[j];
                NN_LOG("  Checking %s <-> %s\n", target0.name, target1.name);

                std::string input;
                input += bytesVersion;
                for(int k = 0; k < entryVariationCount; k++)
                {
                    auto& e = pEntryVariationList[k];
                    if(k == i)
                    {
                        input += target1.bytesOk;
                    }
                    else if(k == j)
                    {
                        input += target0.bytesOk;
                    }
                    else
                    {
                        input += e.bytesOk;
                    }
                }
                opt.Apply(input);
                NN_LOG("    input %d bytes\n", input.size());
                nn::capsrv::server::detail::MakerNoteInfo info = {};
                auto result = nn::capsrv::server::detail::TryParseMakerNote(&info, input.data(), input.size());
                EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
            }
        }
    }// NOLINT(impl/function_size)
}

TEST(UnitTest, ParseMakerNote_Structure)
{
    // 長さが短い
    NN_LOG("Too short input for Version ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION_SHORT),
        nn::capsrv::ResultAlbumInvalidFileData
    );

    //---------------------------
    // Entry の解析チェック
    //---------------------------
    NN_LOG("Too short input for EntryTag ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYTAG_SHORT),
        nn::capsrv::ResultAlbumInvalidFileData
    );
    NN_LOG("Too short input for EntrySize ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYTAG_SIGNATURE
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYSIZE_SHORT),
        nn::capsrv::ResultAlbumInvalidFileData
    );
    NN_LOG("Invalid EntrySize(zero) ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYTAG_SIGNATURE
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYSIZE_ZERO),
        nn::capsrv::ResultAlbumInvalidFileData
    );
    NN_LOG("Invalid EntrySize(huge) ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYTAG_SIGNATURE
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYSIZE_ZERO),
        nn::capsrv::ResultAlbumInvalidFileData
    );

}

TEST(UnitTest, ParseMakerNote_Version0)
{
    using namespace nn::capsrv::server::detail;

    // 正しくパースできるテスト
    NN_LOG("Checking good input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION0
        };

        uint8_t expectedSignature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
        uint8_t expectedAppId[]     = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
        auto result = TryParseMakerNote(&info, input, sizeof(input));
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(MakerNoteVersion_Version0, info.version);
        EXPECT_TRUE(0 == std::memcmp(expectedSignature, &info.signature, sizeof(expectedSignature)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppId, &info.encryptedApplicationId, sizeof(expectedAppId)));
    }

    // バージョン不一致は失敗
    NN_LOG("Checking future entry list ...\n");
    NNT_CAPSRV_TEST_MAKERNOTEPARSER((
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1),
        nn::capsrv::ResultAlbumInvalidFileData
    );


    // エントリーの解析チェック
    const char version[] = { NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0 };
    std::string bytesVersion(version, version + sizeof(version));
    EntryVariation entryVariationList[] = {
        NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_VERSION0( NNT_CAPSRV_MAKE_VARIATIONLISTENTRY )
    };

    int entryVariationCount = sizeof(entryVariationList) / sizeof(entryVariationList[0]);

    TestEntryVariation(bytesVersion, entryVariationList, entryVariationCount, {});
}

TEST(UnitTest, ParseMakerNote_Version1)
{
    using namespace nn::capsrv::server::detail;

    static const int EncryptionHeadOffset = 24;

    // 正しくパースできるテスト
    size_t okLength = 0;
    NN_LOG("Checking good input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION1
        };
        okLength = sizeof(input);

        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);

        uint8_t expectedSignature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
        uint8_t expectedAppId[]     = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
        uint8_t expectedDateTime[]  = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
        uint8_t expectedAppletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
        auto result = TryParseMakerNote(&info, input, sizeof(input));
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(MakerNoteVersion_Version1, info.version);
        EXPECT_TRUE(0 == std::memcmp(expectedSignature, &info.signature, sizeof(expectedSignature)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppId, &info.encryptedApplicationId, sizeof(expectedAppId)));
        EXPECT_TRUE(0 == std::memcmp(expectedDateTime, &info.dateTime, sizeof(expectedDateTime)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppletData, &info.appletData, sizeof(expectedAppletData)));
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH, info.dataWidth);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT, info.dataHeight);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT, info.dataOrientation);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT, info.frameCount);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER, info.frameRateNumerator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM, info.frameRateDenominator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION,   info.dataDurationMilliseconds);
    }

    //// バージョン不一致は失敗
    //NN_LOG("Checking future entry list ...\n");
    //NNT_CAPSRV_TEST_MAKERNOTEPARSER((
    //    NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
    //    NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2),
    //    nn::capsrv::ResultAlbumInvalidFileData
    //);


    // 難読化されていない入力は失敗
    NN_LOG("Checking raw input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION1
        };
        //nn::capsrv::server::detail::EncryptMakerNoteInplace(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }

    // パディングが 0 になっていない入力は失敗
    NN_LOG("Checking bad padding...\n");
    {
        NN_LOG("  case 1\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG1_VERSION1
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }
    {
        NN_LOG("  case 2\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG2_VERSION1
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }


    // エントリーの解析チェック
    const char version[] = { NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1 };
    std::string bytesVersion(version, version + sizeof(version));
    EntryVariation entryVariationList[] = {
        NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_VERSION1( NNT_CAPSRV_MAKE_VARIATIONLISTENTRY )
    };

    int entryVariationCount = sizeof(entryVariationList) / sizeof(entryVariationList[0]);

    {
        TestEntryOption opt = {};
        opt.encryptionOffset = EncryptionHeadOffset;
        opt.fixedInputLength = okLength;
        TestEntryVariation(bytesVersion, entryVariationList, entryVariationCount, opt);
    }
}

TEST(UnitTest, ParseMakerNote_Version2)
{
    using namespace nn::capsrv::server::detail;

    static const int EncryptionHeadOffset = 24;

    // 正しくパースできるテスト
    size_t okLength = 0;
    NN_LOG("Checking good input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION2
        };
        okLength = sizeof(input);

        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);

        uint8_t expectedSignature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
        uint8_t expectedAppId[]     = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
        uint8_t expectedDateTime[]  = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
        uint8_t expectedAppletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
        uint8_t expectedApplicationData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLIDATA };
        NN_ABORT_UNLESS_RESULT_SUCCESS( TryParseMakerNote(&info, input, sizeof(input)) );
        EXPECT_EQ(MakerNoteVersion_Version2, info.version);
        EXPECT_TRUE(0 == std::memcmp(expectedSignature, &info.signature, sizeof(expectedSignature)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppId, &info.encryptedApplicationId, sizeof(expectedAppId)));
        EXPECT_TRUE(0 == std::memcmp(expectedDateTime, &info.dateTime, sizeof(expectedDateTime)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppletData, &info.appletData, sizeof(expectedAppletData)));
        EXPECT_TRUE(0 == std::memcmp(expectedApplicationData, &info.applicationData, sizeof(expectedApplicationData)));
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH, info.dataWidth);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT, info.dataHeight);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT, info.dataOrientation);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT, info.frameCount);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER, info.frameRateNumerator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM, info.frameRateDenominator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION,   info.dataDurationMilliseconds);

        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_COPYRIGHT, info.isCopyrightComposited);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDIT,    info.hasUneditableArea);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITX,   info.uneditableAreaCoordinateX);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITY,   info.uneditableAreaCoordinateY);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITWIDTH, info.uneditableAreaWidth);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITHEIGHT, info.uneditableAreaHeight);
    }

    //// バージョン不一致は失敗
    //NN_LOG("Checking future entry list ...\n");
    //NNT_CAPSRV_TEST_MAKERNOTEPARSER((
    //    NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
    //    NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2),
    //    nn::capsrv::ResultAlbumInvalidFileData
    //);


    // 難読化されていない入力は失敗
    NN_LOG("Checking raw input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION2
        };
        //nn::capsrv::server::detail::EncryptMakerNoteInplace(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }

    // パディングが 0 になっていない入力は失敗
    NN_LOG("Checking bad padding...\n");
    {
        NN_LOG("  case 1\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG1_VERSION2
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }
    {
        NN_LOG("  case 2\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG2_VERSION2
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }


    // エントリーの解析チェック
    const char version[] = { NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2 };
    std::string bytesVersion(version, version + sizeof(version));
    EntryVariation entryVariationList[] = {
        NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_VERSION2( NNT_CAPSRV_MAKE_VARIATIONLISTENTRY )
    };

    int entryVariationCount = sizeof(entryVariationList) / sizeof(entryVariationList[0]);

    {
        TestEntryOption opt = {};
        opt.encryptionOffset = EncryptionHeadOffset;
        opt.encryptionUnitSize = 32;
        TestEntryVariation(bytesVersion, entryVariationList, entryVariationCount, opt);
    }
}

TEST(UnitTest, ParseMakerNote_Version3)
{
    using namespace nn::capsrv::server::detail;

    static const int EncryptionHeadOffset = 24;

    // 正しくパースできるテスト
    size_t okLength = 0;
    NN_LOG("Checking good input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION3
        };
        okLength = sizeof(input);

        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);

        uint8_t expectedSignature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
        uint8_t expectedAppId[]     = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
        uint8_t expectedDateTime[]  = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
        uint8_t expectedAppletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
        uint8_t expectedApplicationData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLIDATA };
        uint8_t expectedSystemReservedInfo[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SYSRSVINFO };
        NN_ABORT_UNLESS_RESULT_SUCCESS( TryParseMakerNote(&info, input, sizeof(input)) );
        EXPECT_EQ(MakerNoteVersion_Version3, info.version);
        EXPECT_TRUE(0 == std::memcmp(expectedSignature, &info.signature, sizeof(expectedSignature)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppId, &info.encryptedApplicationId, sizeof(expectedAppId)));
        EXPECT_TRUE(0 == std::memcmp(expectedDateTime, &info.dateTime, sizeof(expectedDateTime)));
        EXPECT_TRUE(0 == std::memcmp(expectedAppletData, &info.appletData, sizeof(expectedAppletData)));
        EXPECT_TRUE(0 == std::memcmp(expectedApplicationData, &info.applicationData, sizeof(expectedApplicationData)));
        EXPECT_TRUE(0 == std::memcmp(expectedSystemReservedInfo, &info.systemReservedInfo, sizeof(expectedSystemReservedInfo)));
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH, info.dataWidth);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT, info.dataHeight);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT, info.dataOrientation);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT, info.frameCount);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER, info.frameRateNumerator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM, info.frameRateDenominator);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION,   info.dataDurationMilliseconds);

        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_COPYRIGHT, info.isCopyrightComposited);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDIT,    info.hasUneditableArea);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITX,   info.uneditableAreaCoordinateX);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITY,   info.uneditableAreaCoordinateY);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITWIDTH, info.uneditableAreaWidth);
        EXPECT_EQ(NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITHEIGHT, info.uneditableAreaHeight);
    }

    //// バージョン不一致は失敗
    //NN_LOG("Checking future entry list ...\n");
    //NNT_CAPSRV_TEST_MAKERNOTEPARSER((
    //    NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
    //    NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3),
    //    nn::capsrv::ResultAlbumInvalidFileData
    //);


    // 難読化されていない入力は失敗
    NN_LOG("Checking raw input...\n");
    {
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION3
        };
        //nn::capsrv::server::detail::EncryptMakerNoteInplace(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }

    // パディングが 0 になっていない入力は失敗
    NN_LOG("Checking bad padding...\n");
    {
        NN_LOG("  case 1\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG1_VERSION3
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }
    {
        NN_LOG("  case 2\n");
        MakerNoteInfo info = {};
        uint8_t input[] ={
            NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3
            NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_NG2_VERSION3
        };
        nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(input, sizeof(input), EncryptionHeadOffset);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
            TryParseMakerNote(&info, input, sizeof(input))
        );
    }


    // エントリーの解析チェック
    const char version[] = { NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3 };
    std::string bytesVersion(version, version + sizeof(version));
    EntryVariation entryVariationList[] = {
        NNT_CAPSRV_MAKERNOTE_FOREACH_ENTRY_VERSION3( NNT_CAPSRV_MAKE_VARIATIONLISTENTRY )
    };

    int entryVariationCount = sizeof(entryVariationList) / sizeof(entryVariationList[0]);

    {
        TestEntryOption opt = {};
        opt.encryptionOffset = EncryptionHeadOffset;
        opt.encryptionUnitSize = 32;
        TestEntryVariation(bytesVersion, entryVariationList, entryVariationCount, opt);
    }
}
