﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

#include "detail/testCapsrv_MakerNoteForTestFoundation.h"

#include <nnt.h>

namespace {

    std::string CreateBytesVersion(uint32_t version) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<const char*>(&version);
        std::string v;
        v.append(p, p + sizeof(uint32_t));
        return v;
    }

    std::string CreateBytesIntegerEntry(uint16_t tag, uint32_t value) NN_NOEXCEPT
    {
        uint16_t size = sizeof(value);

        auto pTag = reinterpret_cast<const char*>(&tag);
        auto pSize = reinterpret_cast<const char*>(&size);
        auto pValue = reinterpret_cast<const char*>(&value);

        std::string v;
        v.append(pTag, pTag + 2);
        v.append(pSize, pSize + 2);
        v.append(pValue, pValue + size);
        return v;
    }

    std::string CreateBytesArrayEntry(uint16_t tag, const nnt::capsrv::ArrayValue& value) NN_NOEXCEPT
    {
        uint16_t size = sizeof(value);

        auto pTag = reinterpret_cast<const char*>(&tag);
        auto pSize = reinterpret_cast<const char*>(&size);
        auto pValue = reinterpret_cast<const char*>(&value);

        std::string v;
        v.append(pTag, pTag + 2);
        v.append(pSize, pSize + 2);
        v.append(pValue, pValue + size);
        return v;
    }

    template<int Length>
    std::string CreateBytesTArrayEntry(uint16_t tag, const nnt::capsrv::ArrayValueImpl<Length>& value) NN_NOEXCEPT
    {
        uint16_t size = sizeof(value);

        auto pTag = reinterpret_cast<const char*>(&tag);
        auto pSize = reinterpret_cast<const char*>(&size);
        auto pValue = reinterpret_cast<const char*>(&value);

        std::string v;
        v.append(pTag, pTag + 2);
        v.append(pSize, pSize + 2);
        v.append(pValue, pValue + size);
        return v;
    }

    //------------------------------------------------------
    void CheckEntryCountMinMax() NN_NOEXCEPT
    {
        NN_LOG("checking defined entry count min max\n");
#define CHECK_ENTRYCOUNT(versionAndTag, expectedMin, expectedMax)    \
    EXPECT_EQ(expectedMin, nnt::capsrv::GetEntryMinCountForTestFoundataion versionAndTag);  \
    EXPECT_EQ(expectedMax, nnt::capsrv::GetEntryMaxCountForTestFoundataion versionAndTag);

        // v0
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer0   ), 1, 1);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Bytes0     ), 1, 1);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer1   ), 0, 0);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Bytes1     ), 0, 0);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer2Req), 0, 0);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer2Opt), 0, 0);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer3Req), 0, 0);
        CHECK_ENTRYCOUNT((0, nnt::capsrv::MakerNoteEntryTag_Integer3Opt), 0, 0);

        // v1
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer0   ), 1, 1);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Bytes0     ), 1, 1);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer1   ), 1, 1);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Bytes1     ), 1, 1);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer2Req), 0, 0);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer2Opt), 0, 0);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer3Req), 0, 0);
        CHECK_ENTRYCOUNT((1, nnt::capsrv::MakerNoteEntryTag_Integer3Opt), 0, 0);

        // v2
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer0   ), 1, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Bytes0     ), 1, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer1   ), 1, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Bytes1     ), 1, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer2Req), 1, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer2Opt), 0, 1);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer3Req), 0, 0);
        CHECK_ENTRYCOUNT((2, nnt::capsrv::MakerNoteEntryTag_Integer3Opt), 0, 0);

        // v3
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer0   ), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Bytes0     ), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer1   ), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Bytes1     ), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer2Req), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer2Opt), 0, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer3Req), 1, 1);
        CHECK_ENTRYCOUNT((3, nnt::capsrv::MakerNoteEntryTag_Integer3Opt), 0, 1);

#undef CHECKENTRYCOUNT
    }// NOLINT(impl/function_size)

    void CheckValidInput() NN_NOEXCEPT
    {
        typedef nnt::capsrv::MakerNoteInfo MakerNoteInfo;
        typedef nnt::capsrv::ArrayValue ArrayValue;

        NN_LOG("checking valid v0\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 0);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, info.Integer1Default);
            EXPECT_EQ(info.bytes1, ArrayValue{});
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer2req, info.Integer2ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
        }

        NN_LOG("checking valid v1\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 1);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer2req, info.Integer2ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
        }

        NN_LOG("checking valid v2 min\n");
        {
            auto bytes
                = CreateBytesVersion(2)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 2);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
        }

        NN_LOG("checking valid v2 ful\n");
        {
            auto bytes
                = CreateBytesVersion(2)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Opt, 2222)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 2);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, 2222);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
        }

        NN_LOG("checking valid v3 min\n");
        {
            auto bytes
                = CreateBytesVersion(3)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Req, 3131)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 3);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer3req, 3131);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
        }

        NN_LOG("checking valid v3 ful\n");
        {
            auto bytes
                = CreateBytesVersion(3)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Opt, 2222)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Req, 3131)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Opt, 3232)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 3);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, 2222);
            EXPECT_EQ(info.integer3req, 3131);
            EXPECT_EQ(info.integer3opt, 3232);
        }
    }// NOLINT(impl/function_size)

    // MakerNoteEntryParser, MakerNoteEntryVisitor のチェック
    void CheckIllFormedInput() NN_NOEXCEPT
    {
        typedef nnt::capsrv::MakerNoteInfo MakerNoteInfo;

        NN_LOG("checking null input\n");
        {
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, nullptr, 0)
            );
        }

        NN_LOG("checking empty input\n");
        {
            auto bytes = "";
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, &bytes, 0)
            );
        }

        NN_LOG("checking short input(version)\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 1) // version に足りない
            );
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 3) // version に足りない
            );
        }

        NN_LOG("checking short input(entry/tag)\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 4) // entry/tag に足りない
            );
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 5) // entry/tag に足りない
            );
        }

        NN_LOG("checking short input(entry/size)\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 6) // entry/size に足りない
            );
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 7) // entry/size に足りない
            );
        }

        NN_LOG("checking short input(entry/data)\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 8) // entry/data に足りない
            );
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), 11) // entry/data に足りない
            );
        }

        NN_LOG("checking invalid entry order input\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking duplicated entry input\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking duplicated entry input\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }
    }// NOLINT(impl/function_size)

    void CheckDecoderInvalidDataLength() NN_NOEXCEPT
    {
        typedef nnt::capsrv::MakerNoteInfo MakerNoteInfo;

        NN_LOG("checking too short integer value\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesTArrayEntry<3>(nnt::capsrv::MakerNoteEntryTag_Integer1, {})
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking too long integer value\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesTArrayEntry<5>(nnt::capsrv::MakerNoteEntryTag_Integer1, {})
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking too short array value\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesTArrayEntry<15>(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking too long array value\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesTArrayEntry<17>(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }
    }// NOLINT(impl/function_size)

    void CheckFormatVersion() NN_NOEXCEPT
    {
        typedef nnt::capsrv::MakerNoteInfo MakerNoteInfo;
        typedef nnt::capsrv::ArrayValue ArrayValue;

        NN_LOG("checking unknown tag (error)\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Unknown, 0x7777);
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking unknown tag (ignore)\n");
        {
            auto bytes
                = CreateBytesVersion(2)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Unknown, 0x7777);
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 2);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
        }

        NN_LOG("checking unsupported tag (error)\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                ;
            MakerNoteInfo info = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.c_str(), bytes.size())
            );
        }

        NN_LOG("checking unsupported tag (ignore)\n");
        {
            auto bytes
                = CreateBytesVersion(2)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Req, 3131)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            NNT_EXPECT_RESULT_SUCCESS(nnt::capsrv::ParseMakerNoteForTestFoundation(&info, bytes.data(), bytes.size()));
            EXPECT_EQ(info.version, 2);
            EXPECT_EQ(info.integer0,  101);
            EXPECT_EQ(info.bytes0, ArrayValue{"bytes0"});
            EXPECT_EQ(info.integer1, 1111);
            EXPECT_EQ(info.bytes1, ArrayValue{"bytes1"});
            EXPECT_EQ(info.integer2req, 2121);
            EXPECT_EQ(info.integer2opt, info.Integer2OptDefault);
            EXPECT_EQ(info.integer3req, info.Integer3ReqDefault);
            EXPECT_EQ(info.integer3opt, info.Integer3OptDefault);
        }
    }// NOLINT(impl/function_size)

    void CheckGeneratorOutput() NN_NOEXCEPT
    {
        typedef nnt::capsrv::MakerNoteInfo MakerNoteInfo;
        static const int BufferSize = 4096;

        NN_LOG("checking generation v0\n");
        {
            auto bytes
                = CreateBytesVersion(0)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            info.version = 0;
            info.integer0 = 101;
            info.bytes0 = {"bytes0"};
            char buf[BufferSize] = {};
            auto size = nnt::capsrv::GenerateMakerNoteForTestFoundation(buf, sizeof(buf), info);
            std::string generated(buf, buf + size);
            EXPECT_EQ(bytes, generated);
        }

        NN_LOG("checking generation v1\n");
        {
            auto bytes
                = CreateBytesVersion(1)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            info.version = 1;
            info.integer0 = 101;
            info.bytes0 = {"bytes0"};
            info.integer1 = 1111;
            info.bytes1 = {"bytes1"};
            char buf[BufferSize] = {};
            auto size = nnt::capsrv::GenerateMakerNoteForTestFoundation(buf, sizeof(buf), info);
            std::string generated(buf, buf + size);
            EXPECT_EQ(bytes, generated);
        }

        NN_LOG("checking generation v2\n");
        {
            auto bytes
                = CreateBytesVersion(2)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Opt, 2222)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            info.version = 2;
            info.integer0 = 101;
            info.bytes0 = {"bytes0"};
            info.integer1 = 1111;
            info.bytes1 = {"bytes1"};
            info.integer2req = 2121;
            info.integer2opt = 2222;
            char buf[BufferSize] = {};
            auto size = nnt::capsrv::GenerateMakerNoteForTestFoundation(buf, sizeof(buf), info);
            std::string generated(buf, buf + size);
            EXPECT_EQ(bytes, generated);
        }

        NN_LOG("checking generation v3\n");
        {
            auto bytes
                = CreateBytesVersion(3)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer0,  101)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes0, {"bytes0"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer1, 1111)
                + CreateBytesArrayEntry(nnt::capsrv::MakerNoteEntryTag_Bytes1, {"bytes1"})
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Req, 2121)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer2Opt, 2222)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Req, 3131)
                + CreateBytesIntegerEntry(nnt::capsrv::MakerNoteEntryTag_Integer3Opt, 3232)
                ;
            MakerNoteInfo info = {};
            info.Fill(0xFF);
            info.version = 3;
            info.integer0 = 101;
            info.bytes0 = {"bytes0"};
            info.integer1 = 1111;
            info.bytes1 = {"bytes1"};
            info.integer2req = 2121;
            info.integer2opt = 2222;
            info.integer3req = 3131;
            info.integer3opt = 3232;
            char buf[BufferSize] = {};
            auto size = nnt::capsrv::GenerateMakerNoteForTestFoundation(buf, sizeof(buf), info);
            std::string generated(buf, buf + size);
            EXPECT_EQ(bytes, generated);
        }
    }// NOLINT(impl/function_size)


}

TEST(UnitTest, MakerNoteParserFoundation)
{
    CheckEntryCountMinMax();

    CheckValidInput();
    CheckIllFormedInput();
    CheckDecoderInvalidDataLength();
    CheckFormatVersion();
}

TEST(UnitTest, MakerNoteGeneratorFoundation)
{
    CheckGeneratorOutput();
}



