﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_GenerateMakerNote.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptMakerNote.h"
#include "detail/testCapsrv_TestMakerNote.h"

TEST(UnitTest, GenerateMakerNote_Version0)
{
    using namespace nn::capsrv::server::detail;
    static const size_t BufferSize = 6 * 1024;

    MakerNoteInfo info = {};
    info.version = MakerNoteVersion_Version0;

    size_t expectedSize =
        sizeof(MakerNoteVersionType) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(Signature) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(EncryptedApplicationId) );
    NN_LOG("expected makernote size = %llu bytes\n", expectedSize);
    NN_ASSERT(BufferSize >= expectedSize);

    uint8_t signature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
    NN_STATIC_ASSERT(sizeof(signature) == sizeof(info.signature));
    std::memcpy(&info.signature, &signature[0], sizeof(info.signature));

    uint8_t encryptedApplicationId[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
    NN_STATIC_ASSERT(sizeof(encryptedApplicationId) == sizeof(info.encryptedApplicationId));
    std::memcpy(&info.encryptedApplicationId, &encryptedApplicationId[0], sizeof(info.encryptedApplicationId));

    // MakerNote を生成してみる
    char buffer[BufferSize];
    size_t writtenSize = GenerateMakerNote(buffer, sizeof(buffer), info);
    EXPECT_EQ(expectedSize, writtenSize);

    uint8_t expectedBuffer[] ={
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION0
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION0
    };

    for(size_t i = 0; i < writtenSize; i++)
    {
        uint8_t expectedValue = expectedBuffer[i];
        uint8_t actualValue = static_cast<uint8_t>(buffer[i]);
        EXPECT_EQ(expectedValue, actualValue);
        if(expectedValue != actualValue)
        {
            NN_LOG("i = %d\n", static_cast<int>(i));
        }
    }

}

TEST(UnitTest, GenerateMakerNote_Version1)
{
    using namespace nn::capsrv::server::detail;
    static const size_t BufferSize = 6 * 1024;

    MakerNoteInfo info = {};
    info.version = MakerNoteVersion_Version1;

    static const size_t expectedSize =
        sizeof(MakerNoteVersionType) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(Signature) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(EncryptedApplicationId) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDateTimeType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFileDescriptionType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteAppletDataType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataOrientationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameCountType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateNumeratorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateDenominatorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataDurationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteKeyFrameIntervalType) ) +
        20; // padding
    NN_STATIC_ASSERT((expectedSize - 24) % 32 == 0);
    NN_LOG("expected makernote size = %llu bytes\n", expectedSize);
    NN_ASSERT(BufferSize >= expectedSize);

    uint8_t signature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
    NN_STATIC_ASSERT(sizeof(signature) == sizeof(info.signature));
    std::memcpy(&info.signature, &signature[0], sizeof(info.signature));

    uint8_t encryptedApplicationId[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
    NN_STATIC_ASSERT(sizeof(encryptedApplicationId) == sizeof(info.encryptedApplicationId));
    std::memcpy(&info.encryptedApplicationId, &encryptedApplicationId[0], sizeof(info.encryptedApplicationId));

    uint8_t dateTime[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
    NN_STATIC_ASSERT(sizeof(dateTime) == sizeof(info.dateTime));
    std::memcpy(&info.dateTime, &dateTime[0], sizeof(info.dateTime));

    info.fileDescription = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FILEDESC;

    uint8_t appletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
    NN_STATIC_ASSERT(sizeof(appletData) == sizeof(info.appletData));
    std::memcpy(&info.appletData, &appletData[0], sizeof(info.appletData));

    info.dataWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH;
    info.dataHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT;
    info.dataOrientation = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT;
    info.frameCount = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT;
    info.frameRateNumerator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER;
    info.frameRateDenominator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM;
    info.dataDurationMilliseconds = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION;
    info.keyFrameInterval = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_KEYINTVL;

    // MakerNote を生成してみる
    char buffer[BufferSize];
    size_t writtenSize = GenerateMakerNote(buffer, sizeof(buffer), info);
    EXPECT_EQ(expectedSize, writtenSize);

    uint8_t expectedBufferRaw[expectedSize] ={
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION1
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION1
        NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION1
    };
    uint8_t expectedBuffer[expectedSize] = {};
    std::memcpy(expectedBuffer, expectedBufferRaw, expectedSize);
    nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(expectedBuffer, sizeof(expectedBuffer), 24);
    EXPECT_EQ(0, std::memcmp(expectedBuffer, expectedBufferRaw, 24));
    EXPECT_NE(0, std::memcmp(expectedBuffer, expectedBufferRaw, sizeof(expectedBuffer)));

    for(size_t i = 0; i < writtenSize; i++)
    {
        uint8_t expectedValue = expectedBuffer[i];
        uint8_t actualValue = static_cast<uint8_t>(buffer[i]);
        EXPECT_EQ(expectedValue, actualValue);
        if(expectedValue != actualValue)
        {
            NN_LOG("i = %d\n", static_cast<int>(i));
        }
    }
}

TEST(UnitTest, GenerateMakerNote_Version2)
{
    using namespace nn::capsrv::server::detail;
    static const size_t BufferSize = 6 * 1024;

    MakerNoteInfo info = {};
    info.version = MakerNoteVersion_Version2;

    static const size_t expectedSize =
        sizeof(MakerNoteVersionType) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(Signature) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(EncryptedApplicationId) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDateTimeType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFileDescriptionType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteAppletDataType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataOrientationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameCountType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateNumeratorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateDenominatorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataDurationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteKeyFrameIntervalType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteCopyrightCompositionType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaCoordinateXType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaCoordinateYType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteApplicationDataType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataHeightType) ) +
        12; // padding
    NN_STATIC_ASSERT((expectedSize - 24) % 32 == 0);
    NN_LOG("expected makernote size = %llu bytes\n", expectedSize);
    NN_ASSERT(BufferSize >= expectedSize);

    uint8_t signature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
    NN_STATIC_ASSERT(sizeof(signature) == sizeof(info.signature));
    std::memcpy(&info.signature, &signature[0], sizeof(info.signature));

    uint8_t encryptedApplicationId[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
    NN_STATIC_ASSERT(sizeof(encryptedApplicationId) == sizeof(info.encryptedApplicationId));
    std::memcpy(&info.encryptedApplicationId, &encryptedApplicationId[0], sizeof(info.encryptedApplicationId));

    uint8_t dateTime[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
    NN_STATIC_ASSERT(sizeof(dateTime) == sizeof(info.dateTime));
    std::memcpy(&info.dateTime, &dateTime[0], sizeof(info.dateTime));

    info.fileDescription = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FILEDESC;

    uint8_t appletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
    NN_STATIC_ASSERT(sizeof(appletData) == sizeof(info.appletData));
    std::memcpy(&info.appletData, &appletData[0], sizeof(info.appletData));

    info.dataWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH;
    info.dataHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT;
    info.dataOrientation = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT;
    info.frameCount = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT;
    info.frameRateNumerator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER;
    info.frameRateDenominator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM;
    info.dataDurationMilliseconds = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION;
    info.keyFrameInterval = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_KEYINTVL;

    info.isCopyrightComposited = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_COPYRIGHT;
    info.hasUneditableArea = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDIT;
    info.uneditableAreaCoordinateX = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITX;
    info.uneditableAreaCoordinateY = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITY;
    info.uneditableAreaWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITWIDTH;
    info.uneditableAreaHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITHEIGHT;
    info.movieDataWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_MOVIEWIDTH;
    info.movieDataHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_MOVIEHEIGHT;

    uint8_t applicationData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLIDATA };
    NN_STATIC_ASSERT(sizeof(applicationData) == sizeof(info.applicationData));
    std::memcpy(&info.applicationData, &applicationData[0], sizeof(info.applicationData));

    // MakerNote を生成してみる
    char buffer[BufferSize];
    size_t writtenSize = GenerateMakerNote(buffer, sizeof(buffer), info);
    EXPECT_EQ(expectedSize, writtenSize);

    uint8_t expectedBufferRaw[expectedSize] ={
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION2
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION2
        NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION2
    };
    uint8_t expectedBuffer[expectedSize] = {};
    std::memcpy(expectedBuffer, expectedBufferRaw, expectedSize);
    nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(expectedBuffer, sizeof(expectedBuffer), 24);
    EXPECT_EQ(0, std::memcmp(expectedBuffer, expectedBufferRaw, 24));
    EXPECT_NE(0, std::memcmp(expectedBuffer, expectedBufferRaw, sizeof(expectedBuffer)));

    for(size_t i = 0; i < writtenSize; i++)
    {
        uint8_t expectedValue = expectedBuffer[i];
        uint8_t actualValue = static_cast<uint8_t>(buffer[i]);
        EXPECT_EQ(expectedValue, actualValue);
        if(expectedValue != actualValue)
        {
            NN_LOG("i = %d\n", static_cast<int>(i));
        }
    }
}

TEST(UnitTest, GenerateMakerNote_Version3)
{
    using namespace nn::capsrv::server::detail;
    static const size_t BufferSize = 6 * 1024;

    MakerNoteInfo info = {};
    info.version = MakerNoteVersion_Version3;

    static const size_t expectedSize =
        sizeof(MakerNoteVersionType) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(Signature) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(EncryptedApplicationId) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDateTimeType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFileDescriptionType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteAppletDataType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataOrientationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameCountType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateNumeratorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteFrameRateDenominatorType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataDurationType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteKeyFrameIntervalType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteCopyrightCompositionType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaCoordinateXType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaCoordinateYType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteUneditableAreaHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteApplicationDataType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataWidthType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteDataHeightType) ) +
        ( sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(MakerNoteSystemReservedInfoType) ) +
        8; // padding
    NN_STATIC_ASSERT((expectedSize - 24) % 32 == 0);
    NN_LOG("expected makernote size = %llu bytes\n", expectedSize);
    NN_ASSERT(BufferSize >= expectedSize);

    uint8_t signature[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SIGNATURE };
    NN_STATIC_ASSERT(sizeof(signature) == sizeof(info.signature));
    std::memcpy(&info.signature, &signature[0], sizeof(info.signature));

    uint8_t encryptedApplicationId[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_ENCAPPID };
    NN_STATIC_ASSERT(sizeof(encryptedApplicationId) == sizeof(info.encryptedApplicationId));
    std::memcpy(&info.encryptedApplicationId, &encryptedApplicationId[0], sizeof(info.encryptedApplicationId));

    uint8_t dateTime[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_DATETIME };
    NN_STATIC_ASSERT(sizeof(dateTime) == sizeof(info.dateTime));
    std::memcpy(&info.dateTime, &dateTime[0], sizeof(info.dateTime));

    info.fileDescription = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FILEDESC;

    uint8_t appletData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLETDATA };
    NN_STATIC_ASSERT(sizeof(appletData) == sizeof(info.appletData));
    std::memcpy(&info.appletData, &appletData[0], sizeof(info.appletData));

    info.dataWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAWIDTH;
    info.dataHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAHEIGHT;
    info.dataOrientation = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DATAORIENT;
    info.frameCount = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRAMECOUNT;
    info.frameRateNumerator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATENUMER;
    info.frameRateDenominator = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_FRATEDENOM;
    info.dataDurationMilliseconds = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_DURATION;
    info.keyFrameInterval = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_KEYINTVL;

    info.isCopyrightComposited = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_COPYRIGHT;
    info.hasUneditableArea = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDIT;
    info.uneditableAreaCoordinateX = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITX;
    info.uneditableAreaCoordinateY = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITY;
    info.uneditableAreaWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITWIDTH;
    info.uneditableAreaHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_UNEDITHEIGHT;
    info.movieDataWidth = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_MOVIEWIDTH;
    info.movieDataHeight = NNT_CAPSRV_MAKERNOTE_VALUE_ENTRYBODY_OK_MOVIEHEIGHT;

    uint8_t applicationData[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_APPLIDATA };
    NN_STATIC_ASSERT(sizeof(applicationData) == sizeof(info.applicationData));
    std::memcpy(&info.applicationData, &applicationData[0], sizeof(info.applicationData));

    uint8_t systemReservedInfo[] = { NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYBODY_OK_SYSRSVINFO };
    NN_STATIC_ASSERT(sizeof(systemReservedInfo) == sizeof(info.systemReservedInfo));
    std::memcpy(&info.systemReservedInfo, &systemReservedInfo[0], sizeof(info.systemReservedInfo));

    // MakerNote を生成してみる
    char buffer[BufferSize];
    size_t writtenSize = GenerateMakerNote(buffer, sizeof(buffer), info);
    EXPECT_EQ(expectedSize, writtenSize);

    uint8_t expectedBufferRaw[expectedSize] ={
        NNT_CAPSRV_MAKERNOTE_BYTES_VERSION3
        NNT_CAPSRV_MAKERNOTE_BYTES_ENTRYLIST_OK_VERSION3
        NNT_CAPSRV_MAKERNOTE_BYTES_PADDING_OK_VERSION3
    };
    uint8_t expectedBuffer[expectedSize] = {};
    std::memcpy(expectedBuffer, expectedBufferRaw, expectedSize);
    nn::capsrv::server::detail::EncryptMakerNoteInplaceVersion1(expectedBuffer, sizeof(expectedBuffer), 24);
    EXPECT_EQ(0, std::memcmp(expectedBuffer, expectedBufferRaw, 24));
    EXPECT_NE(0, std::memcmp(expectedBuffer, expectedBufferRaw, sizeof(expectedBuffer)));

    for(size_t i = 0; i < writtenSize; i++)
    {
        uint8_t expectedValue = expectedBuffer[i];
        uint8_t actualValue = static_cast<uint8_t>(buffer[i]);
        EXPECT_EQ(expectedValue, actualValue);
        if(expectedValue != actualValue)
        {
            NN_LOG("i = %d\n", static_cast<int>(i));
        }
    }
}
