﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <list>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_MovieReaderFileSystem.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/threading/testCapsrv_WorkerThreadPool.h"
#include "testCapsrv_StartupTestCase.h"
#include "detail/testCapsrv_TestMovieStreamWrapperBase.h"
#include "detail/testCapsrv_TestMovieStreamRandomAccessImpl.h"

static int RepeatCount1 = 100; // 1000
static int RepeatCount2 = 10; // 100
static const size_t WorkerThreadStackSize = 16 * 1024;

#define NNT_CAPSRV_IO_LOG(...) NN_LOG(__VA_ARGS__)

#ifndef NNT_CAPSRV_IO_LOG
#define NNT_CAPSRV_IO_LOG(...)
#endif

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 03, 10, 23, 44, 00, 0x0245215BF465F8EE, .mp4, 10000000, 12346),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]);   \
    NN_UNUSED(varCount);    \
    NN_UNUSED(varList);

namespace {

    std::vector<std::vector<nn::capsrv::AlbumEntry>> SetupTestMovieFiles() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

        NNT_CAPSRV_FILELIST(fileList, fileCount);
        // ファイル作成
        for(int i = 0; i < fileCount; i++){ fileList[i].CreateMovie(); }

        std::vector<std::vector<nn::capsrv::AlbumEntry>> entryListList;
        entryListList.resize(nn::capsrv::AlbumStorageCount);

        // ファイルリスト取得
        NNT_CAPSRV_FOREACH_STORAGE_I(s, storage)
        {
            auto& list = entryListList[s];
            list = nnt::capsrv::GetAlbumEntryList(storage);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }
        nn::capsrv::FinalizeAlbumAccess();

        return entryListList;
    }
}

namespace {

    enum MovieReadStreamTestOperation
    {
        MovieReadStreamTestOperation_RegisterUnregister,
        MovieReadStreamTestOperation_OpenClose,
        MovieReadStreamTestOperation_Read,
    };

    struct MovieReadStreamTestOperationSetupArgument
    {
        std::vector<std::pair<nn::capsrv::AlbumMovieReadStreamHandle, const char*>> preRegisterHandles;
    };

    struct MovieReadStreamTestOperationThreadArgument
    {
    public:
        void SetupForRegisterUnregister(
            int repeatCount,
            nn::capsrv::AlbumMovieReadStreamHandle streamHandle,
            const char* filename
        ) NN_NOEXCEPT
        {
            this->operation = MovieReadStreamTestOperation_RegisterUnregister;
            this->repeatCount = repeatCount;
            this->registerUnregister.streamHandle = streamHandle;
            this->registerUnregister.filename = filename;
        }

        void SetupForOpenClose(
            int repeatCount,
            const char* filepath
        ) NN_NOEXCEPT
        {
            this->operation = MovieReadStreamTestOperation_OpenClose;
            this->repeatCount = repeatCount;
            this->openClose.filepath = filepath;
        }

        void SetupForRead(
            int repeatCount,
            const char* filepath,
            uint64_t seed
        ) NN_NOEXCEPT
        {
            this->operation = MovieReadStreamTestOperation_Read;
            this->repeatCount = repeatCount;
            this->read.filepath = filepath;
            this->read.seed = seed;
        }

    public:
        MovieReadStreamTestOperation operation;
        int repeatCount;

        struct RegisterUnregisterArgument
        {
            nn::capsrv::AlbumMovieReadStreamHandle streamHandle;
            const char* filename;
        } registerUnregister;

        struct OpenCloseArgument
        {
            const char* filepath;
        } openClose;

        struct ReadArgument
        {
            const char* filepath;
            uint64_t seed;
        } read;
    };

    template<int WorkerThreadCount>
    void TestMultithreadOperationImpl(
        MovieReadStreamTestOperationSetupArgument& setupArg,
        MovieReadStreamTestOperationThreadArgument* argList
    ) NN_NOEXCEPT
    {
        const size_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
        const int CacheCount = 2;

        nnt::capsrv::threading::WorkerThreadPool workerThreads;
        workerThreads.Initialize(WorkerThreadCount, WorkerThreadStackSize, nn::os::DefaultThreadPriority);
        NN_UTIL_SCOPE_EXIT{ workerThreads.Finalize(); };

        // ワークメモリサイズ
        size_t mwfsSize = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemorySizeForFile(ChunkSize, CacheCount);
        size_t mwfsAlignment = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile();
        NN_LOG("memorySize  = %llu\n", mwfsSize);
        NN_LOG("memoryAlign = %llu\n", mwfsAlignment);

        // ファイルシステム
        nn::capsrv::movie::MovieReaderFileSystem mrfs;
        mrfs.Initialize("Movie");
        NN_UTIL_SCOPE_EXIT{ mrfs.Finalize(); };

        // 事前に登録しておくものを登録
        NN_LOG("pre-registering files(%lld)\n", setupArg.preRegisterHandles.size());
        std::vector<void*> preMem(setupArg.preRegisterHandles.size(), nullptr);
        for(size_t i = 0; i < setupArg.preRegisterHandles.size(); i++)
        {
            auto& e = setupArg.preRegisterHandles[i];
            NN_LOG("  h=%llu, filename=%s\n", e.first, e.second);
            preMem[i] = aligned_alloc(mwfsAlignment, mwfsSize);
            NN_ASSERT_NOT_NULL(preMem[i]);
            NN_ABORT_UNLESS_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection(e.second, e.first, ChunkSize, CacheCount, preMem[i], mwfsSize));
        }
        NN_UTIL_SCOPE_EXIT{
            for(size_t i = 0; i < setupArg.preRegisterHandles.size(); i++)
            {
                auto& e = setupArg.preRegisterHandles[i];
                mrfs.UnregisterMovieReadStreamDataSection(e.second);
                free(preMem[i]);
                NN_ASSERT_NOT_NULL(preMem[i]);
            }
        };

        NN_LOG("reserving cache memory for workers\n");
        void* mem[WorkerThreadCount] = {};
        for(int t = 0; t < WorkerThreadCount; t++)
        {
            mem[t] = aligned_alloc(mwfsAlignment, mwfsSize);
            NN_ASSERT_NOT_NULL(mem[t]);
        }
        NN_UTIL_SCOPE_EXIT{
            for(int t = 0; t < WorkerThreadCount; t++)
            {
                free(mem[t]);
                mem[t] = nullptr;
            }
        };

        NN_LOG("starting workers\n");
        workerThreads.StartWorker();
        workerThreads.InvokeFunction([&](int tid) -> void
        {
            auto& arg = argList[tid];
            int repeatCount = arg.repeatCount;

            // Register - Unregister を繰り返す
            if(arg.operation == MovieReadStreamTestOperation_RegisterUnregister)
            {
                auto handle = arg.registerUnregister.streamHandle;
                auto filename = arg.registerUnregister.filename;
                for(int i = 0; i < repeatCount; i++)
                {
                    auto result = mrfs.RegisterMovieReadStreamDataSection(filename, handle, ChunkSize, CacheCount, mem[tid], mwfsSize);
                    //NN_LOG("register[%d] -> %d-%d\n", tid, result.GetModule(), result.GetDescription());
                    EXPECT_TRUE(
                        result.IsSuccess() ||
                        nn::fs::ResultAlreadyExists::Includes(result)
                    );
                    NN_UNUSED(result);
                    if(result.IsSuccess())
                    {
                        mrfs.UnregisterMovieReadStreamDataSection(filename);
                    }
                }
            }

            // Open - Close を繰り返す
            if(arg.operation == MovieReadStreamTestOperation_OpenClose)
            {
                auto filepath = arg.openClose.filepath;
                for(int i = 0; i < repeatCount; i++)
                {
                    nn::fs::FileHandle hFile = {};
                    auto result = nn::fs::OpenFile(&hFile, filepath, nn::fs::OpenMode_Read);
                    //NN_LOG("open[%d] -> %d-%d\n", tid, result.GetModule(), result.GetDescription());
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    if(result.IsSuccess())
                    {
                        nn::fs::CloseFile(hFile);
                    }
                }
            }

            // ランダムな Read をし続ける
            if(arg.operation == MovieReadStreamTestOperation_Read)
            {
                auto filepath = arg.read.filepath;
                auto seed = arg.read.seed;
                nn::fs::FileHandle hFile = {};
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&hFile, filepath, nn::fs::OpenMode_Read));

                std::mt19937 rand(seed);
                int64_t fileSize = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, hFile));

                std::vector<char> buf(fileSize);

                std::uniform_int_distribution<int64_t> offsetDist(0, fileSize);

                for(int i = 0; i < repeatCount; i++)
                {
                    int64_t offset = 0;
                    int64_t size = 0;

                    offset = offsetDist(rand);
                    size = offsetDist(rand);
                    size = std::min(fileSize - offset, size);

                    auto result = nn::fs::ReadFile(hFile, offset, buf.data(), size);
                    //NN_LOG("open[%d] -> %d-%d\n", tid, result.GetModule(), result.GetDescription());
                    NNT_EXPECT_RESULT_SUCCESS(result);


                }
                nn::fs::CloseFile(hFile);
            }
        });
        workerThreads.StopWorker();

    }// NOLINT(impl/function_size)

}

TEST(AlbumAccessApi, MovieReaderFileSystem_ThreadSafety_RegisterUnregister)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();
    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    nn::capsrv::AlbumMovieReadStreamHandle handles[3] = {};
    for(int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&handles[i], fileList[i].GetAlbumFileId()));
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < 3; i++)
        {
            nn::capsrv::CloseAlbumMovieReadStream(handles[i]);
        }
    };

    int repeatCount = RepeatCount1;

    nn::capsrv::AlbumMovieReadStreamHandle sameStreamHandleList[] = {
        handles[0],
        handles[0],
        handles[0],
    };
    nn::capsrv::AlbumMovieReadStreamHandle differentStreamHandleList[] = {
        handles[0],
        handles[1],
        handles[2],
    };

    const char* sameFilenameList[] = {
        "movie0.mp4",
        "movie0.mp4",
        "movie0.mp4",
    };
    const char* differentFilenameList[] = {
        "movie1.mp4",
        "movie2.mp4",
        "movie3.mp4",
    };

    auto runTest = [&](nn::capsrv::AlbumMovieReadStreamHandle* handles, const char** filenames)
    {
        MovieReadStreamTestOperationSetupArgument sArg = {};
        MovieReadStreamTestOperationThreadArgument argList[3] = {};
        argList[0].SetupForRegisterUnregister(repeatCount, handles[0], filenames[0]);
        argList[1].SetupForRegisterUnregister(repeatCount, handles[1], filenames[1]);
        argList[2].SetupForRegisterUnregister(repeatCount, handles[2], filenames[2]);
        TestMultithreadOperationImpl<3>(sArg, argList);
    };

    NN_LOG("strm:same/path:same\n");
    runTest(sameStreamHandleList, sameFilenameList);
    NN_LOG("strm:same/path:diff\n");
    runTest(sameStreamHandleList, differentFilenameList);
    NN_LOG("strm:diff/path:same\n");
    runTest(differentStreamHandleList, sameFilenameList);
    NN_LOG("strm:diff/path:diff\n");
    runTest(differentStreamHandleList, differentFilenameList);
}

TEST(AlbumAccessApi, MovieReaderFileSystem_ThreadSafety_OpenClose)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();
    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    nn::capsrv::AlbumMovieReadStreamHandle handles[3] = {};
    for(int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&handles[i], fileList[i].GetAlbumFileId()));
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < 3; i++)
        {
            nn::capsrv::CloseAlbumMovieReadStream(handles[i]);
        }
    };

    int repeatCount = RepeatCount1;

    MovieReadStreamTestOperationSetupArgument setupArg = {};
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[0], "movie0.mp4"));
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[1], "movie1.mp4"));
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[2], "movie2.mp4"));

    const char* sameFilepathList[] = {
        "Movie:/movie0.mp4",
        "Movie:/movie0.mp4",
        "Movie:/movie0.mp4",
    };
    const char* diffFilepathList[] = {
        "Movie:/movie0.mp4",
        "Movie:/movie1.mp4",
        "Movie:/movie2.mp4",
    };

    auto runTest = [&](const char** filepaths)
    {
        MovieReadStreamTestOperationThreadArgument argList[3] = {};
        argList[0].SetupForOpenClose(repeatCount, filepaths[0]);
        argList[1].SetupForOpenClose(repeatCount, filepaths[1]);
        argList[2].SetupForOpenClose(repeatCount, filepaths[2]);
        TestMultithreadOperationImpl<3>(setupArg, argList);
    };

    NN_LOG("path:same\n");
    runTest(sameFilepathList);
    NN_LOG("path:diff\n");
    runTest(diffFilepathList);
}

TEST(AlbumAccessApi, MovieReaderFileSystem_ThreadSafety_Read)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();
    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    nn::capsrv::AlbumMovieReadStreamHandle handles[3] = {};
    for(int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&handles[i], fileList[i].GetAlbumFileId()));
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < 3; i++)
        {
            nn::capsrv::CloseAlbumMovieReadStream(handles[i]);
        }
    };

    int repeatCount = RepeatCount2;

    MovieReadStreamTestOperationSetupArgument setupArg = {};
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[0], "movie0.mp4"));
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[1], "movie1.mp4"));
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[2], "movie2.mp4"));

    const char* sameFilepathList[] = {
        "Movie:/movie0.mp4",
        "Movie:/movie0.mp4",
        "Movie:/movie0.mp4",
    };
    const char* diffFilepathList[] = {
        "Movie:/movie0.mp4",
        "Movie:/movie1.mp4",
        "Movie:/movie2.mp4",
    };

    auto runTest = [&](const char** filepaths)
    {
        MovieReadStreamTestOperationThreadArgument argList[3] = {};
        argList[2].SetupForRead(repeatCount, filepaths[2], 0xAB87F9321);
        argList[0].SetupForRead(repeatCount, filepaths[0], 0x012345678);
        argList[1].SetupForRead(repeatCount, filepaths[1], 0x097665282);
        TestMultithreadOperationImpl<3>(setupArg, argList);
    };

    NN_LOG("path:same\n");
    runTest(sameFilepathList);
    NN_LOG("path:diff\n");
    runTest(diffFilepathList);
}

#if 0
//
//  タイミング依存で失敗する場合があるので、とりあえず無効化しておく。
//
TEST(AlbumAccessApi, MovieReaderFileSystem_ThreadSafety_Mix)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();
    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    nn::capsrv::AlbumMovieReadStreamHandle handles[3] = {};
    for(int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&handles[i], fileList[i].GetAlbumFileId()));
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < 3; i++)
        {
            nn::capsrv::CloseAlbumMovieReadStream(handles[i]);
        }
    };

    MovieReadStreamTestOperationSetupArgument setupArg = {};
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[0], "movie0.mp4"));
    setupArg.preRegisterHandles.push_back(std::make_pair(handles[1], "movie1.mp4"));

    MovieReadStreamTestOperationThreadArgument argList[9] = {};
    argList[0].SetupForRegisterUnregister(RepeatCount1, handles[2], "movie2.mp4");
    argList[1].SetupForOpenClose(RepeatCount1, "Movie:/movie0.mp4");
    argList[2].SetupForOpenClose(RepeatCount1, "Movie:/movie0.mp4");
    argList[3].SetupForOpenClose(RepeatCount1, "Movie:/movie1.mp4");
    argList[4].SetupForOpenClose(RepeatCount1, "Movie:/movie1.mp4");
    argList[5].SetupForRead(RepeatCount2, "Movie:/movie0.mp4", 0x9779BA843E2F);
    argList[6].SetupForRead(RepeatCount2, "Movie:/movie0.mp4", 0x9F73B87CA342);
    argList[7].SetupForRead(RepeatCount2, "Movie:/movie1.mp4", 0x013BA35EF2AA);
    argList[8].SetupForRead(RepeatCount2, "Movie:/movie1.mp4", 0xE2BACD243DF0);

    TestMultithreadOperationImpl<9>(setupArg, argList);
}
#endif
