﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nv/nv_MemoryManagement.h>

#ifdef _WIN32
    #pragma warning(push)
    #pragma warning(disable:4201)
#endif
#include <nvnTool/nvnTool_GlslcInterface.h>
#ifdef _WIN32
    #pragma warning(pop)
#endif

#include "testCapsrv_Server.h"
#include "../Common/testCapsrv_DirectAlbumAccessor.h"

static const size_t ApplicationMemorySize       = 1024 * 1024 * 1024;

// アプリケーションメモリの割り当て
// 合計で ApplicationMemorySize に一致するように
static const size_t GraphicsMemorySize          =   8 * 1024 * 1024;
static const size_t GraphicsAllocatorMemorySize = 256 * 1024 * 1024;
static const size_t MallocMemorySize            = ApplicationMemorySize - GraphicsMemorySize - GraphicsAllocatorMemorySize;

nn::mem::StandardAllocator g_GraphicsAllocator;
void* GraphicsAllocate(size_t size, size_t alignment, void*)
{
    return g_GraphicsAllocator.Allocate(size, alignment);
}
void GraphicsDeallocate(void* ptr, void*)
{
    g_GraphicsAllocator.Free(ptr);
}
void* GraphicsReallocate(void* p, size_t size, void*)
{
    return g_GraphicsAllocator.Reallocate(p, size);
}

extern "C" void nninitStartup() NN_NOEXCEPT
{
    nn::Result result;
    result = nn::os::SetMemoryHeapSize(ApplicationMemorySize);
    NN_ASSERT(result.IsSuccess());
    {
        uintptr_t mallocMemoryAddress;
        result = nn::os::AllocateMemoryBlock(&mallocMemoryAddress, MallocMemorySize);
        NN_ASSERT(result.IsSuccess());

        nn::init::InitializeAllocator(reinterpret_cast<void*>(mallocMemoryAddress), MallocMemorySize);
    }
}

extern "C" void nnMain()
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    NN_LOG("Running nnMain() from testCapsrv_main.cpp\n");
    {
        nn::Result result;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        {
            uintptr_t address;
            nn::Result result = nn::os::AllocateMemoryBlock(&address, GraphicsAllocatorMemorySize);
            NN_ASSERT(result.IsSuccess());
            g_GraphicsAllocator.Initialize(reinterpret_cast<void*>(address), GraphicsAllocatorMemorySize);
            nv::SetGraphicsAllocator(GraphicsAllocate, GraphicsDeallocate, GraphicsReallocate, nullptr);
            nv::SetGraphicsDevtoolsAllocator(GraphicsAllocate, GraphicsDeallocate, GraphicsReallocate, nullptr);
            glslcSetAllocator(GraphicsAllocate, GraphicsDeallocate, GraphicsReallocate, nullptr);
        }
        {
            uintptr_t address;
            nn::Result result = nn::os::AllocateMemoryBlock(&address, GraphicsMemorySize);
            NN_ASSERT(result.IsSuccess());
            nv::InitializeGraphics(reinterpret_cast<void*>(address), GraphicsMemorySize);
        }
#endif
    }

    nn::fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

#if defined(NNT_CAPSRV_ENABLE_LOCAL_HIPC)
    nnt::capsrv::StartServer(nullptr);
#elif defined(NNT_CAPSRV_ENABLE_DIRECT_ALBUM_ACCESS)
    nnt::capsrv::DirectAlbumAccessor::MountAllStorages();
#endif

    int result = RUN_ALL_TESTS();

#if defined(NNT_CAPSRV_ENABLE_LOCAL_HIPC)
    nnt::capsrv::StopServer();
#elif defined(NNT_CAPSRV_ENABLE_DIRECT_ALBUM_ACCESS)
    nnt::capsrv::DirectAlbumAccessor::UnmountAllStorages();
#endif

    ::nnt::Exit(result);
}

