﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include "TestBtmModule_Usecase.h"

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Types.h>
#include <nn/btm/btm_Result.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/btm/debug/btm_DebugApi.h>

#include <nn/btm/user/btm_UserApi.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/user/btm_UserResult.h>

#include <nn/bluetooth/bluetooth_BleScanParameterIdTestPeripheral.h>

namespace BtmUsecase
{
    void TestBtmModule_Usecase::TestPrintAllUsecase()
    {
        NN_TEST_BTM_MODULE_LOG("%s: Print all BTM usecase with the combinations of all related parameters\n", NN_CURRENT_FUNCTION_NAME);
        nn::btm::debug::GeneralTest(10);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_Usecase::TestDynamic2Slot2()
    {
        nn::btm::DeviceConditionList    deviceConditionList;
        nn::btm::DeviceSlotModeList     slotModeList;

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差のある、1, 2 で見る
        for (int deviceCount = 1; deviceCount < 3; ++deviceCount)
        {
            for (int i = 0; i < deviceCount; ++i)
            {
                NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, i + 1, deviceCount);

                SetToDynamic2Slot2(deviceCount);
                nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
                NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

                slotModeList.deviceCount = deviceConditionList.deviceCount;

                for (int j = 0; j < deviceCount; ++j)
                {
                    if (j <= i)
                    {
                        slotModeList.device[j].bdAddress    = deviceConditionList.device[j].bdAddress;
                        slotModeList.device[j].slotMode     = nn::btm::SlotMode_4;
                    }
                    else
                    {
                        slotModeList.device[j].bdAddress    = deviceConditionList.device[j].bdAddress;
                        slotModeList.device[j].slotMode     = deviceConditionList.device[j].hidDeviceCondition.slotMode;
                    }
                }

                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));

                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, i + 1, 10000);

                NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
            }
        }

        // Connect BLE device
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot2(1);
        WaitBleConnection(1);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 1000);
        WaitGattClientCondition(1, false, 0, 1000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestDynamic2Slot6()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(3, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(3, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(3, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差のある、3, 5 でだけ見る
        for (int deviceCount = 3; deviceCount < 7; ++deviceCount)
        {
            if (deviceCount != 3 && deviceCount != 5)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
            SetToDynamic2Slot6(deviceCount);

            nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
            if (deviceCount <= 4)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, deviceCount, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local4)));
            }
            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、3 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、3 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        WaitDeviceCondition(3, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差のある、(3, 0), (3, 1), (5, 0), (5, 1) でだけ見る
        for (int deviceCount = 3; deviceCount < 7; ++deviceCount)
        {
            for (int i = 0; i < deviceCount; ++i)
            {
                if (!(deviceCount == 3 && i == 0) &&
                    !(deviceCount == 3 && i == 1) &&
                    !(deviceCount == 5 && i == 0) &&
                    !(deviceCount == 5 && i == 1))
                {
                    continue;
                }

                NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, i + 1, deviceCount);

                SetToDynamic2Slot6(deviceCount);
                nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
                NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

                slotModeList.deviceCount = deviceConditionList.deviceCount;

                for (int j = 0; j < deviceCount; ++j)
                {
                    if (j <= i)
                    {
                        slotModeList.device[j].bdAddress = deviceConditionList.device[j].bdAddress;
                        slotModeList.device[j].slotMode = nn::btm::SlotMode_4;
                    }
                    else
                    {
                        slotModeList.device[j].bdAddress = deviceConditionList.device[j].bdAddress;
                        slotModeList.device[j].slotMode = deviceConditionList.device[j].hidDeviceCondition.slotMode;
                    }
                }

                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

                if (i == 0 || deviceCount == 3 || deviceCount == 4)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));

                    WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, i + 1, 10000);
                }
                else
                {
                    NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));
                }

                NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
            }
        }

        // Connect BLE device
        // Usecase 上差がないので、3 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot6(3);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestDynamic2Slot8()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(7, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(7, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(7, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、7 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local4)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がないので、7 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、7 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        WaitDeviceCondition(7, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差のある、(7, 0), (7, 1) でだけ見る
        for (int i = 0; i < 7; ++i)
        {
            if (i != 0 && i != 1)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, i + 1, 7);

            SetToDynamic2Slot8(7);
            nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
            NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 7);

            slotModeList.deviceCount = deviceConditionList.deviceCount;

            for (int j = 0; j < 7; ++j)
            {
                if (j <= i)
                {
                    slotModeList.device[j].bdAddress = deviceConditionList.device[j].bdAddress;
                    slotModeList.device[j].slotMode = nn::btm::SlotMode_4;
                }
                else
                {
                    slotModeList.device[j].bdAddress = deviceConditionList.device[j].bdAddress;
                    slotModeList.device[j].slotMode = deviceConditionList.device[j].hidDeviceCondition.slotMode;
                }
            }

            nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

            if (i == 0)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));

                WaitDeviceCondition(7, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, i + 1, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));
            }

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Connect BLE device
        // Usecase 上差がないので、7 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToDynamic2Slot8(7);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestStaticJoy4()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4(2, 2);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(2, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 2, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4(2, 2);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(2, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 2, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4(2, 2);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、(2, 2) でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4(2, 2);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
        WaitDeviceCondition(2, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 2, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がある、(2, 2), (3, 2) でのみ見る
        for (int deviceCount = 2; deviceCount < 5; ++deviceCount)
        {
            for (int largeSlotDeviceCount = 2; largeSlotDeviceCount <= deviceCount; ++largeSlotDeviceCount)
            {
                if (!(deviceCount == 2 && largeSlotDeviceCount == 2) &&
                    !(deviceCount == 3 && largeSlotDeviceCount == 2))
                {
                    continue;
                }

                NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
                SetToStaticJoy4(deviceCount, largeSlotDeviceCount);

                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
                if (deviceCount <= 2)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));
                    WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, largeSlotDeviceCount, 10000);
                }
                else
                {
                    NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
                }
                NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
            }
        }

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、2 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4(2, 2);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        WaitDeviceCondition(2, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 2, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差がないので、(3, 2) -> (3, 3) だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, 1, 3);
        SetToStaticJoy4(3, 2);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 3);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[1].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[2].hidDeviceCondition.slotMode, nn::btm::SlotMode_2);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress  = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode   = nn::btm::SlotMode_4;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));
        WaitDeviceCondition(3, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 3, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がある、(2, 2) -> (2, 1), (3, 3) -> (3, 2) だけ見る
        for (int deviceCount = 2; deviceCount < 5; ++deviceCount)
        {
            for (int largeSlotDeviceCount = 2; largeSlotDeviceCount <= deviceCount; ++largeSlotDeviceCount)
            {
                for (int i = 0; i < largeSlotDeviceCount; ++i)
                {
                    if (!(deviceCount == 2 && largeSlotDeviceCount == 2 && i == 0) &&
                        !(deviceCount == 3 && largeSlotDeviceCount == 3 && i == 0))
                    {
                        continue;
                    }

                    NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, i + 1, deviceCount);
                    SetToStaticJoy4(deviceCount, largeSlotDeviceCount);

                    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
                    NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

                    slotModeList.deviceCount = deviceConditionList.deviceCount;

                    for (int j = 0; j <= deviceCount; ++j)
                    {
                        slotModeList.device[j].bdAddress = deviceConditionList.device[j].bdAddress;

                        if (j <= i)
                        {
                            slotModeList.device[j].slotMode = nn::btm::SlotMode_2;
                        }
                        else
                        {
                            slotModeList.device[j].slotMode = deviceConditionList.device[j].hidDeviceCondition.slotMode;
                        }
                    }

                    NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));
                    WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, deviceCount - i - 1, 10000);

                    NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
                }
            }
        }

        // Connect BLE device
        // Usecase 上差がある、(2, 2), (3, 2) で見る
        for (int deviceCount = 2; deviceCount < 5; ++deviceCount)
        {
            if (deviceCount != 2 && deviceCount != 3)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy4(deviceCount, 2);

            nn::os::ClearEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

            NN_TEST_BTM_MODULE_LOG0("\n");
            NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
            NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

            nn::os::WaitEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

            NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

            if (deviceCount <= 2)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, deviceCount, 10000);
                WaitGattClientCondition(1, 0, 0, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestStaticJoy8()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy8(1);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy8(1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy8(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がある1, 5 でのみ見る
        for (int deviceCount = 1; deviceCount < 9; ++deviceCount)
        {
            if (deviceCount != 1 && deviceCount != 5)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy8(deviceCount);

            if (deviceCount <= 4)
            {
                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local4)));
            }
            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がある1, 3 でのみ見る
        for (int deviceCount = 2; deviceCount < 9; ++deviceCount)
        {
            if (deviceCount != 1 && deviceCount != 3)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy8(deviceCount);

            if (deviceCount <= 2)
            {
                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
            }
            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy8(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差がないので、(2, 1) -> (2, 2) だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, 1, 2);
        SetToStaticJoy8(2);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 2);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[1].hidDeviceCondition.slotMode, nn::btm::SlotMode_2);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress  = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode   = nn::btm::SlotMode_4;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));
        WaitDeviceCondition(2, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 2, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、(1, 1) -> (1, 0) だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy8(1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        // Connect BLE device
        // Usecase 上差がある、1, 3 で見る
        for (int deviceCount = 2; deviceCount < 5; ++deviceCount)
        {
            if (deviceCount != 1 && deviceCount != 3)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy8(deviceCount);

            nn::os::ClearEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

            NN_TEST_BTM_MODULE_LOG0("\n");
            NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
            NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

            nn::os::WaitEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

            NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

            if (deviceCount <= 2)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, deviceCount, 10000);
                WaitGattClientCondition(1, 0, 0, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestActive()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Auto
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Auto\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Auto));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local4)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_User8)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_4\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToActive(1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_Active);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_4;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_4
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToActive(1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_Active);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToActive(1);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        // Scan で発見するより先に、HID がBluetoothMode_Auto に書き換えてしまうため、あらためてActive に設定しなおす
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestStaticJoy2Local8()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        NN_ABORT_UNLESS(nn::btm::system::ResultPairingFailureLdnEnabled().Includes(nn::btm::system::StartGamepadPairing()));
        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, true, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_None
        // Usecase 上差があるが、Dynamic2Slot なので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_None\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_None));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy2Local8(1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(1);

        // BrEdr が1台なら、BLE も1 台接続可能
        WaitBleConnection(1);
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 10000);
        WaitGattClientCondition(1, false, 0, 10000);

        // BrEdr が1台なら、2台目以降の BLE は接続不可
        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local8(2);

        // BrEdr が2台なら、BLE は接続不可
        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestStaticJoy4Local4()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4Local4(1);

        NN_ABORT_UNLESS(nn::btm::system::ResultPairingFailureLdnEnabled().Includes(nn::btm::system::StartGamepadPairing()));
        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4Local4(1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, true, 1, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4Local4(1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        // Usecase 上差がある1, 3 で見る
        for (int deviceCount = 0; deviceCount < 5; ++deviceCount)
        {
            if (deviceCount != 1 && deviceCount != 3)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy4Local4(deviceCount);

            if (deviceCount <= 2)
            {
                nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, deviceCount, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
            }
            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_User8
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4Local4(1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_None
        // Usecase 上差がないので、1 でのみ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_None\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy4Local4(1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_None));
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy4Local4(1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        // Usecase 上差がある、1, 3 で見る
        for (int deviceCount = 1; deviceCount < 5; ++deviceCount)
        {
            if (deviceCount != 1 && deviceCount != 3)
            {
                continue;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy4Local4(deviceCount);

            nn::os::ClearEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

            NN_TEST_BTM_MODULE_LOG0("\n");
            NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
            NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

            nn::os::WaitEvent(&m_BleScanEvent);
            NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

            NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

            if (deviceCount <= 2)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
                WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, deviceCount, 10000);
                WaitGattClientCondition(1, 0, 0, 10000);
            }
            else
            {
                NN_ABORT_UNLESS(nn::btm::user::ResultInvalidOperationForCurrentUsecase().Includes(nn::btm::user::BleConnect(bleScanResult[0].address)));
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestBle1()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        for (int brEdrDeviceCount = 0; brEdrDeviceCount < 3; ++brEdrDeviceCount)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy2Local4BleN(brEdrDeviceCount, 1);

            if (brEdrDeviceCount > 1)
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));
                WaitDeviceCondition(brEdrDeviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, brEdrDeviceCount, 10000);
                WaitGattClientCondition(1, false, 0, 10000);
            }

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_User8
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_User8)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy2Local4BleN(1, 1);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 1);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
        WaitGattClientCondition(2, 0, 0, 10000);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_Usecase::TestBle2()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        for (int brEdrDeviceCount = 0; brEdrDeviceCount < 2; ++brEdrDeviceCount)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
            SetToStaticJoy2Local4BleN(brEdrDeviceCount, 2);

            if (brEdrDeviceCount > 0)
            {
                NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));
                WaitDeviceCondition(brEdrDeviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, brEdrDeviceCount, 10000);
                WaitGattClientCondition(2, false, 0, 10000);
            }

            NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
        }

        // Set Wlan Mode WlanMode_User8
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_User8)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy2Local4BleN(1, 2);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 2);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
        WaitGattClientCondition(3, 0, 0, 10000);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    }

    void TestBtmModule_Usecase::TestBle3()
    {
        nn::btm::DeviceConditionList            deviceConditionList;
        nn::btm::DeviceSlotModeList             slotModeList;
        nn::btm::user::BleAdvFilterForGeneral   bleScanFilter;
        nn::btm::user::ScanResult               bleScanResult[nn::bluetooth::BleScanResultCountMax / 2];

        // Slot Saving for pairing
        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 10000);
        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Enable Slot Saving
        NN_TEST_BTM_MODULE_LOG("%s: Enable slot saving\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        nn::btm::EnableSlotSaving(true);

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Bluetooth Mode Active
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetBluetoothMode(nn::btm::BluetoothMode_Active)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local4
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 10000);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_Local8
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_Local8)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Wlan Mode WlanMode_User8
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetWlanMode(nn::btm::WlanMode_User8)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Set Slot Mode SlotMode_2
        // Usecase 上差がないので、1 だけ見る
        NN_TEST_BTM_MODULE_LOG("%s: Set %d of %d connection to SlotMode_2\n", NN_CURRENT_FUNCTION_NAME, 1, 1);
        SetToStaticJoy2Local4BleN(1, 3);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.device[0].hidDeviceCondition.slotMode, nn::btm::SlotMode_4);

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < slotModeList.deviceCount; ++i)
        {
            slotModeList.device->bdAddress = deviceConditionList.device[i].bdAddress;
            slotModeList.device->slotMode = nn::btm::SlotMode_2;
        }

        NN_ABORT_UNLESS(nn::btm::ResultInvalidUsecase().Includes(TrySetSlotMode(slotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Connect BLE device
        NN_TEST_BTM_MODULE_LOG("%s: Connect to BLE device\n", NN_CURRENT_FUNCTION_NAME);
        SetToStaticJoy2Local4BleN(0, 3);

        nn::os::ClearEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&bleScanFilter, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(bleScanFilter));

        NN_TEST_BTM_MODULE_LOG0("\n");
        NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
        NN_TEST_BTM_MODULE_LOG0("Start BLE advertising with a specific packet\n");

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(bleScanResult, NN_ARRAY_SIZE(bleScanResult)), 0);

        NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleConnect(bleScanResult[0].address));
        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
        WaitGattClientCondition(4, 0, 0, 10000);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    }

    void TestBtmModule_Usecase::SetPlain()
    {
        DisconnectAllControllers();
        UnpairAllControllers();
        DisconnectAllBleDevices();
        UnpairAllBleDevices();

        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Auto));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_None));
        nn::btm::EnableSlotSaving(false);

        if (nn::btm::system::IsGamepadPairingStarted())
        {
            nn::os::ClearEvent(&m_GamepadPairingEvent);
            nn::btm::system::CancelGamepadPairing();
            nn::os::WaitEvent(&m_GamepadPairingEvent);
        }

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
    }

    void TestBtmModule_Usecase::SetToDynamic2Slot2(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 0, 3);

        SetPlain();

        if (deviceCount == 0)
        {
            // 何もしない
        }
        else
        {
            WaitControllerConnection(deviceCount);
        }

        // 余計な接続は切る
        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

        if (deviceConditionList.deviceCount < deviceCount)
        {
            return SetToDynamic2Slot2(deviceCount);
        }

        for (int i = deviceCount; i < deviceConditionList.deviceCount; ++i)
        {
            nn::btm::HidDisconnect(&deviceConditionList.device[i].bdAddress);
        }

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
    }

    void TestBtmModule_Usecase::SetToDynamic2Slot6(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 3, 7);

        SetPlain();

        WaitControllerConnection(deviceCount);

        // 余計な接続は切る
        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        if (deviceConditionList.deviceCount < deviceCount)
        {
            return SetToDynamic2Slot6(deviceCount);
        }

        for (int i = deviceCount; i < deviceConditionList.deviceCount; ++i)
        {
            nn::btm::HidDisconnect(&deviceConditionList.device[i].bdAddress);
        }

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
    }

    void TestBtmModule_Usecase::SetToDynamic2Slot8(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 7, 9);

        SetPlain();

        WaitControllerConnection(deviceCount);

        // 余計な接続は切る
        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        if (deviceConditionList.deviceCount < deviceCount)
        {
            return SetToDynamic2Slot8(deviceCount);
        }

        for (int i = deviceCount; i < deviceConditionList.deviceCount; ++i)
        {
            nn::btm::HidDisconnect(&deviceConditionList.device[i].bdAddress);
        }

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);
    }

    void TestBtmModule_Usecase::SetToStaticJoy4(uint8_t deviceCount, uint8_t largeSlotDeviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 2, 5);
        NN_ABORT_UNLESS_RANGE(largeSlotDeviceCount, 2, 5);
        NN_ABORT_UNLESS_LESS_EQUAL(largeSlotDeviceCount, deviceCount);

        if (deviceCount <= 2)
        {
            SetToDynamic2Slot2(deviceCount);
        }
        else
        {
            SetToDynamic2Slot6(deviceCount);
        }

        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_GREATER_EQUAL(deviceConditionList.deviceCount, deviceCount);

        // largeSlotDeviceCount 台のスロットモードを変更する
        nn::btm::DeviceSlotModeList slotModeList;

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < deviceConditionList.deviceCount; ++i)
        {
            slotModeList.device[i].bdAddress = deviceConditionList.device[i].bdAddress;

            if (i < largeSlotDeviceCount)
            {
                slotModeList.device[i].slotMode = nn::btm::SlotMode_4;
            }
            else
            {
                slotModeList.device[i].slotMode = deviceConditionList.device[i].hidDeviceCondition.slotMode;
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, largeSlotDeviceCount, 10000);
    }

    void TestBtmModule_Usecase::SetToStaticJoy8(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 1, 9);

        if (deviceCount <= 2)
        {
            SetToDynamic2Slot2(deviceCount);
        }
        else if(3 <= deviceCount && deviceCount <= 6)
        {
            SetToDynamic2Slot6(deviceCount);
        }
        else
        {
            SetToDynamic2Slot8(deviceCount);
        }

        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

        // 1 台のスロットモードを変更する
        nn::btm::DeviceSlotModeList slotModeList;

        slotModeList.deviceCount = deviceConditionList.deviceCount;

        for (int i = 0; i < deviceConditionList.deviceCount; ++i)
        {
            slotModeList.device[i].bdAddress = deviceConditionList.device[i].bdAddress;

            if (i == 0)
            {
                slotModeList.device[i].slotMode = nn::btm::SlotMode_4;
            }
            else
            {
                slotModeList.device[i].slotMode = deviceConditionList.device[i].hidDeviceCondition.slotMode;
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(slotModeList));

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 10000);
    }

    void TestBtmModule_Usecase::SetToActive(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 0, 9);

        if (deviceCount <= 2)
        {
            SetToDynamic2Slot2(deviceCount);
        }
        else if (3 <= deviceCount && deviceCount <= 6)
        {
            SetToDynamic2Slot6(deviceCount);
        }
        else
        {
            SetToDynamic2Slot8(deviceCount);
        }

        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);
    }

    void TestBtmModule_Usecase::SetToStaticJoy2Local8(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 0, 3);

        SetToDynamic2Slot2(deviceCount);

        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, deviceCount, 10000);
    }

    void TestBtmModule_Usecase::SetToStaticJoy4Local4(uint8_t deviceCount)
    {
        NN_ABORT_UNLESS_RANGE(deviceCount, 0, 5);

        if (deviceCount <= 2)
        {
            SetToDynamic2Slot2(deviceCount);
        }
        else
        {
            SetToDynamic2Slot6(deviceCount);
        }

        nn::btm::DeviceConditionList deviceConditionList;
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, deviceCount);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));

        WaitDeviceCondition(deviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, deviceCount, 10000);
    }

    void TestBtmModule_Usecase::SetToStaticJoy2Local4BleN(uint8_t brEdrDeviceCount, uint8_t bleDeviceCount)
    {
        SetToDynamic2Slot2(brEdrDeviceCount);

        WaitBleConnection(bleDeviceCount);

        // 余計な接続は切る
        nn::btm::user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

        uint8_t connectionCount = nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState));
        if (connectionCount < bleDeviceCount)
        {
            return SetToStaticJoy2Local4BleN(brEdrDeviceCount, bleDeviceCount);
        }

        for (int i = bleDeviceCount; i < connectionCount; ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleDisconnect(connState[i].connectionHandle));
        }

        WaitDeviceCondition(brEdrDeviceCount, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, brEdrDeviceCount, 10000);
        WaitGattClientCondition(bleDeviceCount, false, 0, 10000);
    }

    nn::Result TestBtmModule_Usecase::TrySetBluetoothMode(nn::btm::BluetoothMode bluetoothMode)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetBluetoothMode(bluetoothMode);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_Usecase::TrySetWlanMode(nn::btm::WlanMode wlanMode)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetWlanMode(wlanMode);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_Usecase::TrySetSlotMode(const nn::btm::DeviceSlotModeList& slotModeList)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetSlotMode(&slotModeList);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_Usecase::TryStartBleScanGeneral(const nn::btm::user::BleAdvFilterForGeneral& filter)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StartBleScanForGeneral(filter);

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_Usecase::TryStopBleScanGeneral()
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StopBleScanForGeneral();

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }
}   // namespace BtmUsecase

