﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os/os_EventTypes.h>

#include <nn/btm/btm_Types.h>

#define TEST_NAME "testBtmModule"

#define NN_TEST_BTM_MODULE_LOG0(...)            NN_LOG("[" TEST_NAME "]: %s", ##__VA_ARGS__)
#define NN_TEST_BTM_MODULE_LOG(format, ...)     NN_LOG("[" TEST_NAME "]: " format, ##__VA_ARGS__)

//! テストのベースとなるインタフェースクラス
class TestBtmModule_TestBase
{
public:
    //!< デストラクタ
    ~TestBtmModule_TestBase();

    //!< ConnectedDeviceConditionEvent を処理する
    virtual void HandleConnectedDeviceConditionEvent();

    //!< RegisterDeviceInfoEvent を処理する
    virtual void HandleRegisterDeviceInfoEvent();

    //!< LlrStateEvent を処理する
    virtual void HandleLlrStateEvent();

    //!< RadioEvent を処理する
    virtual void HandleRadioEvent();

    //!< GamepadPairingEvent を処理する
    virtual void HandleGamepadPairingEvent();

    //!< BleScanEvent を処理する
    virtual void HandleBleScanEvent();

    //!< BleConnectionEvent を処理する
    virtual void HandleBleConnectionEvent();

    //!< BlePairingEvent を処理する
    virtual void HandleBlePairingEvent();

    //!< BleServiceDiscoveryEvent を処理する
    virtual void HandleBleServiceDiscoveryEvent();

    //!< BleMtuConfigEvent を処理する
    virtual void HandleBleMtuConfigEvent();

protected:
    //!< 内部のブロック解除用イベントのタイプ。値は nn::btm::SystemEventContainer::EventType に準拠
    enum InternalEventType
    {
        InternalEventType_ConnectedDeviceCondition  = 1,
        InternalEventType_RegisterDeviceInfo        = 2,
        InternalEventType_LlrState                  = 6,

        InternalEventType_Radio                     = 4,
        InternalEventType_GamepadPairing            = 5,

        InternalEventType_BleScan                   = 7,
        InternalEventType_BleConnection             = 8,
        InternalEventType_BlePairing                = 9,
        InternalEventType_BleServiceDiscovery       = 10,
        InternalEventType_BleMtuConfig              = 11,
    };

    //!< コンストラクタ
    TestBtmModule_TestBase();

    //!< ConnectedDeviceConditionEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_ConnectedDeviceConditionEvent;

    //!< RegisterDeviceEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_RegisterDeviceInfoEvent;

    //!< LlrStateEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_LlrStateEvent;

    //!< RadioEvent がシグナルされたとき、内部のブロ国を解くためのイベント
    nn::os::EventType m_RadioEvent;

    //!< GamepadPairingEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_GamepadPairingEvent;

    //!< BleScanEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_BleScanEvent;

    //!< BleConnection がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_BleConnectionEvent;

    //!< BlePairingEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_BlePairingEvent;

    //!< BleServiceDiscoveryEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_BleServiceDiscoveryEvent;

    //!< BleMtuConfigEvent がシグナルされたとき、内部のブロックを解くためのイベント
    nn::os::EventType m_BleMtuConfigEvent;

    //!< 無線接続中の全コントローラーを切断します
    void DisconnectAllControllers();

    //!< 全コントローラーのペアリングを削除します
    void UnpairAllControllers();

    //!< DisconnectAllControllers()、 UnpairAllControllers() 後、deviceCount 台以上のコントローラーのペアリング、接続を待ち受けます
    void WaitControllerConnection(uint8_t deviceCount);

    //!< DeviceConditionList が指定したパラメータになるまで、指定した時間(msec)待機します
    void WaitDeviceCondition(uint8_t deviceCount, nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode,
                             bool isSlotSavingForPairing, bool isSlotSaving, uint8_t largeSlotDeviceCount,
                             uint64_t durationMsec);

    //!< 全BLE デバイスを切断します
    void DisconnectAllBleDevices();

    //!< 全BLE デバイスのペアリングを削除します
    void UnpairAllBleDevices();

    //!< DisconnectAllBleDevices(), UnpairAllBleDevices() 後、1 台以上のBLE デバイスの接続を待ち受けます
    void WaitBleConnection(uint8_t deviceCount);

    //!< BleClientConnState が指定したパラメータになるまで、指定した時間(msec)待機します
    void WaitGattClientCondition(uint8_t deviceCount,
                                 bool largeCeLengthRequired, uint8_t largeCeLengthDeviceCount,
                                 uint64_t durationMsec);

    //!< Radio OFF にします
    void DisableRadio();
};

inline void HandleUnexpectedEvent()
{
    NN_ABORT(TEST_NAME ": This Event is not unexpected for this test.\n");
}

class TestBtmModule_Init : public TestBtmModule_TestBase
{
public:
    //!< ConnectedDeviceConditionEvent を処理する
    virtual void HandleConnectedDeviceConditionEvent() { TestBtmModule_TestBase::HandleConnectedDeviceConditionEvent(); }

    //!< RegisterDeviceInfoEvent を処理する
    virtual void HandleRegisterDeviceInfoEvent() { TestBtmModule_TestBase::HandleRegisterDeviceInfoEvent(); }

    //!< LlrStateEvent を処理する
    virtual void HandleLlrStateEvent(){ HandleUnexpectedEvent(); }

    //!< RadioEvent を処理する
    virtual void HandleRadioEvent() { TestBtmModule_TestBase::HandleRadioEvent(); }

    //!< GamepadPairingEvent を処理する
    virtual void HandleGamepadPairingEvent() { TestBtmModule_TestBase::HandleGamepadPairingEvent(); }

    //!< BleScanEvent を処理する
    virtual void HandleBleScanEvent() { HandleUnexpectedEvent(); }

    //!< BleConnectionEvent を処理する
    virtual void HandleBleConnectionEvent() { TestBtmModule_TestBase::HandleBleConnectionEvent(); }

    //!< BlePairingEvent を処理する
    virtual void HandleBlePairingEvent() { TestBtmModule_TestBase::HandleBlePairingEvent(); }

    //!< BleServiceDiscoveryEvent を処理する
    virtual void HandleBleServiceDiscoveryEvent() { HandleUnexpectedEvent(); }

    //!< BleMtuConfigEvent を処理する
    virtual void HandleBleMtuConfigEvent() { TestBtmModule_TestBase::HandleBleMtuConfigEvent(); }

    //!< Radio On する
    void ResetRadio();

    //!< Stop gamepad pairing する
    void ResetGamepadPairing();

    //!< Device Info をクリアする
    void ResetDeviceInfo();

    //!< BLE の状態をリセットする
    void ResetBle();

    //!< Wlan Mode をWlanMode_None にする
    void ResetWlanMode();

    //!< Bluetooth Mode をBluetoothMode_Auto にする
    void ResetBluetoothMode();

    //!< SlotSaving をfalse にする
    void ResetSlotSaving();

    //!< Device Info のプロテクトを解除する
    void ResetDeviceInfoProtection();
};

