﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/oe.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nn/nfp/nfp_Lib.h>
#include <nn/nfp/nfp_TagApi.h>
#include <nn/nfp/nfp_MiscApi.h>
#include <nn/nfp/nfp_NpadApi.h>

#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>
#include "../Common/testBtm_Utility.h"

nn::os::SystemEventType g_SystemEventForCdc;
nn::os::SystemEventType g_SystemEventForRdi;
nn::os::SystemEventType g_SystemEventForRadio;
nn::os::SystemEventType g_SystemEventForGpp;
nn::os::SystemEventType g_SystemEventForLlr;

namespace testUtil
{
uint8_t CalcDeviceCountCap(nn::btm::WlanMode wlanMode, uint8_t largeSlotCount)
{
    if(wlanMode == nn::btm::WlanMode_Local8)
        return 2;
    else if(wlanMode == nn::btm::WlanMode_Local4)
        return 4;
    else if(largeSlotCount > 1)
        return 4;
    else
        return 8;
}

nn::btm::SniffMode CalcSniffMode(nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode, bool isPairing, bool isSlotSaving, uint8_t deviceCount, uint8_t largeSlotCount)
{
    if(isPairing)
        return nn::btm::SniffMode_15ms;
    if(isSlotSaving)
        return nn::btm::SniffMode_15ms;
    if(wlanMode == nn::btm::WlanMode_Local4)
        return nn::btm::SniffMode_15ms;
    if(wlanMode == nn::btm::WlanMode_Local8)
        return nn::btm::SniffMode_15ms;
    if(wlanMode == nn::btm::WlanMode_User8)
        return nn::btm::SniffMode_15ms;
    if(bluetoothMode == nn::btm::BluetoothMode_Active)
        return nn::btm::SniffMode_Active;
    if(bluetoothMode == nn::btm::BluetoothMode_Auto)
    {
        if(largeSlotCount > 0)
            return nn::btm::SniffMode_15ms;
        if(deviceCount <= 2)
            return nn::btm::SniffMode_5ms;
        if(deviceCount <= 6)
            return nn::btm::SniffMode_10ms;
        else
            return nn::btm::SniffMode_15ms;
    }
    NN_ABORT("[testBtm_Auto]Invalid usecase.\n");
}

nn::btm::SlotMode CalcDefaultSlotMode(nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode)
{
    if(wlanMode == nn::btm::WlanMode_Local4)
        return nn::btm::SlotMode_4;
    else if(wlanMode == nn::btm::WlanMode_Local8)
        return nn::btm::SlotMode_4;
    else if(bluetoothMode == nn::btm::BluetoothMode_Active)
        return nn::btm::SlotMode_Active;
    else
        return nn::btm::SlotMode_2;
}

enum CheckDclResult{
    CheckDclResult_Success,
    CheckDclResult_DeviceCountErr,
    CheckDclResult_BtModeErr,
    CheckDclResult_WlModeErr,
    CheckDclResult_IsPairingErr,
    CheckDclResult_ProfileErr,
    CheckDclResult_AddressErr,
    CheckDclResult_NameErr,
    CheckDclResult_SniffErr,
    CheckDclResult_SlotErr,
    CheckDclResult_DeviceCountCapErr,
    CheckDclResult_IsSlotSavingErr,
};
CheckDclResult CheckDcl(nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode, bool isPairing, bool isSlotSaving, uint8_t deviceCount, uint8_t largeSlotCount )
{
    nn::btm::DeviceConditionList dcl;
    nn::btm::GetConnectedDeviceCondition(&dcl);

    if(dcl.deviceCount != deviceCount)
        return CheckDclResult_DeviceCountErr;

    //NXで期待される値
    const char deviceName[nn::btm::SIZE_OF_BDNAME] = "Joy-Con";//NULL終端。NintendoGamepad, Pro Controllerはここでは除外
    static const int DN_SIZE = 7;//NULL含まないサイズ

    if(dcl.bluetoothMode != bluetoothMode)
        return CheckDclResult_BtModeErr;

    if(dcl.wlanMode != wlanMode)
        return CheckDclResult_WlModeErr;

    if(dcl.isSlotSavingForPairing != isPairing)
        return CheckDclResult_IsPairingErr;

    if(dcl.isSlotSaving != isSlotSaving)
        return CheckDclResult_IsSlotSavingErr;

    uint8_t deviceCountCap;
    deviceCountCap = CalcDeviceCountCap(wlanMode, largeSlotCount);
    if(dcl.deviceCountCapacity != deviceCountCap)
        return CheckDclResult_DeviceCountCapErr;

    nn::btm::SniffMode sniffMode;
    nn::btm::SlotMode slotMode;
    sniffMode = CalcSniffMode(bluetoothMode, wlanMode, isPairing, isSlotSaving, deviceCount, largeSlotCount);
    slotMode = CalcDefaultSlotMode(bluetoothMode, wlanMode);

    uint8_t lsCount = 0;
    for(int i=0;i<dcl.deviceCount;i++)
    {
        if(dcl.device[i].profile != nn::btm::Profile_Hid)
            return CheckDclResult_ProfileErr;

        if(dcl.device[i].bdAddress.address[0] == 0x00)//OUI 0x00始まりのみチェックする
            return CheckDclResult_AddressErr;

        int valid;
        valid = memcmp(&dcl.device[i].bdName.name[0], &deviceName[0], DN_SIZE);
        if(valid != 0)
            return CheckDclResult_NameErr;

        if(dcl.device[i].hidDeviceCondition.sniffMode != sniffMode)
            return CheckDclResult_SniffErr;

        //[Todo]Activeへの明示変更をケア
        if(largeSlotCount == 0)
        {
            if(dcl.device[i].hidDeviceCondition.slotMode != slotMode)
                return CheckDclResult_SlotErr;
        }

        if(dcl.device[i].hidDeviceCondition.slotMode == nn::btm::SlotMode_4)
        {
            lsCount++;
        }

        //NN_SDK_REQUIRES(dcl.device[i].hidDeviceCondition.zeroRetransmissionList);//[Todo]実装され次第追加
        //NN_SDK_REQUIRES(dcl.device[i].hidDeviceCondition.vid == );//[Todo]実装され次第追加
        //NN_SDK_REQUIRES(dcl.device[i].hidDeviceCondition.pid == );//[Todo]実装され次第追加
    }

    if(largeSlotCount != lsCount)
        return CheckDclResult_SlotErr;

    return CheckDclResult_Success;
}

void WaitDcl(nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode, bool isPairing, bool isSlotSaving, uint8_t deviceCount, uint8_t largeSlotCount )
{
    for(;;)
    {
        nn::os::WaitSystemEvent(&g_SystemEventForCdc);
        CheckDclResult result = CheckDcl(bluetoothMode, wlanMode, isPairing, isSlotSaving, deviceCount, largeSlotCount);
        if(result == CheckDclResult_Success)
            break;
    }
}

void WaitDcl(uint8_t deviceCount, uint8_t largeSlotCount)
{
    WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, deviceCount, largeSlotCount);
}

void ValidateDcl(nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode, bool isPairing, bool isSlotSaving, uint8_t deviceCount, uint8_t largeSlotCount )
{
    CheckDclResult result = CheckDcl(bluetoothMode, wlanMode, isPairing, isSlotSaving, deviceCount, largeSlotCount);
    if(result == CheckDclResult_Success)
    {

    }
    else
    {
        NN_ABORT("[TestBtm_Auto]Invalid Dcl. enum(%d)",result);
    }
}

void CheckAndPrindHdp(const nn::btm::HostDeviceProperty& hdp)
{
    NN_LOG("[TestBtm_Auto]BdAddress   = ");nnt::btm::PrintBdAddress(hdp.bdAddress);
    NN_LOG("[TestBtm_Auto]DeviceName  = %s\n",hdp.bdName);
    NN_LOG("[TestBtm_Auto]Cod         = %02X %02X %02X\n", hdp.classOfDevice.cod[0], hdp.classOfDevice.cod[1],hdp.classOfDevice.cod[2]);
    NN_LOG("[TestBtm_Auto]Feature set = 0x%02X\n",hdp.featureSet);

    //NXで期待される値
    const char hostBdName[]          = "Nintendo Switch";//終端文字含む
    const uint8_t hostBdNameSize     = 16;//終端文字含む
    const nn::btm::ClassOfDevice cod = {{0x00,0x04,0x3C}};
    const uint8_t featureSet         = 0x68;

    //比較。BdAddressは比較しない
    //[Todo]BdAddressのOUI判定
    int valid;
    valid = memcmp(&hdp.bdName.name[0], &hostBdName[0], hostBdNameSize);
    NN_SDK_REQUIRES(valid == 0);
    valid = memcmp(&hdp.classOfDevice.cod[0], &cod.cod[0], nn::btm::SIZE_OF_COD);
    NN_SDK_REQUIRES(valid == 0);
    NN_SDK_REQUIRES(hdp.featureSet == featureSet);
    NN_UNUSED(featureSet);
}

void ToggleNfp(bool isEnable)
{
    //[Todo]ひとまずはじめに見つけたNFCをトグルしているが、より細かく指定できるようにする
    static const int maxCountOfJoyConR = 4;//右コンは4台まで想定
    nn::nfp::DeviceHandle deviceHandle[maxCountOfJoyConR];
    int count;
    nn::Result result;

    static const int cycleCountMax = 5;
    static const int delayForRetryMs = 1000;
    for(int i=0;i<cycleCountMax;i++)
    {
        result = nn::nfp::ListDevices(&deviceHandle[0], &count, maxCountOfJoyConR);
        if(result.IsSuccess())
            break;
        if(i == cycleCountMax - 1)
            NN_ABORT("[TestBtm_Auto]nfp::ListDevices failed.\n");
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delayForRetryMs));
    }
    NN_SDK_REQUIRES(count != 0);

    if(isEnable)
    {
        result = nn::nfp::StartDetection(deviceHandle[0]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else
    {
        result = nn::nfp::StopDetection(deviceHandle[0]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

//Interval()直後と同等の状態でのみ使用可能
inline void WaitConnection(const char* pControllerName)
{
    NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please joint a %s.\n", pControllerName);
    //登録確認
    {
        nn::btm::DeviceInfoList deviceInfoList;
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_SDK_REQUIRES(deviceInfoList.deviceCount == 1);//SDK以外のマクロ無いか
    }

    NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please unjoint a %s.\n", pControllerName);
    //接続確認
    {
        testUtil::WaitDcl(1, 0);
    }
}

}// namespace testUtil



namespace testSys
{
void Interval()
{
    //テスト開始条件固定

    //RadioOn
    {
        nn::btm::EnableRadio(true);//同期API
        nn::btm::BtmState state;
        nn::btm::GetState(&state);
        NN_SDK_REQUIRES(state == nn::btm::BtmState_Initialized);
    }

    //ペアリング停止
    {
        nn::btm::system::CancelGamepadPairing();
    }

    //全削除
    {
        nn::btm::DeviceInfoList targetDeviceInfoList;
        nn::btm::GetDeviceInfo(&targetDeviceInfoList);
        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        for(int i=0;i<targetDeviceInfoList.deviceCount;i++)
        {
            nn::btm::RemoveDeviceInfo(&targetDeviceInfoList.device[i].bdAddress);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        }
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 0);//SDK以外のマクロ無いか
    }

    //Wlan, Bluetoothモードリセット
    {
        nn::Result result;
        for(;;)
        {
            result = nn::btm::SetWlanMode(nn::btm::WlanMode_None);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        for(;;)
        {
            result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Auto);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
    }

    //SlotSaving無効
    {
        nn::btm::EnableSlotSaving(false);
    }

    //プロテクト情報クリア
    {
        nn::btm::debug::GeneralTest(11);
    }

    //Dclが規定値になるまでWait
    testUtil::WaitDcl(0, 0);
    NN_SDK_REQUIRES(nn::btm::IsLlrStarted() == false);

    //イベントクリア
    {
        nn::os::ClearSystemEvent(&g_SystemEventForCdc);
        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::os::ClearSystemEvent(&g_SystemEventForRadio);
        nn::os::ClearSystemEvent(&g_SystemEventForGpp);
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
    }
}

void Initialize()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    {
        nn::oe::Initialize();
        nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);
        nn::oe::EnterExitRequestHandlingSection();
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::nfp::Initialize();
    }

    //セッション初期化
    {
        nn::btm::InitializeBtmInterface();
        nn::btm::debug::InitializeBtmDebugInterface();
    }

    //システムイベント登録
    {
        nn::btm::RegisterSystemEventForConnectedDeviceCondition(&g_SystemEventForCdc);
        nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&g_SystemEventForRdi);
        nn::btm::system::AcquireRadioEvent(&g_SystemEventForRadio);
        nn::btm::system::AcquireGamepadPairingEvent(&g_SystemEventForGpp);
        nn::btm::AcquireLlrStateEvent(&g_SystemEventForLlr);
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void Finalize()
{
    //セッション末期化
    {
        nn::btm::FinalizeBtmInterface();
        nn::btm::debug::FinalizeBtmDebugInterface();
    }

    {
        nn::nfp::Finalize();
        nn::oe::LeaveExitRequestHandlingSection();
    }
}

}//namespace testSys



namespace testSuite
{

namespace function
{

void Interface()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //セッション多重初期化、多重末期化
    {
        nn::btm::FinalizeBtmInterface();
        nn::btm::InitializeBtmInterface();
        nn::btm::InitializeBtmInterface();
        nn::btm::FinalizeBtmInterface();
        nn::btm::FinalizeBtmInterface();
        nn::btm::InitializeBtmInterface();
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void Btm()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //State取得
    {
        nn::btm::BtmState state;
        nn::btm::GetState(&state);
        NN_SDK_REQUIRES(state == nn::btm::BtmState_Initialized);
    }

    //HostDevicePropery取得
    {
        nn::btm::HostDeviceProperty hdp;
        nn::btm::GetHostDeviceProperty(&hdp);
        testUtil::CheckAndPrindHdp(hdp);
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void PowerMode()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    testUtil::WaitConnection("Joy-Con");

    nn::btm::DeviceConditionList dcl;
    nn::btm::GetConnectedDeviceCondition(&dcl);

    //SlotSaving有効
    {
        nn::btm::EnableSlotSaving(true);
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 1, 0);
    }

    //SlotSaving無効
    {
        nn::btm::EnableSlotSaving(false);
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 0);
    }

    //接続ありRadioOff
    {
        nn::btm::EnableRadio(false);
        nn::btm::BtmState state;
        nn::btm::GetState(&state);
        NN_SDK_REQUIRES(state == nn::btm::BtmState_RadioOff);
        WaitSystemEvent(&g_SystemEventForRadio);
        testUtil::WaitDcl(0, 0);
    }

    //RadioOn
    {
        nn::btm::EnableRadio(true);
        nn::btm::BtmState state;
        nn::btm::GetState(&state);
        NN_SDK_REQUIRES(state == nn::btm::BtmState_Initialized);
        WaitSystemEvent(&g_SystemEventForRadio);
    }

    //未ペアリングデバイスへのLLR
    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
        nn::btm::BdAddress bdAddress = {{0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF}};
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&bdAddress);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else if(nn::btm::ResultFailureLowLayerGeneral::Includes(result))
            {
                //期待の結果
                //下層はResultBadParameter
                break;
            }
            else
            {
                NN_ABORT("[testBtm_Auto]Invalid Result.\n");
                break;
            }
        }
        if(nn::os::TryWaitSystemEvent(&g_SystemEventForLlr))
            NN_ABORT("[testBtm_Auto]Invalid Signal.\n");
        if(nn::btm::IsLlrStarted())
            NN_ABORT("[testBtm_Auto]Invalid LLR State.\n");
    }

    //不在デバイスへのLLR発行
    {
        nn::btm::DeviceInfoList deviceInfoList;
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_SDK_REQUIRES(deviceInfoList.deviceCount == 1);

        deviceInfoList.device[0].bdAddress.address[3] = 0xAA;
        deviceInfoList.device[0].bdAddress.address[4] = 0xAA;
        deviceInfoList.device[0].bdAddress.address[5] = 0xAA;

        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);

        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&deviceInfoList.device[0].bdAddress);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }

        //API Success後、タイムアウトでの停止を期待
        for(;;)
        {
            nn::os::WaitSystemEvent(&g_SystemEventForLlr);
            if(!nn::btm::IsLlrStarted())
                break;
        }
    }

    //LLR
    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&dcl.device[0].bdAddress);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(1, 0);
        WaitSystemEvent(&g_SystemEventForLlr);
        NN_SDK_REQUIRES(nn::btm::IsLlrStarted() == false);
    }

    //接続済みデバイスへの　LLR
    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&dcl.device[0].bdAddress);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else if(nn::btm::ResultFailureLowLayerGeneral::Includes(result))
            {
                //期待の結果
                //下層はResultConnectionError
                break;
            }
            else
            {
                NN_ABORT("[testBtm_Auto]Invalid Result.\n");
                break;
            }
        }
        if(nn::os::TryWaitSystemEvent(&g_SystemEventForLlr))
            NN_ABORT("[testBtm_Auto]Invalid Signal.\n");
        if(nn::btm::IsLlrStarted())
            NN_ABORT("[testBtm_Auto]Invalid LLR State.\n");
    }

    //ペアリング情報異常デバイスへのLLR
    {
        //切断
        nn::btm::DeviceConditionList dcList;
        nn::btm::GetConnectedDeviceCondition(&dcList);
        NN_SDK_REQUIRES(dcList.deviceCount == 1);
        nn::btm::HidDisconnect(&dcList.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);

        nn::btm::DeviceInfoList diList;
        nn::btm::GetDeviceInfo(&diList);
        NN_SDK_REQUIRES(diList.deviceCount == 2);

        diList.device[0].linkKey.key[3] = 0xAA;
        diList.device[0].linkKey.key[4] = 0xAA;
        diList.device[0].linkKey.key[5] = 0xAA;

        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::btm::AddDeviceInfo(&diList.device[0]);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);

        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&diList.device[0].bdAddress);//[0]が上記で書き換えたデバイス
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }

        //API Success後、デバイスを見つけた後のペアリング情報確認シーケンスでの停止を期待
        for(;;)
        {
            nn::os::WaitSystemEvent(&g_SystemEventForLlr);
            if(!nn::btm::IsLlrStarted())
                break;
        }
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

void DeviceInfo()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please joint a Joy-Con.\n");

    nn::btm::DeviceInfoList deviceInfoList;

    //登録確認
    {
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_SDK_REQUIRES(deviceInfoList.deviceCount == 1);//SDK以外のマクロ無いか
    }

    //削除
    {
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::RemoveDeviceInfo(&deviceInfoList.device[0].bdAddress);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 0);//SDK以外のマクロ無いか
    }

    //境界登録。カウントのみ評価
    {
        nn::btm::DeviceInfo inputDeviceInfo;
        for(int i=0;i<nn::btm::COUNT_OF_DI_LIST;i++)
        {
            inputDeviceInfo = deviceInfoList.device[0];
            inputDeviceInfo.bdAddress.address[5] = i;
            nn::btm::AddDeviceInfo(&inputDeviceInfo);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        }
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == nn::btm::COUNT_OF_DI_LIST);//SDK以外のマクロ無いか
    }

    //ペアリング情報の生存優先度変更
    {
        nn::btm::DeviceInfoList dil;
        nn::btm::GetDeviceInfo(&dil);
        nn::btm::IncreaseDeviceInfoOrder(&dil.device[0].bdAddress);//もっとも古いペアリング情報を指定
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);

        nn::btm::DeviceInfoList currentDil;
        nn::btm::GetDeviceInfo(&currentDil);
        NN_SDK_REQUIRES(currentDil.deviceCount == nn::btm::COUNT_OF_DI_LIST);
        //当該情報が最後尾に回っていることの確認
        bool isValid;
        isValid = nnt::btm::IsSameBdAddress(&dil.device[0].bdAddress, &currentDil.device[nn::btm::COUNT_OF_DI_LIST - 1].bdAddress);
        NN_SDK_REQUIRES(isValid);
        //当該情報以外が前詰めされていることの確認
        for(int i=0;i<nn::btm::COUNT_OF_DI_LIST - 1;i++)
        {
            isValid = nnt::btm::IsSameBdAddress(&dil.device[i + 1].bdAddress, &currentDil.device[i].bdAddress);
            NN_SDK_REQUIRES(isValid);
        }
    }

    //境界越え登録。カウントのみ評価
    {
        nn::btm::DeviceInfo inputDeviceInfo;
        inputDeviceInfo = deviceInfoList.device[0];
        inputDeviceInfo.bdAddress.address[5] = 0xFF;
        nn::btm::AddDeviceInfo(&inputDeviceInfo);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == nn::btm::COUNT_OF_DI_LIST);//SDK以外のマクロ無いか
    }

    //全削除
    {
        nn::btm::DeviceInfoList targetDeviceInfoList;
        nn::btm::GetDeviceInfo(&targetDeviceInfoList);
        for(int i=0;i<targetDeviceInfoList.deviceCount;i++)
        {
            nn::btm::RemoveDeviceInfo(&targetDeviceInfoList.device[i].bdAddress);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        }
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 0);//SDK以外のマクロ無いか
    }

    //1件書き戻し
    {
        nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 1);//SDK以外のマクロ無いか
    }

    NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please unjoint a Joy-Con.\n");

    //無線接続確認
    {
        testUtil::WaitDcl(1, 0);
        //BdAddress確認
        nn::btm::DeviceConditionList outputDcl;
        nn::btm::GetConnectedDeviceCondition(&outputDcl);
        bool isValid;
        isValid = nnt::btm::IsSameBdAddress(&outputDcl.device[0].bdAddress, &deviceInfoList.device[0].bdAddress);
        NN_SDK_REQUIRES(isValid);
    }

    //削除
    {
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::RemoveDeviceInfo(&deviceInfoList.device[0].bdAddress);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 0);//SDK以外のマクロ無いか
    }

    //Null BdAddressを書き込み
    {
        nn::btm::BdAddress bdAddress ={{0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};
        nn::btm::DeviceInfo deviceInfo;
        deviceInfo.bdAddress = bdAddress;
        nn::btm::AddDeviceInfo(&deviceInfo);
        //nn::os::WaitSystemEvent(&g_SystemEventForRdi);//Nullの場合はAddが失敗し、イベントのシグナルもされない
        nn::btm::DeviceInfoList outputDeviceInfoList;
        nn::btm::GetDeviceInfo(&outputDeviceInfoList);
        NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == 0);//SDK以外のマクロ無いか
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

void Hid()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    testUtil::WaitConnection("Joy-Con");

    nn::btm::DeviceConditionList dcl;
    nn::btm::GetConnectedDeviceCondition(&dcl);

    //再送モード変更
    {
        nn::btm::ZeroRetransmissionList zrList;
        zrList.enabledReportId[0]   = 0x01;
        zrList.enabledReportIdCount = 1;
        for(;;)
        {
            auto result = HidSetRetransmissionMode(&dcl.device[0].bdAddress, &zrList);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }

        //[Todo]イベントの評価は下層が実装され次第足す
    }

    //切断
    {
        nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);
    }

    //dbgポートの機能で再接続、再切断
    /*{
        nn::btm::debug::HidConnect(&dcl.device[0].bdAddress);
        testUtil::WaitDcl(1, 0);
        nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);
    }*/

    //切断済みを切断
    {
        nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
        //下層がfatalしなければそれで良い
    }

    //切断済みに再送モード変更
    {
        nn::btm::ZeroRetransmissionList zrList;
        zrList.enabledReportId[0]   = 0x01;
        zrList.enabledReportIdCount = 1;
        for(;;)
        {
            auto result = HidSetRetransmissionMode(&dcl.device[0].bdAddress, &zrList);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }

        //[Todo]イベントの評価は下層が実装され次第足す
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void ConnectedDeviceCondition()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //空DeviceConditionList取得
    {
        nn::btm::DeviceConditionList dcl;
        nn::btm::GetConnectedDeviceCondition(&dcl);
        NN_SDK_REQUIRES(dcl.deviceCount == 0);
    }

    testUtil::WaitConnection("Joy-Con(R)");

    //BurstMode
    {
        //[Todo]
    }

    //WlanMode_Local4
    {
        for(;;)
        {
            auto result = nn::btm::SetWlanMode(nn::btm::WlanMode_Local4);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 1);
    }

    //WlanMode_Local8
    {
        for(;;)
        {
            auto result = nn::btm::SetWlanMode(nn::btm::WlanMode_Local8);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 1);
    }

    //BluetoothMode, SlotMode
    {
        testUtil::ToggleNfp(true);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 1);//[Todo]IsLargeSlotRequiredのチェックを追加
        testUtil::ToggleNfp(false);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 1);//[Todo]IsLargeSlotRequiredのチェックを追加
    }

    //WlanMode_User8
    //メジャーケースではないので、SlotModeとの絡みはLocal8で行い、User8では行わない
    {
        for(;;)
        {
            auto result = nn::btm::SetWlanMode(nn::btm::WlanMode_User8);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_User8, false, false, 1, 0);
    }

    //WlanMode_None
    {
        for(;;)
        {
            auto result = nn::btm::SetWlanMode(nn::btm::WlanMode_None);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(1, 0);
    }

    //BluetoothMode, SlotMode
    {
        testUtil::ToggleNfp(true);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 1);//[Todo]IsLargeSlotRequiredのチェックを追加。btm側のイベントシグナル有効化してWaitDcl
        testUtil::ToggleNfp(false);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 0);//[Todo]IsLargeSlotRequiredのチェックを追加。btm側のイベントシグナル有効化してWaitDcl
    }

    //BluetoothMode_Active
    {
        for(;;)
        {
            auto result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Active);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 1, 0);
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

void DeviceTreatment()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    testUtil::WaitConnection("Joy-Con");

    nn::btm::DeviceInfoList deviceInfoList;
    nn::btm::GetDeviceInfo(&deviceInfoList);

    //切断
    {
        nn::btm::HidDisconnect(&deviceInfoList.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);
    }

    //プロテクトしたペアリング情報が上書きされないこと
    {
        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, true);
        nn::btm::DeviceInfo dummyDi;
        dummyDi = deviceInfoList.device[0];
        dummyDi.bdAddress.address[0] = 0xAA;
        dummyDi.bdAddress.address[1] = 0xBB;
        dummyDi.bdAddress.address[2] = 0xCC;
        for(int i=0;i<15;i++)
        {
            dummyDi.bdAddress.address[5] = i;
            nn::btm::AddDeviceInfo(&dummyDi);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);//最後の書き込みまで保障されるように、都度待つ
        }
        nn::btm::DeviceInfoList diList;
        nn::btm::GetDeviceInfo(&diList);
        bool isFound = false;
        for(int i=0;i<nn::btm::COUNT_OF_DI_LIST;i++)
        {
            if(nnt::btm::IsSameBdAddress(&diList.device[i].bdAddress, &deviceInfoList.device[0].bdAddress))
            {
                isFound = true;
                break;
            }
            else
            {

            }
        }
        NN_SDK_REQUIRES(isFound == true);
    }

    //プロテクト中でも、明示的なペアリング情報の削除ができること
    {
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        uint8_t deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 0);
        NN_UNUSED(deviceCount);
    }

    //プロテクト中でも、ペアリング情報の追加は出来ること。また上書きもできること
    {
        for(int i=0;i<3;i++)
        {
            nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
            nn::btm::DeviceInfoList diList;
            nn::btm::GetDeviceInfo(&diList);
            NN_SDK_REQUIRES(diList.deviceCount == 1);
            if(nnt::btm::IsSameBdAddress(&diList.device[0].bdAddress, &deviceInfoList.device[0].bdAddress))
            {
                //想定の位置に情報があった
            }
            else
            {
                NN_ABORT("[testBtm_Auto]device info not found.\n");
            }
        }
    }

    //接続確認
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Push A button the joycon.\n");
        testUtil::WaitDcl(1, 0);
        nn::btm::HidDisconnect(&deviceInfoList.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);
    }

    //同じプロテクト設定は上書きされること
    {
        nn::btm::GeneralInfoList giList;
        nn::btm::debug::GeneralGet(0, &giList);
        NN_SDK_REQUIRES(giList.count == 1);
        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, true);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
        nn::btm::debug::GeneralGet(0, &giList);
        NN_SDK_REQUIRES(giList.count == 1);
    }

    //プロテクト設定を消せること。消した場合はペアリング情報が上書きされること
    {
        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, false);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
        nn::btm::GeneralInfoList giList;
        nn::btm::debug::GeneralGet(0, &giList);
        NN_SDK_REQUIRES(giList.count == 0);

        uint8_t deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 1);
        NN_UNUSED(deviceCount);

        nn::os::ClearSystemEvent(&g_SystemEventForRdi);

        nn::btm::DeviceInfo dummyDi;
        dummyDi = deviceInfoList.device[0];
        dummyDi.bdAddress.address[0] = 0xAA;
        dummyDi.bdAddress.address[1] = 0xBB;
        dummyDi.bdAddress.address[2] = 0xCC;
        for(int i=0;i<15;i++)
        {
            dummyDi.bdAddress.address[5] = i;
            nn::btm::AddDeviceInfo(&dummyDi);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);//最後の書き込みまで保障されるように、都度待つ
        }
        nn::btm::DeviceInfoList diList;
        nn::btm::GetDeviceInfo(&diList);
        for(int i=0;i<nn::btm::COUNT_OF_DI_LIST;i++)
        {
            if(nnt::btm::IsSameBdAddress(&diList.device[i].bdAddress, &deviceInfoList.device[0].bdAddress))
            {
                NN_ABORT("[testBtm_Auto]unexpected device info.");
            }
            else
            {

            }
        }
    }

    //上限越えのプロテクト設定が出来ること
    {
        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        uint8_t deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 0);
        NN_UNUSED(deviceCount);

        nn::btm::BdAddress dummyAddress = deviceInfoList.device[0].bdAddress;
        dummyAddress.address[0] = 0xAA;
        dummyAddress.address[1] = 0xBB;
        dummyAddress.address[2] = 0xCC;
        for(int i=0;i<15;i++)
        {
            dummyAddress.address[5] = i;
            nn::btm::ProtectDeviceInfo(&dummyAddress, true);
        }
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
        nn::btm::GeneralInfoList giList;
        nn::btm::debug::GeneralGet(0, &giList);
        NN_SDK_REQUIRES(giList.count == nn::btm::COUNT_OF_DI_LIST);
    }

    //上限のプロテクト設定が有効なこと
    {
        //プロテクト設定の追加
        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, true);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
        //ペアリング情報の追加
        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        uint8_t deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 1);
        NN_UNUSED(deviceCount);
        //ペアリング情報の15件追加
        nn::btm::DeviceInfo dummyDi;
        dummyDi = deviceInfoList.device[0];
        dummyDi.bdAddress.address[0] = 0xAB;
        dummyDi.bdAddress.address[1] = 0xCD;
        dummyDi.bdAddress.address[2] = 0xEF;
        for(int i=0;i<15;i++)
        {
            dummyDi.bdAddress.address[5] = i;
            nn::btm::AddDeviceInfo(&dummyDi);
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);//最後の書き込みまで保障されるように、都度待つ
        }
        //ペアリング情報取得。当該情報が残っていること
        nn::btm::DeviceInfoList diList;
        nn::btm::GetDeviceInfo(&diList);
        bool isFound = false;
        for(int i=0;i<nn::btm::COUNT_OF_DI_LIST;i++)
        {
            if(nnt::btm::IsSameBdAddress(&diList.device[i].bdAddress, &deviceInfoList.device[0].bdAddress))
            {
                isFound = true;
                break;
            }
            else
            {

            }
        }
        NN_SDK_REQUIRES(isFound == true);
    }

    //接続確認
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Push A button the joycon.\n");
        testUtil::WaitDcl(1, 0);
        nn::btm::HidDisconnect(&deviceInfoList.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);
    }

    //プロテクト情報のクリア
    {
        nn::btm::GeneralInfoList giList;
        nn::btm::debug::GeneralTest(11);
        nn::btm::debug::GeneralGet(0, &giList);
        NN_SDK_REQUIRES(giList.count == 0);
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

void System()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //ペアリング開始、システムイベントのシグナル、状態確認
    {
        bool isGpp;
        isGpp = nn::btm::system::IsGamepadPairingStarted();
        NN_SDK_REQUIRES(isGpp == false);

        nn::Result result;
        result = nn::btm::system::StartGamepadPairing();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        WaitSystemEvent(&g_SystemEventForGpp);

        isGpp = nn::btm::system::IsGamepadPairingStarted();
        NN_SDK_REQUIRES(isGpp == true);

        //DCLの変化も確認
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 0);
    }

    //ペアリング終了、システムイベントのシグナル、状態確認
    {
        nn::btm::system::CancelGamepadPairing();
        WaitSystemEvent(&g_SystemEventForGpp);

        bool isGpp = nn::btm::system::IsGamepadPairingStarted();
        NN_SDK_REQUIRES(isGpp == false);

        //DCLの変化も確認
        testUtil::WaitDcl(0, 0);
        NN_UNUSED(isGpp);
    }

    //ペアリング多重開始、終了
    {
        for(int i=0;i<2;i++)
        {
            nn::Result result = nn::btm::system::StartGamepadPairing();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            WaitSystemEvent(&g_SystemEventForGpp);
            bool isGpp = nn::btm::system::IsGamepadPairingStarted();
            NN_SDK_REQUIRES(isGpp == true);
            //testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 0, 0);//2回目はCdcがシグナルしないため、待たない
            NN_UNUSED(isGpp);
        }

        {
            nn::btm::system::CancelGamepadPairing();
            WaitSystemEvent(&g_SystemEventForGpp);
            bool isGpp = nn::btm::system::IsGamepadPairingStarted();
            NN_SDK_REQUIRES(isGpp == false);
            testUtil::WaitDcl(0, 0);
            NN_UNUSED(isGpp);
        }
    }

    NN_LOG("[TestBtm_Auto]!! Recommended Condition !!: There are no Switches that enabling pairing.\n");
    NN_LOG("[TestBtm_Auto]## Need User Operation ##: Need 2 devices. Please start BT pairing simultaneously.\n");

    //デバイスとペアリング。2クライアント同時
    {
        nn::Result result;
        result = nn::btm::system::StartGamepadPairing();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, 2, 0);
    }

    //ペアリング終了
    {
        nn::btm::system::CancelGamepadPairing();
        testUtil::WaitDcl(2, 0);
    }

    //RadioOff。状態確認
    {
        bool isRadio;
        isRadio = nn::btm::system::IsRadioEnabled();
        NN_SDK_REQUIRES(isRadio == true);

        nn::btm::system::EnableRadio(false);
        WaitSystemEvent(&g_SystemEventForRadio);

        isRadio = nn::btm::system::IsRadioEnabled();
        NN_SDK_REQUIRES(isRadio == false);

        testUtil::WaitDcl(0, 0);
        NN_UNUSED(isRadio);
    }

    //RadioOn。状態確認
    {
        nn::btm::system::EnableRadio(true);
        WaitSystemEvent(&g_SystemEventForRadio);
        bool isRadio = nn::btm::system::IsRadioEnabled();
        NN_SDK_REQUIRES(isRadio == true);

        NN_UNUSED(isRadio);
    }

    //多重RadioOff
    {
        {
            nn::btm::system::EnableRadio(false);
            WaitSystemEvent(&g_SystemEventForRadio);
            bool isRadio = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadio == false);
            NN_UNUSED(isRadio);
        }

        {
            nn::btm::system::EnableRadio(false);
            bool isSignaled = TryWaitSystemEvent(&g_SystemEventForRadio);
            NN_SDK_REQUIRES(isSignaled == false);
            bool isRadio = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadio == false);
            NN_UNUSED(isSignaled);
            NN_UNUSED(isRadio);
        }
    }

    //多重RadioOn
    {
        {
            nn::btm::system::EnableRadio(true);
            WaitSystemEvent(&g_SystemEventForRadio);
            bool isRadio = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadio == true);
            NN_UNUSED(isRadio);
        }

        {
            nn::btm::system::EnableRadio(true);
            bool isSignaled = TryWaitSystemEvent(&g_SystemEventForRadio);
            NN_SDK_REQUIRES(isSignaled == false);
            bool isRadio = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadio == true);
            NN_UNUSED(isRadio);
            NN_UNUSED(isSignaled);
        }
    }

    //再接続確認
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Push A button of 2 joycons.\n");
        testUtil::WaitDcl(2, 0);
    }

    //ペアリング情報削除
    {
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        int deviceCount;
        deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 2);

        nn::btm::system::ClearGamepadPairingDatabase();
        testUtil::WaitDcl(0, 0);

        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        deviceCount = nn::btm::system::GetPairedGamepadCount();
        NN_SDK_REQUIRES(deviceCount == 0);
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

}//namespace function


namespace usecase
{

//RadioOff中の全API発行。RadioOff中も動作すべきAPIについては、その動作確認。
//そうでないAPIについては、正しいエラーが返ること、fatalしないことを確認。
//非接続デバイスへのAPI発行も兼ねる
void RadioOff()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //RadioOff
    {
        nn::btm::EnableRadio(false);
        nn::btm::BtmState state;
        nn::btm::GetState(&state);
        NN_SDK_REQUIRES(state == nn::btm::BtmState_RadioOff);
    }

    //Btm_Api.h発行
    {
        //Interface
        {
            nn::btm::FinalizeBtmInterface();
            nn::btm::InitializeBtmInterface();
        }

        //btm
        {
            nn::btm::BtmState state;
            nn::btm::GetState(&state);
            NN_SDK_REQUIRES(state == nn::btm::BtmState_RadioOff);

            nn::btm::HostDeviceProperty hdp;
            nn::btm::GetHostDeviceProperty(&hdp);
            testUtil::CheckAndPrindHdp(hdp);
        }

        //Power
        {
            //LLR
            {
                nn::btm::BdAddress DummyBdAddress = {{0}};
                nn::os::ClearSystemEvent(&g_SystemEventForLlr);
                for(;;)
                {
                    auto result = nn::btm::LlrNotify(&DummyBdAddress);
                    if(nn::btm::ResultBusy::Includes(result))
                    {
                        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                    }
                    else if(nn::btm::ResultInvalidStateRadioOff::Includes(result))
                    {
                        //期待の結果
                        break;
                    }
                    else
                    {
                        NN_ABORT("[testBtm_Auto]Invalid Result.\n");
                        break;
                    }
                }
                if(nn::os::TryWaitSystemEvent(&g_SystemEventForLlr))
                    NN_ABORT("[testBtm_Auto]Invalid Signal.\n");
                if(nn::btm::IsLlrStarted())
                    NN_ABORT("[testBtm_Auto]Invalid LLR State.\n");
            }

            //SlotSaving有効
            {
                nn::btm::EnableSlotSaving(true);
                testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, true, 0, 0);
            }

            //SlotSaving無効
            {
                nn::btm::EnableSlotSaving(false);
                testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 0);
            }
        }

        //Cdc
        {
            //[Todo]Register系のテストを足す。リブート系テストと絡める
            testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 0);
            //[Todo]BurstMode
            nn::Result result;
            nn::btm::DeviceSlotModeList dsl;
            dsl.deviceCount = 0;
            for(;;)
            {
                result = nn::btm::SetSlotMode(&dsl);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
                else if(nn::btm::ResultInvalidStateRadioOff::Includes(result))
                {
                    //IncvalidStateRadioOffが期待される
                    break;
                }
                else
                {
                    NN_ABORT("[TestBtm_Auto]Bad result.\n");
                    break;
                }
            }

            //接続デバイスが無いため、Nfpとhidを用いた方法は使えない
            //testUtil::ToggleNfp();
            //testUtil::WaitDcl(1);
            //testUtil::ToggleNfp();
            //testUtil::WaitDcl(1);

            //代わりに直接BluetoothModeを制御する
            for(;;)
            {
                result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Active);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }
            }
            //hidがAutoに書き戻すので、それを待つ
            testUtil::WaitDcl(0, 0);

            for(;;)
            {
                result = nn::btm::SetWlanMode(nn::btm::WlanMode_Local4);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }
            }
            testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 0);
        }

        //Rdi
        {
            nn::btm::DeviceInfoList dil;
            //[Todo]Register系のテストを足す。リブート系テストと絡める
            NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please joint a Joy-Con.\n");
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
            nn::btm::GetDeviceInfo(&dil);
            NN_SDK_REQUIRES(dil.deviceCount == 1);//SDK以外のマクロ無いか

            //接続が確立しないことを確かめるために10秒待つ。[Todo]ユーザ操作依存をなくす
            NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please unjoint a Joy-Con within 10 seconds.\n");
            for(int i=0;i<10;i++)
            {
                NN_LOG("%d sec\n", i);
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            testUtil::ValidateDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 0);

            //境界登録。カウントのみ評価
            {
                nn::btm::DeviceInfoList dil;
                nn::btm::GetDeviceInfo(&dil);
                NN_SDK_REQUIRES(dil.deviceCount == 1);

                nn::os::ClearSystemEvent(&g_SystemEventForRdi);
                nn::btm::RemoveDeviceInfo(&dil.device[0].bdAddress);
                nn::os::WaitSystemEvent(&g_SystemEventForRdi);

                nn::btm::DeviceInfo inputDeviceInfo;
                for(int i=0;i<nn::btm::COUNT_OF_DI_LIST;i++)
                {
                    inputDeviceInfo = dil.device[0];
                    inputDeviceInfo.bdAddress.address[5] = i;
                    nn::btm::AddDeviceInfo(&inputDeviceInfo);
                    nn::os::WaitSystemEvent(&g_SystemEventForRdi);
                }
                nn::btm::DeviceInfoList outputDeviceInfoList;
                nn::btm::GetDeviceInfo(&outputDeviceInfoList);
                NN_SDK_REQUIRES(outputDeviceInfoList.deviceCount == nn::btm::COUNT_OF_DI_LIST);//SDK以外のマクロ無いか
            }

            //ペアリング情報の生存優先度変更
            {
                nn::btm::DeviceInfoList dil;
                nn::btm::GetDeviceInfo(&dil);
                nn::btm::IncreaseDeviceInfoOrder(&dil.device[0].bdAddress);//もっとも古いペアリング情報を指定
                nn::os::WaitSystemEvent(&g_SystemEventForRdi);

                nn::btm::DeviceInfoList currentDil;
                nn::btm::GetDeviceInfo(&currentDil);
                NN_SDK_REQUIRES(currentDil.deviceCount == nn::btm::COUNT_OF_DI_LIST);
                //当該情報が最後尾に回っていることの確認
                bool isValid;
                isValid = nnt::btm::IsSameBdAddress(&dil.device[0].bdAddress, &currentDil.device[nn::btm::COUNT_OF_DI_LIST - 1].bdAddress);
                NN_SDK_REQUIRES(isValid);
                //当該情報以外が前詰めされていることの確認
                for(int i=0;i<nn::btm::COUNT_OF_DI_LIST - 1;i++)
                {
                    isValid = nnt::btm::IsSameBdAddress(&dil.device[i + 1].bdAddress, &currentDil.device[i].bdAddress);
                    NN_SDK_REQUIRES(isValid);
                }
            }

            //全件削除
            {
                nn::btm::DeviceInfoList dil;
                nn::btm::GetDeviceInfo(&dil);
                NN_SDK_REQUIRES(dil.deviceCount == nn::btm::COUNT_OF_DI_LIST);
                nn::os::ClearSystemEvent(&g_SystemEventForRdi);

                for(int i=0;i<dil.deviceCount;i++)
                {
                    nn::btm::RemoveDeviceInfo(&dil.device[i].bdAddress);
                    nn::os::WaitSystemEvent(&g_SystemEventForRdi);
                }
                nn::btm::GetDeviceInfo(&dil);
                NN_SDK_REQUIRES(dil.deviceCount == 0);
            }
        }

        //DeviceTreatment
        {
            nn::btm::BdAddress bdAddress;
            bdAddress.address[0] = 0xAA;
            bdAddress.address[1] = 0xBB;
            bdAddress.address[2] = 0xCC;
            bdAddress.address[3] = 0xDD;
            bdAddress.address[4] = 0xEE;
            bdAddress.address[5] = 0xFF;
            nn::btm::ProtectDeviceInfo(&bdAddress, true);
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
            nn::btm::GeneralInfoList giList;
            nn::btm::debug::GeneralGet(0, &giList);
            NN_SDK_REQUIRES(giList.count == 1);
            nn::btm::ProtectDeviceInfo(&bdAddress, false);
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//同期手段が無いので、少し待つ
            nn::btm::debug::GeneralGet(0, &giList);
            NN_SDK_REQUIRES(giList.count == 0);
        }

        //Hid
        {
            nn::btm::ZeroRetransmissionList zrList;
            zrList.enabledReportId[0]   = 0x01;
            zrList.enabledReportIdCount = 1;
            nn::btm::BdAddress DummyBdAddress = {{0}};
            for(;;)
            {
                auto result = nn::btm::HidSetRetransmissionMode(&DummyBdAddress, &zrList);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
                else if(nn::btm::ResultInvalidStateRadioOff::Includes(result))
                {
                    //RadioOffのresultを期待
                    break;
                }
                else
                {
                    NN_ABORT("[testBtm_Auto]Invalid Result\n");
                    break;
                }
            }
        }

        //ペアリング情報追加
        {
            NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please joint a Joy-Con.\n");
            nn::btm::DeviceInfoList dil;
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
            nn::btm::GetDeviceInfo(&dil);
            NN_SDK_REQUIRES(dil.deviceCount == 1);//SDK以外のマクロ無いか
        }

        //System（テストステップ削減のため、ジョイント状態で実行）
        {
            nn::os::ClearSystemEvent(&g_SystemEventForGpp);
            auto result = nn::btm::system::StartGamepadPairing();
            if(!nn::btm::system::ResultPairingFailureRadioOff::Includes(result))
                NN_ABORT("[TestBtm_Auto]Bad result.\n");
            //停滞ステートにおけるエラー時はシグナルされない
            /*nn::os::WaitSystemEvent(&g_SystemEventForGpp);
            bool isGpp;
            isGpp = nn::btm::system::IsGamepadPairingStarted();
            if(isGpp)
                NN_ABORT("[TestBtm_Auto]Bad state.\n");*/
            //キャンセルはステートチェックしないため、シグナルする
            nn::btm::system::CancelGamepadPairing();
            nn::os::WaitSystemEvent(&g_SystemEventForGpp);
            if(nn::btm::system::IsGamepadPairingStarted())
                NN_ABORT("[TestBtm_Auto]Bad state.\n");
            nn::btm::system::ClearGamepadPairingDatabase();
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
            int deviceCount;
            deviceCount = nn::btm::system::GetPairedGamepadCount();
            NN_SDK_REQUIRES(deviceCount == 0);
            bool isRadioOn;
            isRadioOn = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadioOn == false);
            nn::btm::system::EnableRadio(true);
            nn::btm::system::EnableRadio(false);
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            isRadioOn = nn::btm::system::IsRadioEnabled();
            NN_SDK_REQUIRES(isRadioOn == false);
            NN_UNUSED(isRadioOn);
        }

        //ペアリング情報追加
        {
            //Hidの仕様変更により、再ジョイントせずともペアリング情報が登録されるようになった
            //NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please unjoint and joint again a Joy-Con.\n");
            nn::btm::DeviceInfoList dil;
            nn::os::WaitSystemEvent(&g_SystemEventForRdi);
            nn::btm::GetDeviceInfo(&dil);
            NN_SDK_REQUIRES(dil.deviceCount == 1);//SDK以外のマクロ無いか
        }

        //RadioOn（テストステップ削減のため、ジョイント状態で実行）
        {
            nn::btm::EnableRadio(true);
            nn::btm::BtmState state;
            nn::btm::GetState(&state);
            NN_SDK_REQUIRES(state == nn::btm::BtmState_Initialized);
        }

        //接続確認（RadioOff状態でのペアリング情報、BluetoothMode、WlanModeが残ることを確認）
        {
            NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please unjoint a Joy-Con.\n");
            testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 1);
        }
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}//NOLINT(impl/function_size)

void Recovery()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    nn::btm::debug::GeneralTest(9);
    NN_LOG("[TestBtm_Auto]Recovery queued.\n");
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void Recalculation()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    nn::btm::debug::GeneralTest(10);
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

}//namespace usecase


namespace aging
{

void Nfp(int connectionCount, int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    const int CONNECTION_COUNT = connectionCount;
    const int TEST_COUNT = testCount;

    //CONNECTION_COUNT台接続待ち
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please connect %d controllers by joint pairing.\n", CONNECTION_COUNT);
        testUtil::WaitDcl(CONNECTION_COUNT, 0);
    }

    //Nfpのトグルエージング
    {
        for(int i=0;i<TEST_COUNT;i++)
        {
            NN_LOG("[TestBtm_Auto]count %d/%d\n",i, TEST_COUNT);
            testUtil::ToggleNfp(true);
            testUtil::WaitDcl(CONNECTION_COUNT, 1);//[Todo]IsLargeSlotRequiredの確認
            testUtil::ToggleNfp(false);
            testUtil::WaitDcl(CONNECTION_COUNT, 0);//[Todo]IsLargeSlotRequiredの確認
        }
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}


void Tsi(int connectionCount, int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    const int CONNECTION_COUNT = connectionCount;
    nn::os::Tick sTick;//[Todo]tick系はUtilityにしておく
    nn::os::Tick fTick;
    const int INTERVAL_TIMEOUT = CONNECTION_COUNT * 12000;
    uint16_t interval;
    const int TEST_COUNT = testCount;

    //CONNECTION_COUNT 台接続待ち
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please connect %d controllers by joint pairing.\n", CONNECTION_COUNT);
        testUtil::WaitDcl(CONNECTION_COUNT, 0);
    }

    //GamePadPairingによるTSI変更をTEST_COUNT回試行
    {
        for(int i=0;i<TEST_COUNT;i++)
        {
            NN_LOG("[TestBtm_Auto]count %d/%d\n",i + 1,TEST_COUNT);

            sTick = nn::os::GetSystemTick();
            nn::Result result;
            result = nn::btm::system::StartGamepadPairing();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            testUtil::WaitDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, true, false, CONNECTION_COUNT, 0);//[Todo]GamepadPairingEventで待つようにする
            fTick = nn::os::GetSystemTick();
            interval = nn::os::ConvertToTimeSpan(fTick).GetMilliSeconds() - nn::os::ConvertToTimeSpan(sTick).GetMilliSeconds();
            if(interval > INTERVAL_TIMEOUT)
            {
                NN_ABORT("[testBtm_Auto]interval timeout. %d", interval);
            }

            sTick = nn::os::GetSystemTick();
            nn::btm::system::CancelGamepadPairing();
            testUtil::WaitDcl(CONNECTION_COUNT, 0);
            fTick = nn::os::GetSystemTick();
            interval = nn::os::ConvertToTimeSpan(fTick).GetMilliSeconds() - nn::os::ConvertToTimeSpan(sTick).GetMilliSeconds();
            if(interval > INTERVAL_TIMEOUT)
            {
                NN_ABORT("[testBtm_Auto]interval timeout. %d", interval);
            }
        }
    }

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void EmuSleep(int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    const int TEST_COUNT = testCount;

    for(int i=0;i<TEST_COUNT;i++)
    {
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        NN_LOG("[TestBtm_Auto]Test Count %d/%d -----------.\n", i, TEST_COUNT);

        nn::btm::debug::GeneralTest(7);
        NN_LOG("[TestBtm_Auto]Sleep.\n");
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        nn::btm::debug::GeneralTest(8);
        NN_LOG("[TestBtm_Auto]Awake.\n");
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void LlrStrictOperation(int connectionCount, int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    const int DEVICE_COUNT = connectionCount;
    const int TEST_COUNT = testCount;

    //DEVICE_COUNT台接続待ち
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please connect %d controllers via BT by using joint pairing.\n", DEVICE_COUNT);
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: After above, TBFC will be periodically triggered. Please push reset button of controller periodically.\n", DEVICE_COUNT);
        testUtil::WaitDcl(DEVICE_COUNT, 0);
    }

    //試験対象のジョイコン情報を確定
    nn::btm::DeviceInfoList dil;
    {
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&dil);
        NN_SDK_REQUIRES(dil.deviceCount == DEVICE_COUNT);
    }

    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
    }

    for(int testC = 0;testC < TEST_COUNT;testC++)
    {
        //DEVICE_COUNT台を切断
        for(int i=0;i<DEVICE_COUNT;i++)
        {
            nn::btm::HidDisconnect(&dil.device[i].bdAddress);
        }
        testUtil::WaitDcl(0, 0);

        NN_LOG("[TestBtm_Auto]Test Count %d/%d----------------\n", testC, TEST_COUNT);

        //1台づつの LLR Notifyと接続確認をDEVICE_COUNT台行う
        for(int i=0;i<DEVICE_COUNT;i++)
        {
            for(;;)
            {
                NN_LOG("[TestBtm_Auto]TBFC to ");nnt::btm::PrintBdAddress(dil.device[i].bdAddress);
                bool isApiSuccess = false;
                auto result = nn::btm::LlrNotify(&dil.device[i].bdAddress);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns busy.\n");
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
                else if(result.IsSuccess())
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns success.\n");
                    isApiSuccess = true;
                }
                else
                {
                    //イベントを見ている場合ここには来ない。（LLR 実行中の別のLLR発行はしない）
                    NN_LOG("[TestBtm_Auto]TBFC API returns error.\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }

                if(isApiSuccess)
                {
                    //LLRの終了待ち
                    for(;;)
                    {
                        nn::os::WaitSystemEvent(&g_SystemEventForLlr);
                        if(nn::btm::IsLlrStarted())
                        {
                            NN_LOG("[TestBtm_Auto]TBFC working.\n");
                        }
                        else
                        {
                            NN_LOG("[TestBtm_Auto]TBFC finished.\n");
                            break;
                        }
                    }

                    //当該デバイスの接続確認
                    //CheckDclだと、TSI変更未完了でエラーが返るため、直接接続デバイスのアドレスを確認する
                    bool isConnectionSuccess = false;
                    nn::btm::DeviceConditionList dcList;
                    nn::btm::GetConnectedDeviceCondition(&dcList);
                    for(int j=0;j<dcList.deviceCount;j++)
                    {
                        if(nnt::btm::IsSameBdAddress(&dcList.device[j].bdAddress, &dil.device[i].bdAddress))
                        {
                            isConnectionSuccess = true;
                        }
                    }

                    //接続できたら、次のデバイスへのLLRに移行
                    if(isConnectionSuccess)
                    {
                        NN_LOG("[TestBtm_Auto]connection success.\n");
                        break;
                    }
                    else
                    {
                        //LLRが、外乱やReset押下の影響で失敗したパターン
                        NN_LOG("[TestBtm_Auto]connection failure.\n");
                    }
                }
            }
        }
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void Llr(int connectionCount, int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    const int DEVICE_COUNT = connectionCount;
    const int TEST_COUNT = testCount;

    //DEVICE_COUNT台接続待ち
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please connect %d controllers via BT by using joint pairing.\n", DEVICE_COUNT);
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: After above, TBFC will be periodically triggered. Please push reset button of controller periodically.\n", DEVICE_COUNT);
        testUtil::WaitDcl(DEVICE_COUNT, 0);
    }

    //試験対象のジョイコン情報を確定
    nn::btm::DeviceInfoList dil;
    {
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&dil);
        NN_SDK_REQUIRES(dil.deviceCount == DEVICE_COUNT);
    }

    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlr);
    }

    for(int testC = 0;testC < TEST_COUNT;testC++)
    {
        //DEVICE_COUNT台を切断
        for(int i=0;i<DEVICE_COUNT;i++)
        {
            nn::btm::HidDisconnect(&dil.device[i].bdAddress);
        }
        testUtil::WaitDcl(0, 0);

        NN_LOG("[TestBtm_Auto]Test Count %d/%d----------------\n", testC, TEST_COUNT);
        //DEVICE_COUNT台がつながるまでLLR Notifyを継続
        for(;;)
        {
            //DEVICE_COUNT台に対して、LLR Notifyを発行。busy, error時の再発行はしない
            for(int i=0;i<DEVICE_COUNT;i++)
            {
                NN_LOG("[TestBtm_Auto]TBFC to ");nnt::btm::PrintBdAddress(dil.device[i].bdAddress);
                auto result = nn::btm::LlrNotify(&dil.device[i].bdAddress);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns busy.\n");
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
                else if(result.IsSuccess())
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns success.\n");
                }
                else
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns error.\n");
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
            }

            //DEVICE_COUNT台つながったらbreak
            testUtil::CheckDclResult result = testUtil::CheckDcl(nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, DEVICE_COUNT, 0);
            if(result == testUtil::CheckDclResult_Success)
            {
                break;
            }
        }
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void LlrSleep(int connectionCount, int testCount)
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    const int DEVICE_COUNT = connectionCount;
    const int TEST_COUNT = testCount;

    //DEVICE_COUNT台接続待ち
    {
        NN_LOG("[TestBtm_Auto]## Need User Operation ##: Please connect %d controllers via BT by using joint pairing.\n", DEVICE_COUNT);
        testUtil::WaitDcl(DEVICE_COUNT, 0);
    }

    //試験対象のジョイコン情報を確定
    nn::btm::DeviceInfoList dil;
    {
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&dil);
        NN_SDK_REQUIRES(dil.deviceCount == DEVICE_COUNT);
    }

    //DEVICE_COUNT台を切断
    for(int i=0;i<DEVICE_COUNT;i++)
    {
        nn::btm::HidDisconnect(&dil.device[i].bdAddress);
    }
    testUtil::WaitDcl(0, 0);//台数とタイミングによっては、ここでブロックしうるため、waitを抜いておく

    for(int testC = 0;testC<TEST_COUNT;testC++)
    {
        NN_LOG("[TestBtm_Auto]Test Count %d/%d ----------------\n", testC, TEST_COUNT);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));//btmのキューがたまらないように少し待つ
        //LLR Notify。少なくとも1つ以上のTBFCが動作するまで試行
        for(;;)
        {
            bool isSuccess = false;
            for(int i=0;i<DEVICE_COUNT;i++)
            {
                NN_LOG("[TestBtm_Auto]TBFC to ");nnt::btm::PrintBdAddress(dil.device[i].bdAddress);
                auto result = nn::btm::LlrNotify(&dil.device[i].bdAddress);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns busy.\n");
                }
                else if(result.IsSuccess())
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns success.\n");
                    isSuccess = true;
                }
                else
                {
                    NN_LOG("[TestBtm_Auto]TBFC API returns error.\n");
                }
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            }
            if(isSuccess)
            {
                break;
            }
        }

        //Sleep
        nn::btm::debug::GeneralTest(7);
        NN_LOG("[TestBtm_Auto]Sleep success.\n");

        //Awake
        nn::btm::debug::GeneralTest(8);
        NN_LOG("[TestBtm_Auto]Awake success.\n");
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void IterationGpp()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    for(int i=0;i<50;i++)
    {
        nn::btm::system::StartGamepadPairing();
        nn::btm::system::CancelGamepadPairing();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void IterationRadio()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);
    for(int i=0;i<50;i++)
    {
        nn::btm::system::EnableRadio(true);
        nn::btm::system::EnableRadio(false);
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

}//namespace aging

namespace abnormal{

void SystemEvent()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //テスト実行状態で余っているイベント
    //const int REMAIN_DISC_EVENT_COUNT   = 3;
    const int REMAIN_CDC_EVENT_COUNT    = 1;//ldnが取得していないため。
    const int REMAIN_RDI_EVENT_COUNT    = 1;
    //const int REMAIN_AWAKER_EVENT_COUNT = 2;
    const int REMAIN_RADIO_EVENT_COUNT  = 2;//Oceanが取得していないため。
    const int REMAIN_GPP_EVENT_COUNT    = 2;//Oceanが取得していないため。

    //nn::os::SystemEventType eventDisc[REMAIN_DISC_EVENT_COUNT];
    nn::os::SystemEventType eventCdc[REMAIN_CDC_EVENT_COUNT];
    nn::os::SystemEventType eventRdi[REMAIN_RDI_EVENT_COUNT];
    //nn::os::SystemEventType eventAwaker[REMAIN_AWAKER_EVENT_COUNT];
    nn::os::SystemEventType eventRadio[REMAIN_RADIO_EVENT_COUNT];
    nn::os::SystemEventType eventGpp[REMAIN_GPP_EVENT_COUNT];

    //for(int i=0;i<REMAIN_DISC_EVENT_COUNT;i++)
    //    nn::btm::RegisterSystemEventForDiscovery(&eventDisc[i]);
    for(int i=0;i<REMAIN_CDC_EVENT_COUNT;i++)
        nn::btm::RegisterSystemEventForConnectedDeviceCondition(&eventCdc[i]);
    for(int i=0;i<REMAIN_RDI_EVENT_COUNT;i++)
        nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&eventRdi[i]);
    //for(int i=0;i<REMAIN_AWAKER_EVENT_COUNT;i++)
    //    nn::btm::RegisterSystemEventForAwakeReq(&eventAwaker[i]);
    for(int i=0;i<REMAIN_RADIO_EVENT_COUNT;i++)
        nn::btm::system::AcquireRadioEvent(&eventRadio[i]);
    for(int i=0;i<REMAIN_GPP_EVENT_COUNT;i++)
        nn::btm::system::AcquireGamepadPairingEvent(&eventGpp[i]);

    //全てのイベントがシグナルすることの確認
    {
        nn::btm::system::StartGamepadPairing();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//暫定対処：キューでの削除機能が働かないように、少し待つ
        nn::btm::system::CancelGamepadPairing();
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::btm::system::EnableRadio(false);
        nn::btm::system::EnableRadio(true);

        for(int i=0;i<REMAIN_CDC_EVENT_COUNT;i++)
            nn::os::WaitSystemEvent(&eventCdc[i]);
        for(int i=0;i<REMAIN_RDI_EVENT_COUNT;i++)
            nn::os::WaitSystemEvent(&eventRdi[i]);
        for(int i=0;i<REMAIN_RADIO_EVENT_COUNT;i++)
            nn::os::WaitSystemEvent(&eventRadio[i]);
        for(int i=0;i<REMAIN_GPP_EVENT_COUNT;i++)
            nn::os::WaitSystemEvent(&eventGpp[i]);
    }
    NN_LOG("[TestBtm_Auto]Signal came.\n");

    //RadioEventとGamepadPairingEventは暫定的に上書き機能を有効にしているため、上限越えの登録と、シグナルの確認
    {
        const int ADD_COUNT = 3;//サーバ側の上限3に合わせる
        nn::os::SystemEventType additionalEventForRadio[ADD_COUNT];
        nn::os::SystemEventType additionalEventForGpp[ADD_COUNT];
        for(int i=0;i<ADD_COUNT;i++)
        {
            nn::btm::system::AcquireRadioEvent(&additionalEventForRadio[i]);
            nn::btm::system::AcquireGamepadPairingEvent(&additionalEventForGpp[i]);
        }

        nn::btm::system::StartGamepadPairing();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));//暫定対処：キューでの削除機能が働かないように、少し待つ
        nn::btm::system::CancelGamepadPairing();
        nn::btm::system::EnableRadio(false);
        nn::btm::system::EnableRadio(true);

        for(int i=0;i<ADD_COUNT;i++)
        {
            nn::os::WaitSystemEvent(&additionalEventForRadio[i]);
            nn::os::WaitSystemEvent(&additionalEventForGpp[i]);
        }
    }
    NN_LOG("[TestBtm_Auto]Additional Signal came.\n");

    //上限+1でbtmライブラリがABORTすることの確認
    /*{
        nn::os::SystemEventType additionalEvent;
        nn::btm::RegisterSystemEventForConnectedDeviceCondition(&additionalEvent);
        //nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&additionalEvent);
        //nn::btm::system::AcquireRadioEvent(&additionalEvent);
        //nn::btm::system::AcquireGamepadPairingEvent(&additionalEvent);
    }*/
    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

void Llr()
{
    NN_LOG("[TestBtm_Auto]%s\n",__FUNCTION__);

    //不在デバイスへのLLR中の別デバイスへのLLR
    {
        testUtil::WaitConnection("Joy-Con");
        nn::btm::DeviceConditionList dcl;
        nn::btm::GetConnectedDeviceCondition(&dcl);
        nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);

        nn::btm::DeviceInfoList deviceInfoList;
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_SDK_REQUIRES(deviceInfoList.deviceCount == 1);

        deviceInfoList.device[0].bdAddress.address[0] = 0xAA;

        nn::os::ClearSystemEvent(&g_SystemEventForRdi);
        nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
        nn::os::WaitSystemEvent(&g_SystemEventForRdi);
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_SDK_REQUIRES(deviceInfoList.deviceCount == 2);

        //ダミーのDeviceへのLLR
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&deviceInfoList.device[1].bdAddress);//ダミーのペアリング情報は2番目に入っている
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }

        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10000));

        //正しいDeviceへのLLR
        for(;;)
        {
            auto result = nn::btm::LlrNotify(&deviceInfoList.device[0].bdAddress);
            if(nn::btm::ResultBusy::Includes(result))
            {
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;
            }
        }
        testUtil::WaitDcl(1, 0);
    }

    //Llr中のデバイスリセット試行用
    /*{
        testUtil::WaitConnection("Joy-Con");
        nn::btm::DeviceConditionList dcl;
        nn::btm::GetConnectedDeviceCondition(&dcl);
        nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
        testUtil::WaitDcl(0, 0);

        for(int i=0;;i++)
        {
            for(;;)
            {
                auto result = nn::btm::LlrNotify(&dcl.device[0].bdAddress);
                if(nn::btm::ResultBusy::Includes(result))
                {
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }
            }
            testUtil::WaitDcl(1, 0);
            nn::btm::GetConnectedDeviceCondition(&dcl);
            nn::btm::HidDisconnect(&dcl.device[0].bdAddress);
            testUtil::WaitDcl(0, 0);
            NN_LOG("[testBtm_Auto]LLR Notify will be triggered after 1 sec.\n");
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }
    }*/

    NN_LOG("[TestBtm_Auto]%s\n\n",__FUNCTION__);
}

}//namespace abnormal

}//namespace testSuite


//[Todo]タイムアウトを入れておく
extern "C" void nnMain()
{
    testSys::Initialize();
    testSys::Interval();

#if 1
    //単機能に着目したテスト。シンプルな条件下で単機能が正常動作することをテストする。
    //テストカテゴリは、Api.h準拠
    //テスト同士はIntervalで初期化される内容については疎。リブートはしない。
    //hidはひとまず見ない。正常動作の判定方法が未定義のため。
    //ユーザの異常操作のケアはしない。不必要な例外処理はテストの目的ではないため。

    testSuite::function::Interface();
    testSys::Interval();

    testSuite::function::Btm();
    testSys::Interval();

    testSuite::function::PowerMode();
    testSys::Interval();

    testSuite::function::DeviceInfo();
    testSys::Interval();

    testSuite::function::Hid();
    testSys::Interval();

    testSuite::function::ConnectedDeviceCondition();
    testSys::Interval();

    testSuite::function::DeviceTreatment();
    testSys::Interval();

    testSuite::function::System();
    testSys::Interval();
#endif

#if 1
    //ユースケースに着目したテスト。使用頻度の高い、代表的なユースケースをテストする。
    //ユーザの技量が必要となるテストはしない。例：ペアリング中にコントローラリセット

    testSuite::usecase::Recalculation();
    testSys::Interval();

    testSuite::usecase::RadioOff();
    testSys::Interval();

    //[Todo]安定動作後に有効化
    //testSuite::usecase::Recovery();
    //testSys::Interval();

    // RadioOffでのInitialize.特にシステムイベントの追加等、RadioOffでも呼べるべき機能の確認
    // ペアリングの終了条件。WlanMode_LocalX, Sleep, RadioOff
    // リブート挟んでもNandの内容（ペアリング、RadioMode）が正常なことを確認
    // Llrで起きないこと。適切なステートでは起きること
    // 本体パワーモード絡めた試験。スリープでのコントローラ切断。複数台切断。AwakeReqイベント
    // ペアリング上限状態でのジョイントおよび、BTペアリング
    // 左右同時ジョイントペアリング
    // ドックイン、ドックアウトを絡めた試験。Radioモードの変化。スリープ中ドックイン。
    // 接続台数増やし。上限越え

#endif

    //機能同士の組み合わせを機械的に生成したテスト。

#if 1
    //連続動作と処理タイミングに着目したテスト。
    testSuite::aging::IterationGpp();
    testSys::Interval();

    testSuite::aging::IterationRadio();
    testSys::Interval();

    testSuite::aging::Llr(2, 10);
    testSys::Interval();

    testSuite::aging::LlrStrictOperation(2, 10);
    testSys::Interval();

    testSuite::aging::Tsi(2, 30);
    testSys::Interval();

    testSuite::aging::EmuSleep(10);
    testSys::Interval();

    testSuite::aging::LlrSleep(2, 30);
    testSys::Interval();

    testSuite::aging::Nfp(8, 20);
    testSys::Interval();
#endif

#if 0
    //異常系のテスト
    //testSuite::abnormal::Llr();
    //testSys::Interval();

    testSuite::abnormal::SystemEvent();
    testSys::Interval();
#endif

#if 0
    //問題再現用の単発テスト
    //testSuite::aging::Llr(1, 10000);
    //testSys::Interval();

    //testSuite::aging::LlrStrictOperation(2, 10);
    //testSys::Interval();

    testSuite::aging::LlrSleep(2, 1000);
    testSys::Interval();

    //testSuite::aging::Tsi(2, 2000);
    //testSys::Interval();

#endif

    testSys::Finalize();
}
