﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/hid.h>
#include <nn/bluetooth.h>

#include "BleTestTool_Scene.h"

class ScanScene : public SceneBase
{
public:
    /**
     * @brief       コンストラクタです
     */
    NN_IMPLICIT ScanScene(std::string title);

    /**
     * @brief       デストラクタです
     */
    virtual ~ScanScene() {};

    /**
     * @brief       BLE スキャンの結果を描画します
     */
    virtual void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter);

    /**
     * @brief       BLE スキャン結果を更新します
     */
    void UpdateScanResult();

    /**
     * @brief       コントローラの入力を処理します
     */
    virtual void ProcessControllerInput(nns::hid::ButtonSet buttons);

    //!< BLE Scan の対象とする BLE Device/Smart Device を表す列挙型識別子です
    enum ScanTarget
    {
        ScanTarget_None = 0,
        ScanTarget_UnpairedPalma,
        ScanTarget_PairedPalma,
        ScanTarget_DummyDevices,
        ScanTarget_DummyDevice_1111,
        ScanTarget_DummyDevice_2222,
        ScanTarget_DummyDevice_3333,
        ScanTarget_DummyDevice_4444,
        ScanTarget_DummyDevice_5555,
        ScanTarget_DefaultSmartDevice,
        ScanTarget_DummySmartDevices,
        ScanTarget_DummySmartDevice_88,
        ScanTarget_DummySmartDevice_77,
        ScanTarget_DummySmartDevice_66,
        ScanTarget_DummySmartDevice_55,
        ScanTarget_DummySmartDevice_44,
    };

private:
    /**
    * @brief       BLE デバイスを対象とするBLE スキャンを開始します
    */
    bool StartBleScanGeneral();
    /**
    * @brief       BLE デバイスを対象とするBLE スキャンを停止します
    */
    bool StopBleScanGeneral();

    /**
    * @brief       Palma の接続待受を開始します
    */
    bool StartScanPalma(bool includingUnpaired);
    /**
    * @brief       Palma の接続待受を停止します
    */
    bool StopScanPalma(bool includingPaired);

    /**
     * @brief       スマートデバイスアプリケーションを対象とするBLE スキャンを開始します
     */
    bool StartBleScanSmartDevice();
    /**
     * @brief       スマートデバイスアプリケーションを対象とするBLE スキャンを停止します
     */
    bool StopBleScanSmartDevice();

    /**
     * @brief       BLE スキャンの結果をクリアします
     */
    void ClearScanResult();

    /**
     * @brief       AD Type を文字列に変換します
     */
    const char* ConvertAdTypeToString(nn::bluetooth::BleAdType type) const;

    //!< BLE デバイスのBLE スキャン結果です
    nn::bluetooth::BleScanResult m_ScanResults[nn::bluetooth::BleScanResultCountMax];

    //!< 指定した BLE Device/Smart Device が Scan の対象となっているかを調べます
    bool isScanning(ScanTarget target) const;

    //!< BLE Device/Smart Device を Scan 中であるかを調べます
    int isScanning() const;

    //!< 現在 Scan 中の BLE Device/Smart Device のリストに指定した対象を追加します
    void AddScanningTarget(ScanTarget target);

    //!< 現在 Scan 中の BLE Device/Smart Device のリストをクリアします
    void ClearScanningTarget();

    //!< 現在 Scan 中の BLE Device/Smart Device のリストから指定した対象を削除します
    void RemoveScanningTaraget(ScanTarget target);

    //!< 同時に Scan 可能な BLE Device の最大数です
    static const int ScannableBleDeviceCountMax = 5;

    //!< 同時に Scan 可能な Smart Device の最大数です
    static const int ScannableSmartDeviceCountMax = 5;

    //!< 現在 Scan 中の BLE Device/Smart Device のリストです
    ScanTarget m_ScanningDevices[ScannableBleDeviceCountMax + ScannableSmartDeviceCountMax];

    //!< 現在選択中の Scan 対象です
    ScanTarget m_SelectingScanTarget;
};  // class ScanScene
