﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/hid.h>
#include <nn/bluetooth.h>

#include "BleTestTool_Scene.h"
#include "BleTestTool_ConnectionInfoDumper.h"

class GattClientScene : public SceneBase
{
public:
    /**
     * @brief       コンストラクタです
     */
    GattClientScene(std::string title, uint8_t connIndex);

    /**
     * @brief       デストラクタです
     */
    virtual ~GattClientScene();

    /**
     * @brief       GATT Server との接続状態を描画します
     */
    virtual void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter);

    /**
     * @brief       コントローラの入力を処理します
     */
    virtual void ProcessControllerInput(nns::hid::ButtonSet buttons);

    /**
     * @brief       GATT Server との接続状態を更新します
     */
    void UpdateConnectionState();

    /**
     * @brief       GATT Server のGATT Profile を更新します
     */
    void UpdateServerProfile();

    /**
     * @brief       GATT Attribute に対する操作結果を更新します
     */
    void UpdateGattOperationStatus(const nn::bluetooth::BleClientGattOperationInfo& info);

    /**
     * @brief       MTU を更新します
     */
    void UpdateMtu();

    /**
     * @brief       GATT Server との接続ハンドルを取得します
     */
    uint16_t GetConnectionHandle() { return m_ConnInfo.connectionHandle; }

    /**
     * @brief       GATT Server のBluetooth アドレスを取得します
     */
    nn::bluetooth::Address GetBluetoothAddress() { return m_ConnInfo.address; }

    /**
     * @brief       GATT Server との接続情報を取得します
     */
    nn::bluetooth::BleConnectionInfo GetConnectionInfo() { return m_ConnInfo; }

private:
    //!< GATT Server との接続状態です
    nn::bluetooth::BleConnectionInfo m_ConnInfo;

    //!< GATT Server が提供するGATT Service とその数です
    nn::bluetooth::GattService          m_Services[nn::bluetooth::GattAttributeCountMaxClient];
    uint16_t                            m_ServiceNum;

    //!< GATT Server が提供するGATT Characteristic とその数です
    nn::bluetooth::GattCharacteristic   m_Characteristics[nn::bluetooth::GattAttributeCountMaxClient];
    uint16_t                            m_CharacteristicNum;

    //!< GATT Server が提供するGATT Descriptor とその数です
    nn::bluetooth::GattDescriptor       m_Descriptors[nn::bluetooth::GattAttributeCountMaxClient];
    uint16_t                            m_DescriptorNum;

    //!< GATT Server が提供するGATT Characteristic とGATT Descriptor です。ハンドル順に格納されています。
    nn::bluetooth::GattAttribute*   m_CharsAndDescs[nn::bluetooth::GattAttributeCountMaxClient];
    uint16_t                        m_CharsAndDescsNum;

    //!< GATT Attribue に対する操作を同期するためのイベントです
    nn::os::EventType                   m_GattOperationInternalEvent;
    //!< 現在待機中のGATT Attribute に対する操作結果の操作の種類です
    nn::bluetooth::GattOperationType    m_ResultWaitingGattOperation;
    //!< 現在GATT Attribute に対する操作の同期待ちであるかを表すフラグです
    bool m_IsGattOperationOnGoing;

    /**
     * @brief       GATT Attribute に対する操作結果を表す構造体です
     */
    struct GattOperationResult
    {
        nn::bluetooth::BleGattOperationStatus   status;                                             //!< 操作結果の状態です
        nn::bluetooth::GattOperationType        operation;                                          //!< 操作の種類です
        nn::bluetooth::GattService              gattService;                                        //!< 操作したGATT Service です
        nn::bluetooth::GattCharacteristic       gattCharacteristic;                                 //!< 操作したGATT Characteristic です
        nn::bluetooth::GattDescriptor           gattDescriptor;                                     //!< 操作したGATT Descriptor です
        uint8_t                                 value[nn::bluetooth::GattAttributeValueSizeMax];    //!< 操作の結果、受信したデータです
        uint16_t                                length;                                             //!< 操作の結果、受信したデータの長さです
    };

    GattOperationResult m_GattOperationResult;

    /**
     * @brief       GATT Server のGATT Profile をクリアします
     */
    void ClearServerProfile();

    /**
     * @brief       GATT Profile の中から、指定したUUID を持つGATT Service を取得します
     */
    const nn::bluetooth::GattService* GetService(const nn::bluetooth::GattAttributeUuid& uuid);
    /**
     * @brief       GATT Profile の中から、指定したハンドルを持つGATT Service を取得します
     */
    const nn::bluetooth::GattService* GetService(uint16_t attributeHandle);
    /**
     * @brief       GATT Profile の中から、指定したUUID を持つGATT Characteristic を取得します
     */
    const nn::bluetooth::GattCharacteristic* GetCharacteristic(const nn::bluetooth::GattAttributeUuid& uuid);
    /**
     * @brief       GATT Profile の中から、指定したGATT Descriptor が所属するGATT Characteristic を取得します
     */
    const nn::bluetooth::GattCharacteristic* GetCharacteristic(const nn::bluetooth::GattDescriptor& descriptor);
    /**
     * @brief       GATT Profile の中から、指定したUUID を持つGATT Descriptor を取得します
     */
    const nn::bluetooth::GattDescriptor* GetDescriptor(const nn::bluetooth::GattAttributeUuid& uuid);

    /**
     * @brief       GATT Attribute に対する操作を実行します。
     *
     * @param[in]   Notification またはIndication を有効にするか、無効にするかのフラグ
     */
    void ExecuteGattOperation(bool enable);

    /**
     * @brief       GATT Characteristic をRead します
     */
    void ReadCharacteristic(const nn::bluetooth::GattCharacteristic &characteristic);
    /**
    * @brief       GATT Characteristic にWrite します
    */
    void WriteCharacteristic(nn::bluetooth::GattCharacteristic &characteristic);
    /**
    * @brief       GATT Characteristic のNotification / Indication を有効または無効にします
    */
    void EnableNotification(const nn::bluetooth::GattCharacteristic &characteristic, bool enable, bool isNotification);

    /**
    * @brief       GATT Descriptor をRead します
    */
    void ReadDescriptor(const nn::bluetooth::GattDescriptor &descriptor);
    /**
    * @brief       GATT Descriptor にWrite します
    */
    void WriteDescriptor(nn::bluetooth::GattDescriptor &descriptor);

    /**
     * @brief       既知のGATT Attribute のUUID を文字列に変換します
     */
    const char* GetAttributeName(const nn::bluetooth::GattAttributeUuid& uuid) const;

    //!< 現在のMTU です
    uint16_t m_Mtu;

    //!< GATT Server との接続のインデックスです
    uint8_t m_ConnIndex;

    //!< ConnectionInfoDumper オブジェクトです
    ConnectionInfoDumper m_ConnectionInfoDumper;

};  // class GattClientScene

